/**
 * $Id$
 *
 * TeamFileプラグインSDK用ヘッダファイル(ステータスコード、エラーコード用)
 */
#ifndef INCLUDE_TFSP_ERRNO_H
#define INCLUDE_TFSP_ERRNO_H

#include "tfsp.h"
#include "tfs_errno.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * グローバル変数の定義と宣言の調整を行うためのdefine 値を定義
 */
#ifdef TFSP_ERRMSG_GLOBAL_VALUE_DEFINE
#define TFSP_ERRMSG_GLOBAL
#else
#define TFSP_ERRMSG_GLOBAL extern
#endif  /* TFSP_ERRMSG_GLOBAL_VALUE_DEFINE */

/*------------------------------------------------------------------------------
  Define error code
  ----------------------------------------------------------------------------*/
/**
 * (note) １つのエラーコードでOSのエラーも返すための機構について
 *
 *	エラーが発生したとき、その原因は大きく分けて以下の2通りあるだろう.
 *
 *		1. 仕様にマッチしなかったために起きたエラー
 *		2. 外部API(OSなど)から返却されたエラー
 *	
 *	これらのエラーコード体系は異なるので、ある時は仕様エラーを
 *	ある時はOSエラーを返すことができない。
 *	これを改善するため、以下の方針に基づいて実装すること。
 *
 *  (1) プラグイン用エラーコードを定義する
 *	(2) OS から返却されるエラーコードについてはバイアス値を足しこむ
 *
 * 以上により、適切にバイアス値が調整されていれば、OSのエラーコードと
 * プラグインのエラーコードがバッティングすることは無くなる。
 */

/**
 * プラグイン仕様に基づくエラーコードのバイアス値
 */
#define TFSP_SPEC_START_ERRNO	10000

/**
 * プラグイン固有のエラーコードのバイアス値
 * プラグインが内部で利用するライブラリから返却されたエラーコードなど。
 */
#define TFSP_PLUGIN_START_ERRNO	20000

/* エラーコードを組み立てるマクロ */
#define TFSP_PLUGIN_ERRNO(code)	(TFSP_PLUGIN_START_ERRNO + (code))

/**
 * OS固有のエラーコードのバイアス値。
 *
 * この値にOSエラーコードが足された値としてエラーコードは定義されます
 * (see errno.h)
 */
#define TFSP_OS_START_ERRNO	30000

/* エラーコードを組み立てるマクロ */
#define TFSP_OS_ERRNO(code)	(TFSP_OS_START_ERRNO + (code))

/**
 * 汎用ユーティリティエラーコードのバイアス値
 */
#define TFSP_UTIL_START_ERRNO	40000

/* エラーコードを組み立てるマクロ */
#define TFSP_UTIL_ERRNO(code)	(TFSP_UTIL_START_ERRNO + (code))

/*
 * エラーコードからオリジナルエラーコードを取得するマクロ
 */
#define TFSP_NATIVE_ERRNO(code)	\
	((code < TFSP_SPEC_START_ERRNO) ? code : ((TFSP_PLUGIN_START_ERRNO <= code && code < TFSP_OS_START_ERRNO) ? ((code) - TFSP_PLUGIN_START_ERRNO) : ((TFSP_OS_START_ERRNO <= code && code < TFSP_UTIL_START_ERRNO) ? (code - TFSP_OS_START_ERRNO) : ((FSP_UTIL_START_ERRNO <= code) ? (code - FSP_UTIL_START_ERRNO) : code))))

/**
 * エラーコード一覧
 */
enum {
	TFSP_SUCCESS = 0,	/* 処理成功 */
	TFSP_DECLINE,		/* プラグインの処理対象外だったので、処理をしない */
	TFSP_NOPROCESS,		/* 処理対象ではあったが条件にマッチしなかったので処理しなかった */
	TFSP_ERR = TFSP_SPEC_START_ERRNO + 1,	/* 処理が予期しない理由で失敗した */
	TFSP_EINVALID_PARAM,	/* 入力値の不正 */
	TFSP_ENOPARAM,			/* 必要な入力値が無かった */
	TFSP_ENOCTX,			/* コンテキストオブジェクトが無かった */
	TFSP_ENOVALUE,          /* 必須値がなかった */
	TFSP_EUNSUPPORTED_FOLDER,	/* 処理対象外のフォルダ種別が指定された */
	TFSP_ENOPLUGINROOTDIR,	/* 対象ディレクトリはプラグインルートディレクトリではなかった */
	TFSP_ENOLOADERCONF,		/* プラグインローダ用コンフィグファイルがなかった */
	TFSP_ELOADERCONF_READ,	/* プラグインローダ用コンフィグファイルの読み込みに失敗した */
	TFSP_ENOPLUGINDIR,		/* 対象ディレクトリはプラグインディレクトリではなかった */
	TFSP_EINVALID_PLUGINCONF,	/* プラグインコンフィグファイルの状態が正しくなかった   */
	TFSP_EEMPTYELEM_PLUGINCONF,	/* プラグインコンフィグの中に有効なエレメントがなかった */
	TFSP_ELOADPLUGIN,		/* プラグイン共有ライブラリのローディングに失敗した */
	TFSP_ENOTLOADABLE,		/* ダイナミックローディング可能なライブラリではない */
	TFSP_EUNLOADPLUGIN,		/* プラグインライブラリをアンロードできなかった */
	TFSP_ENOPLUGIN,			/* 対象のプラグインは存在しなかった */
};

/*------------------------------------------------------------------------------
  Define error message
  ----------------------------------------------------------------------------*/
TFSP_ERRMSG_GLOBAL const tfs_err_elem tfsp_error_map[]
#ifdef TFSP_ERRMSG_GLOBAL_VALUE_DEFINE
= {
	{ TFSP_SUCCESS,	"Success"	},
	{ TFSP_DECLINE, "Decline this request" },
	{ TFSP_EERR,	"Unexptected Error"	},
	{ TFSP_EINVALID_PARAM,	"Invalid parameters"	},
	{ TFSP_ENOPARAM,	"Parameter was missing."	},
	{ TFSP_ENOCTX,	"Context was missing"	},
	{ TFSP_ENOVALUE,	"The value was EMPTY."	},
	{ TFSP_EUNSUPPORTED_FOLDER,	"Unsupported folder specified"	},
	{ TFSP_ENOPLUGINROOTDIR,	"This dir is not plugin root dir"	},
	{ TFSP_ENOLOADERCONF,	"The plugin loader config file was missing"	},
	{ TFSP_ELOADERCONF_READ,	"Failed to read loader config xml"	},
	{ TFSP_ENOPLUGINDIR,	"This dir is not plugin's dir"	},
	{ TFSP_EINVALID_PLUGINCONF,	"The plugin's config file was not valid."	},
	{ TFSP_ELOADPLUGIN,	"Failed to load plugin's shared library"	},
	{ TFSP_ENOTLOADABLE,	"We could not load this type of file"	},
	{ TFSP_EUNLOADPLUGIN,	"Failed to unload plugin's shared library"	},
	{ TFSP_ENOPLUGIN,	"The plugin not found."	},
}
#endif	/* TFSP_ERRMSG_GLOBAL_VALUE_DEFINE */
;

#ifdef TFSP_ERRMSG_GLOBAL_VALUE_DEFINE
#define TFSP_ERRMSG_MAP_LEN	sizeof(tfsp_error_map)/sizeof(tfs_err_elem)
#else
#define TFSP_ERRMSG_MAP_LEN 0
#endif	/* TFSP_ERRMSG_GLOBAL_VALUE_DEFINE */

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * エラーコードをエラーメッセージに変換する.
 *
 * @param rv tfs_status_t ステータスコード
 * @param const char * const エラーメッセージ
 *		取得したメッセージの中身を書き換えてはなりません。
 *		文字列リテラルなのでCoreDumpします.
 */
/*TFS_DECLARE(const char * const) tfsp_errcd2str(tfs_status_t rv); */


#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFSP_ERRNO_H */

