/**
 * $Id$
 *
 * TeamFileプラグインSDK用ヘッダファイル(プラグインインターフェースハンドラの定義)
 * (note)
 *	各プラグインはこのヘッダファイルを必ずincludeして下さい。
 */
#ifndef INCLUDE_TFSP_IHDR_H
#define INCLUDE_TFSP_IHDR_H

#include "tfsp.h"
#include "tfs_errno.h"
#include "tfs_errlog.h"
#include "tfsp_spec.h"
#include "tfsp_common.h"
#include "tfsp_errno.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * プラグイン用コンフィグディレクトリの名称
 */
#define TFSP_PLUGIN_CONFIG_DIR	"conf"

/**
 * プライベートコンテキストを取得するマクロ
 */
#define GET_PRIVATE_CTX(ctx)	((ctx)->d)

/*------------------------------------------------------------------------------
  Define structure/enum
  ----------------------------------------------------------------------------*/
typedef enum   tfsp_type_plugin	tfsp_type_plugin;
typedef struct tfsp_ihdr_base	tfsp_ihdr_base;
typedef struct tfsp_ctx_global	tfsp_ctx_global;
typedef struct tfsp_ctx_child	tfsp_ctx_child;
typedef struct tfsp_ihdr_abstract tfsp_ihdr_abstract;

/**
 * TeamFileプラグインの種別
 */
enum tfsp_type_plugin {
	TFSP_PTYPE_UNKNOWN = 0,	/* (未分類) */
	TFSP_PTYPE_CCIPHER,		/* (see tfsp_ihdr_ccipher.h) */
};

/**
 * TeamFileサーバプラグインベースインターフェース構造体の定義
 */
struct tfsp_ihdr_base {
	/**
	 * プラグインの種別
	 */
	tfsp_type_plugin ptype;

	/**
	 * プラグイン識別名称
	 */
	const char *pname;

	/**
	 * プラグイン初期化 インターフェース
	 * [ 役割 ]
	 *   プラグインライブラリがローディングされた直後にコールされます.
	 *   ローディング後の初期化、グローバル領域の確保等に利用して下さい.
	 *
	 * (note)
	 *   ここでアロケートされたものは、その後子プロセスにfork() でコピーされて
	 *   そこで初めて実際に利用されます.
	 *   例えば、ファイルディスクリプタは副作用無しに利用可能でしょう.ですが
	 *   セマフォ生成などは、子プロセス(pinit) でも再オープンしなければならない等
	 *   考慮しなければならないものもあります.
	 *   fork()  によって副作用のあるものは利用しない方が無難です.
	 *   どうしてもということであれば、副作用をプラグイン側で解決して下さい.
	 *	
	 *	なお、実作業を行う子プロセスで生成すればよい種類のものは、
	 *	cinit インターフェースで実施すればよいでしょう.
	 *
	 * @param gctx tfsp_ctx_global * プラグイングローバルコンテキストへのポインタ
	 *             gctx のアロケートは呼び出し元で実施.
	 *             gctx->d はNULL初期化します. アロケートはプラグインで.
	 *             アロケートされたメモリ領域はterminate で開放できます.
	 * @param tfs_status_t 処理ステータス (see tfsp_errno.h)
	 */
	tfs_status_t (*init)(tfsp_ctx_global *gctx);

	/**
	 * 子プロセス毎のプラグイン初期化 インターフェース
	 * [ 役割 ]
	 *   実作業を行う子プロセスがfork() されて、実処理を行う直前にコールされます.
	 *   子プロセス毎に行うプラグインの初期化、領域の確保等に利用して下さい.
	 *
	 * @param cctx tfsp_ctx_child * プラグイン子プロセスレベルコンテキストへのポインタ
	 *             cctx のアロケートは呼び出し元で実施.
	 *             cctx->d はNULL初期化します.
	 *             cctx->global には init で初期化したグローバルコンテキスを格納.
	 * @param tfs_status_t 処理ステータス (see tfsp_errno.h)
	 */
	tfs_status_t (*cinit)(tfsp_ctx_child *cctx);

	/**
	 * 子プロセス毎に呼び出されるプラグイン終了 インターフェース
	 *
	 * [ 役割 ]
	 *   子プロセスが終了する直前にコールされます.
	 *   確保した領域の開放などに利用して下さい.
	 *
	 * (note) エラーの返却はできません. エラーになっても何も出来ないためです.
	 *        エラーを発生させるような種類の処理は禁止します.
	 *
	 * @param cctx tfsp_ctx_child * cinitで初期化された子プロセスレベルコンテキストへのポインタ
	 */
	void (*cterminate)(tfsp_ctx_child *cctx);

	/**
	 * プラグイン終了処理 インターフェース
	 * [ 役割 ]
	 *   プラグインライブラリがunload される直前にコールされます.
	 *   確保した領域の開放などに利用して下さい.
	 *
	 * (note) cterminate 同様ステータスは戻せません。
	 *
	 * @param gctx tfsp_ctx_global * init で初期化されたコンテキストへのポインタ
	 */
	void (*terminate)(tfsp_ctx_global *gctx);

	/* ダミーメンバ. プラグインは絶対に触らないこと. */
	void *reserved1;
	void *reserved2;
	void *reserved3;
	void *reserved4;
	void *reserved5;
};

/**
 * プラグイングローバルコンテキスト構造体の定義
 *
 * [ 役割 ]
 *   * プラグイン固有のデータ保持領域の提供
 */
struct tfsp_ctx_global {
	/* プラグインがインストールされたベースディレクトリのパス名.
	 *
	 * [ パスパターン ]
	 *   $plugin_root/$plugin_name
	 *   $plugin_root = $sdk_root/plugins
	 */
	const char *pluginpath;

	/* 個々のプラグイン専用のコンフィグファイルの置き場所のパス名
	 *
	 * [ パスパターン ]
	 *   (pluginpath の値) / (TFSP_PLUGIN_CONFIG_DIR の値)
	 */
	const char *pluginconfdir;

	/* グローバルレベルのメモリアロケータ.
	 * プラグイン毎に持っています.
	 * init ハンドラ直前で生成されて、terminate 直後で破棄されます. */
	tfs_pool_t *p;

	/**
	 * 任意のデータへのポインタ (see GET_PRIVATE_CTX マクロ)
	 *
	 * 各プラグインは自分自身に必要なデータをここに入れて持ちまわすことが出来ます.
	 * SDKやTeamFile本体は、このメンバを触ることはありません.
	 */
	void *d;

	/* ダミーメンバ. プラグインは絶対に触らないこと. */
	void *reserved1;
	void *reserved2;
	void *reserved3;
	void *reserved4;
	void *reserved5;
};

/**
 * プラグイン子プロセスレベルコンテキスト構造体の定義
 *
 * [ 役割 ]
 *   * 子プロセス単位で持たなければならないデータ保持領域の提供
 */
struct tfsp_ctx_child {
	/**
	 * init で初期化されたグローバルコンテキストへのポインタ
	 */
	tfsp_ctx_global *global;

	/* Childレベルのメモリアロケータ.
	 * プラグイン毎に持っています.
	 * cinit ハンドラ直前で生成されて、cterminate 直後で破棄されます. */
	tfs_pool_t *p;

	/**
	 * 任意のデータへのポインタ (see GET_PRIVATE_CTX マクロ)
	 * プラグインが勝手に使って下さい. それ以外が触ることはありません.
	 */
	void *d;

	/* ダミーメンバ. プラグインは絶対に触らないこと. */
	void *reserved1;
	void *reserved2;
	void *reserved3;
	void *reserved4;
	void *reserved5;
};

/**
 * 各種類のプラグインにキャストするために利用する構造体
 * (note)
 *   この構造体を利用するのはプラグインローダーのみです.
 *   各種プラグイン専用の構造体をキャストする目的で利用する.
 */
struct tfsp_ihdr_abstract {
	const tfsp_ihdr_base *bs;
	void *dummy1;
	void *dummy2;
	void *dummy3;
};

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFSP_IHDR_H */

