/**
 * $Id$
 *
 * 汎用ユーティリティ (エラー状態の保持機構)
 * [ 役割 ]
 *   このヘッダでは、呼び出し元などにエラー発生時の状態を通知するために必要な
 *   機構を定義しています。具体的にロギングする手段までは触れられていません。
 *   それは、別のロガーでやってください。
 */
#ifndef INCLUDE_TFS_ERRLOG_H
#define INCLUDE_TFS_ERRLOG_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"
#include "tfs_string.h"

#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDARG_H
#include <stdarg.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * エラーログレベル
 */
#ifdef HAVE_SYSLOG
#include <syslog.h>
#else	/* !HAVE_SYSLOG */
#define LOG_EMERG   0
#define LOG_ALERT   1
#define LOG_CRIT    2
#define LOG_ERR     3
#define LOG_WARNING 4
#define LOG_NOTICE  5
#define	LOG_INFO	6
#define	LOG_DEBUG	7
#endif	/* HAVE_SYSLOG */

/**
 * desc に文字列を入れる際に利用するマクロ
 */
#define TFS_EDESC0(f)		(char *)__func__, __LINE__, f
#define TFS_EDESC1(f,a1)	(char *)__func__, __LINE__, f,a1
#define TFS_EDESC2(f,a1,a2)	(char *)__func__, __LINE__, f,a1,a2
#define TFS_EDESC3(f,a1,a2,a3)	(char *)__func__, __LINE__, f,a1,a2,a3
#define TFS_EDESC4(f,a1,a2,a3,a4)	(char *)__func__, __LINE__, f,a1,a2,a3,a4

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
/**
 * エラー状態を保持する構造体 [ 不完全型の宣言 ]
 */
typedef struct tfs_error_t	tfs_error_t;

/*------------------------------------------------------------------------------
  Declare type of functions
  ----------------------------------------------------------------------------*/
/**
 * 関数 tfs_error_dumpall() に渡すエラーロガー関数型の定義
 *
 * @param data void * エラーロガーに渡される何らかのコンテキストへのポインタ
 * @param level int エラーログレベル
 * @param desc const char * エラーメッセージ
 */
typedef TFS_DECLARE(void) (*tfs_error_logger_t)(void *, int, const char *);

/*------------------------------------------------------------------------------
 * Declare public function
 *----------------------------------------------------------------------------*/
/**
 * エラーメッセージdesc を指定してエラー情報インスタンスを生成する.
 * funcname, lineno, desc メッセージの生成には、TFS_EDESC0 からTFS_EDESC4 までの
 * マクロを使用して下さい.
 * ex)
 *   tfs_error_create(p, LOG_ERR, TFS_XXX, TFS_EDESC1("xxxx = %s", str));
 *
 * @param pool tfs_pool_t * 作業用プール
 * @param level int エラーログレベル
 * @param status int エラーステータス
 * @param funcname char * 関数名
 * @param lineno int 行番号
 * @param fmt const char * フォーマット文字列(printfと同等)
 * @param ... 可変長文字列
 * @return tfs_error_t * エラー情報を保持する構造体
 */
TFS_DECLARE(tfs_error_t *) tfs_error_create(tfs_pool_t *pool, int level, int status,
                                            char *funcname, int lineno, const char *fmt, ...);

/**
 * 指定されたエラー情報を記録し、それを指定されたエラーprev と関連付けて返す.
 * funcname, lineno, desc メッセージの生成には、TFS_EDESC0 からTFS_EDESC4 までの
 * マクロを使用して下さい.
 * (note)
 * 	prev がNULLであれば何もしません.
 *
 * @param prev tfs_error_t * 前回のエラー
 * @param funcname char * 関数名
 * @param lineno int 行番号
 * @param level int エラーログレベル
 * @param status int エラーステータス
 * @param fmt const char * フォーマット文字列(printfと同等)
 * @param ... 可変長文字列
 * @return tfs_error_t * エラー情報を保持する構造体
 */
TFS_DECLARE(tfs_error_t *) tfs_error_push(tfs_error_t *prev, int level, int status,
                                          char *funcname, int lineno, const char *fmt, ...);

/**
 * 指定されたエラーprev に新しいエラーnew_err を連結して1つのエラー情報にする.
 * なお、この処理が終わった後、prev もnew_err も不要ならば破棄して下さい.
 * この関数によって得られるエラーはこれらとは全く別のインスタンスとなります.
 *
 * @param pool tfs_pool_t * メモリアロケータ
 * @param prev tfs_error_t * 一つ前のエラー情報
 * @param new_err tfs_error_t * 連結するエラー情報
 * @return tfs_error_t * 連結後のエラー情報
 */
TFS_DECLARE(tfs_error_t *) tfs_error_append(tfs_pool_t *pool, tfs_error_t *prev,
                                            tfs_error_t *new_err);

/**
 * old が示すエラー情報をp のメモリアロケーターを使って複製する.
 *
 * @param pool tfs_pool_t * メモリアロケーター
 * @param old tfs_error_t * 複製元エラー情報
 * @return tfs_error_t * 複製されたエラー情報
 */
TFS_DECLARE(tfs_error_t *) tfs_error_clone(tfs_pool_t *pool, tfs_error_t *old);

/**
 * err が保持する全てのエラー情報をエラーロガーlogger を使って全て出力する.
 * ここ関数がコールされるとerr が蓄えていた全てのエラー情報はクリアされます.
 *
 * @param err tfs_error_t * エラー情報
 * @param data void * logger に渡される何らかのデータへのポインタ
 * @param logger tfs_error_logger_t エラーロガー関数へのポインタ
 */
TFS_DECLARE(void) tfs_error_dumpall(tfs_error_t *err, void *data,
                                    tfs_error_logger_t logger);

/**
 * err が保持するエラーステータスを取得する.
 * 複数個のエラーがPushされている場合、一番先頭のエラーコードが返されます.
 *
 * @param err tfs_error_t * エラー情報
 * @return tfs_status_t
 */
TFS_DECLARE(tfs_status_t) tfs_error_getstatus(tfs_error_t *err);

/**
 * stderr に出力するエラーロガーの実装(tfs_error_logger_t の実装)
 * エラーレベルとは無関係に出力します. data は使用しません.
 * (note)
 *   tfs_error_logger_t の実装です。関数tfs_error_dumpallに渡して
 *   ログを出力させることが可能です。
 *
 * @param data void * エラーロガーに渡される何らかのコンテキストへのポインタ
 * @param level int エラーログレベル
 * @param desc const char * エラーメッセージ
 */
TFS_DECLARE(void) tfs_error_dump_stderr(void *data, int level, const char *desc);

/**
 * デバッグ用ロガー
 * (note)
 *   基本的にはデバッグ用、良くてインフォメーション表示用ロガーとして使って下さい.
 *   funcname, lineno 等はTFS_EDESC0 からTFS_EDESC4 を使って下さい.
 *   (例)
 *     tfs_log_notice(TFS_EDESC1("aaaa = %s", xxx));
 *
 * @param funcname char * 関数名
 * @param lineno int 行番号
 * @param fmt const char * フォーマット文字列(printfと同等)
 * @param ... 可変長文字列
 */
TFS_DECLARE(void) tfs_log_notice(char *funcname, int lineno, const char *fmt, ...);

/**
 * err が消費していた領域を破棄する.
 *
 * @param err tfs_error_t * エラー情報
 */
TFS_DECLARE(void) tfs_error_destroy(tfs_error_t *err);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_ERRLOG_H */

