/**
 * $Id$
 *
 * 汎用ユーティリティ (ステータスコード、エラーコード用)
 */
#ifndef INCLUDE_TFS_ERRNO_H
#define INCLUDE_TFS_ERRNO_H

#include "tfs.h"

#if HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * グローバル変数の定義と宣言の調整を行うためのdefine 値を定義
 */
#ifdef TFS_ERRMSG_GLOBAL_VALUE_DEFINE
#define TFS_ERRMSG_GLOBAL
#else
#define TFS_ERRMSG_GLOBAL extern
#endif  /* TFS_ERRMSG_GLOBAL_VALUE_DEFINE */

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
/**
 * エラーステータスコードを表す型の別名宣言
 */
typedef tfs_uint32_t	tfs_status_t;

/**
 * エラーコードとエラーメッセージの関係を保持する構造体
 */
typedef struct tfs_err_elem	tfs_err_elem;
struct tfs_err_elem {
	tfs_status_t rv;
	const char *msg;
};

/*------------------------------------------------------------------------------
  Define error code
  ----------------------------------------------------------------------------*/
/**
 * 内部エラーコードのバイアス値
 */
#define TFS_SPEC_START_ERRNO	10000

/* エラーコードを組み立てるマクロ */
#define TFS_ERRNO(code)	(TFS_SPEC_START_ERRNO + (code))

/**
 * OS固有のエラーコードのバイアス値。
 *
 * この値にOSエラーコードが足された値としてエラーコードは定義されます
 * (see errno.h)
 */
#define TFS_OS_START_ERRNO	30000

/* エラーコードを組み立てるマクロ */
#define TFS_OS_ERRNO(code)	(TFS_OS_START_ERRNO + (code))

/**
 * エラーコード一覧
 */
enum {
	TFS_SUCCESS = 0,	/* 処理成功 */
	TFS_FILE_EOF,		/* ファイルのEOFを発見した */
	TFS_FILE_NOCONTENT,	/* 書き込みまたは読み込み対象のファイル内容が無かった */
	TFS_QUEUE_EMPTY,	/* Queue の要素が既に存在しないことを検出した */

	TFS_ERR = TFS_SPEC_START_ERRNO + 1,	/* 処理が予期しない理由で失敗した */
	TFS_EINVALID_PARAM,	/* 入力値の不正 */
	TFS_ENOPARAM,			/* 必要な入力値が無かった */
	TFS_ENOOBJ,		/* 必要なオブジェクトが見つからなかった */
	TFS_ENULLPOINTER,	/* NULL値を許さないケースに該当した */
	TFS_EFILE_WRITE,	/* ファイルの書き込みに失敗した */
	TFS_EREUSEXMLPARSER,	/* XML パーサを別のDOMノード解析に利用しようとした */
	TFS_EPARSEXML,			/* XML のパース処理に失敗した */
};

/*------------------------------------------------------------------------------
  Define error message
  ----------------------------------------------------------------------------*/
TFS_ERRMSG_GLOBAL const tfs_err_elem tfs_err_map[]
#ifdef TFS_ERRMSG_GLOBAL_VALUE_DEFINE
= {
	{ TFS_SUCCESS,	"Success"	},
	{ TFS_FILE_EOF,	"Found file EOF status" },
	{ TFS_FILE_NOCONTENT,	"No content to read / write" },
	{ TFS_QUEUE_EMPTY,	"Queue is empty." },
	{ TFS_EERR,	"Unexptected Error"	},
	{ TFS_EINVALID_PARAM,	"Invalid parameters"	},
	{ TFS_ENOPARAM,	"Parameter was missing."	},
	{ TFS_ENOOBJ,	"Object was missing."	},
	{ TFS_ENULLPOINTER,	"NULL object found."	},
	{ TFS_EFILE_WRITE,	"Could not write data to file"	},
	{ TFS_EREUSEXMLPARSER,	"Could not reuse another DOM parser"	},
	{ TFS_EPARSEXML,	"Failed to parser XML text"	},
}
#endif	/* TFS_ERRMSG_GLOBAL_VALUE_DEFINE */
;

#ifdef TFS_ERRMSG_GLOBAL_VALUE_DEFINE
#define TFS_ERRMSG_MAP_LEN	sizeof(tfs_err_map)/sizeof(tfs_err_elem)
#else
#define TFS_ERRMSG_MAP_LEN 0
#endif	/* TFS_ERRMSG_GLOBAL_VALUE_DEFINE */

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * エラーコードをエラーメッセージに変換する.
 *
 * @param rv tfs_status_t ステータスコード
 * @param const char * const エラーメッセージ
 *		取得したメッセージの中身を書き換えてはなりません。
 *		文字列リテラルなのでCoreDumpします.
 */
/* TFS_DECLARE(const char * const) tfs_errcd2str(tfs_status_t rv); */


#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_ERRNO_H */

