/**
 * $Id$
 *
 * 汎用ユーティリティ (HashSet : ハッシュによるSet インターフェースの実装)
 */
#ifndef INCLUDE_TFS_HASHSET_H
#define INCLUDE_TFS_HASHSET_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * ハッシュセットを表すコンテキスト構造体 [ 不完全型 ]
 */
typedef struct tfs_hset_t	tfs_hset_t;

/**
 * ハッシュセット内の要素列挙に使用するイテレータ [ 不完全型 ]
 */
typedef struct tfs_hset_index_t	tfs_hset_index_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * ハッシュセットの生成
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * (Reentrant)
 *   この関数はスレッドセーフです.他の関数も、hset をスレッド間で共有しない限り
 *   スレッドセーフとなります.
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param hset tfs_hset_t ** 生成されたハッシュセットへのポインタ
 */
TFS_DECLARE(void) tfs_hset_create(tfs_pool_t *pool, tfs_hset_t **hset);

/**
 * 要素element がハッシュセットhset になかった場合、セットに追加する.
 *
 * @param hset tfs_hset_t * ハッシュセットへのポインタ
 * @param element const char * 文字列へのポインタ
 * @return int セットが指定された要素を保持していなかった場合は1/それ以外は0
 */
TFS_DECLARE(int) tfs_hset_add(tfs_hset_t *hset, const char *element);

/**
 * 文字列 element がハッシュセットhset に含まれているかどうか.
 *
 * @param hset tfs_hset_t * ハッシュセットへのポインタ
 * @param element const char * 文字列へのポインタ
 * @return int 持っていた(1) / 持っていない(0)
 */
TFS_DECLARE(int) tfs_hset_contains(tfs_hset_t *hset, const char *element);

/**
 * 指定された文字列element をハッシュセットhset から除去する.
 *
 * @param hset tfs_hset_t * ハッシュセットへのポインタ
 * @param element const char * 削除対象の文字列へのポインタ
 * @return int セットが指定された要素を保持していた場合は1/それ以外は0
 */
TFS_DECLARE(int) tfs_hset_remove(tfs_hset_t *hset, const char *element);

/**
 * ハッシュセットhset が保持する全要素数を返す.
 *
 * @param hset tfs_hset_t * ハッシュセットへのポインタ
 * @return tfs_int32_t 全要素数
 */
TFS_DECLARE(tfs_int32_t) tfs_hset_size(tfs_hset_t *hset);

/**
 * ハッシュセットhset のイテレータを生成して返却する.
 *
 * @param hset tfs_hset_t * ハッシュセットへのポインタ
 * @return tfs_hset_index_t * イテレータへのポインタ
 */
TFS_DECLARE(tfs_hset_index_t *) tfs_hset_first(tfs_hset_t *hset);

/**
 * イテレータiterator が次の要素を指すように移動する.
 *
 * @param iterator tfs_hset_index_t * イテレータ
 * @return tfs_hset_index_t * 次のイテレータ. NULLで終端を示す.
 */
TFS_DECLARE(tfs_hset_index_t *) tfs_hset_next(tfs_hset_index_t *iterator);

/**
 * イテレータiterator からハッシュセットにマップされた文字列を取得する.
 * (note)
 *   取得したオブジェクトelement の中身は書き換えないで下さい. 
 *   置き換えたいのならtfs_hset_remove() で消してから
 *   tfs_hset_add() を追加してください.
 *
 * @param iterator tfs_hset_index_t * イテレータ
 * @param element const char ** 要素へのポインタ
 */
TFS_DECLARE(void) tfs_hset_getvalue(tfs_hset_index_t *iterator, const char **element);

/**
 * ハッシュセットhset の要素をクリアする.
 *
 * ハッシュセット内に格納されたエレメントオブジェクトに関しては何もしません.
 * 呼び出し元で必要であれば適切な終了処理を行ってからこの関数を呼び出して下さい.
 *
 * @param hset tfs_hset_t * ハッシュセットへのポインタ
 */
TFS_DECLARE(void) tfs_hset_clear(tfs_hset_t *hset);

/**
 * hset が示すハッシュセットの破棄.
 *
 * ハッシュセット内に格納されたエレメントオブジェクトに関しては何もしません.
 * 呼び出し元で必要であれば適切な終了処理を行ってからこの関数を呼び出して下さい.
 *
 * @param hset tfs_hset_t * ハッシュセットへのポインタ
 */
TFS_DECLARE(void) tfs_hset_destroy(tfs_hset_t *hset);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_HASHSET_H */

