/**
 * $Id$
 *
 * 汎用ユーティリティ (Queue : FIFO のQueue)
 */
#ifndef INCLUDE_TFS_QUEUE_H
#define INCLUDE_TFS_QUEUE_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * Queueを表すコンテキスト構造体 [ 不完全型の宣言 ]
 */
typedef struct tfs_queue_t	tfs_queue_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * Queue の生成.
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * (Reentrant)
 *   この関数はスレッドセーフです.他の関数も、hmap をスレッド間で共有しない限り
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param queue tfs_queue_t ** 生成されたFIFO Queueへのポインタ
 */
TFS_DECLARE(void) tfs_queue_create(tfs_pool_t *pool, tfs_queue_t **queue);

/**
 * Queue に値value を入れる. (PUSH)
 * (note)
 *   value はコピーせずポインタのままQueueに格納します.
 *   呼び出し元でvalue 領域を開放してしまうとQueueに格納された値も不定値になります.
 *
 * @param queue tfs_queue_t * Queueへのポインタ
 * @param value void * Queueに入れるデータへのポインタ
 *                     NULL ポインタは格納できません.
 * 	TFS_SUCCESS : 成功
 * 	TFS_ENOPARAM : queue がNULLだった
 * 	TFS_ENULLPOINTER : NULL を入れようとした
 */
TFS_DECLARE(tfs_status_t) tfs_queue_push(tfs_queue_t *queue, void *value);

/**
 * Queue から値を取り出す. (POP)
 * 取り出された値はQueueから取り除かれます.
 *
 * @param queue tfs_queue_t * Queueへのポインタ
 * @param value void ** Queue から取り出されたデータ領域へのポインタ
 * @param tfs_status_t POPステータス
 * 	TFS_SUCCESS     : 値の取得に成功.
 * 	TFS_QUEUE_EMPTY : 値が入っていなかった (*value もNULLになります)
 */
TFS_DECLARE(tfs_status_t) tfs_queue_pop(tfs_queue_t *queue, void **value);

/**
 * Queue から値を取り出す. (POP)
 * 取り出された値はQueueから"取り除かれません".(tfs_queue_pop との違い)
 *
 * @param queue tfs_queue_t * Queueへのポインタ
 * @param value void ** Queue から取り出されたデータ領域へのポインタ
 * @param tfs_status_t POPステータス
 * 	TFS_SUCCESS     : 値の取得に成功.
 * 	TFS_QUEUE_EMPTY : 値が入っていなかった (*value もNULLになります)
 */
TFS_DECLARE(tfs_status_t) tfs_queue_peek(tfs_queue_t *queue, void **value);

/**
 * queue が示すQueueをpool のメモリ領域にコピーする.
 * (note)
 *   Queue に格納された値はコピーされません。ポインタが共用されます.
 *   入れ物がCloneされるだけです.
 *
 * @param pool tfs_pool_t *
 * @param queue tfs_queue_t * コピー元
 * @return tfs_queue_t * 複製したQueueへのポインタ
 */
TFS_DECLARE(tfs_queue_t *) tfs_queue_clone(tfs_pool_t *pool, tfs_queue_t *queue);

/**
 * Queue に格納された要素の総数を取得する.
 *
 * @param queue tfs_queue_t * Queueへのポインタ
 */
TFS_DECLARE(tfs_int32_t) tfs_queue_size(tfs_queue_t *queue);

/**
 * queue が示すQueueの要素を全てクリアする.
 *
 * @param queue tfs_queue_t * Queueへのポインタ
 */
TFS_DECLARE(void) tfs_queue_clear(tfs_queue_t *queue);

/**
 * queue が示すQueueの破棄.
 *
 * @param queue tfs_queue_t * Queueへのポインタ
 */
TFS_DECLARE(void) tfs_queue_destroy(tfs_queue_t *queue);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_QUEUE_H */

