/**
 * $Id$
 *
 * 汎用ユーティリティ (StringBuffer : 可変長文字列バッファ)
 */
#ifndef INCLUDE_TFS_SBUF_H
#define INCLUDE_TFS_SBUF_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * 可変長文字列バッファを表すコンテキスト構造体 [ 不完全型 ]
 */
typedef struct tfs_sbuf_t	tfs_sbuf_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 可変長文字列バッファの生成.
 * 指定されたsize バイトの文字列を格納可能な文字列バッファsbuf を生成して
 * 返却する.
 * 確保される文字列バッファには文字列終端文字'\0' を格納する領域が自動的に
 * 追加されます。(つまり size + 1バイト分確保されるということ)
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * バッファが足りなくなったときの増分(delta) は以下のように決定します。
 * (1) size/2 が デフォルト増加サイズ よりも小さいときには
 * 	デフォルト増加サイズを使用する
 * (2) size/2 が デフォルト増加サイズよりも大きい時には
 * 	size/2 が使用される
 * (note)
 * 	size が負の時には、デフォルト値(> 0) のバッファを確保します。
 *
 * (Reentrant)
 *   この関数はスレッドセーフです.他の関数も、sbuf をスレッド間で共有しない限り
 *   スレッドセーフとなります.
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param size tfs_size_t 文字列バッファのサイズ(byte)
 * @param sbuf tfs_sbuf_t ** 文字列バッファ
 */
TFS_DECLARE(void) tfs_sbuf_create(tfs_pool_t *pool, tfs_size_t size, tfs_sbuf_t **sbuf);

/**
 * 指定されたstr が示す文字列の長さを初期値としての文字列バッファsbuf を
 * 初期化して返却する。
 * 確保される文字列バッファは strlen(str) + 1 バイトとなります。
 * つまり、str の文字列全てを保持できるだけのバッファを確保します。
 * なお、str がNULLの場合にはデフォルトサイズのバッファが確保されます。
 *
 * (note)
 * pool が指定された場合、pool がメモリアロケーターとなります。
 * この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 * 自動的に破棄されます.
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param str const char * 最初にセットする文字列
 * @param sbuf tfs_sbuf_t ** 文字列バッファ
 */
TFS_DECLARE(void) tfs_sbuf_createbyStr(tfs_pool_t *pool, const char *str, tfs_sbuf_t **sbuf);

/**
 * 文字列バッファsbuf に設定された文字列バッファの現在のサイズ(byte)を取得する。
 * (note)
 * 	格納されている文字列の長さではありません。バッファ全体の長さです。
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @return tfs_size_t 文字列バッファの現在の長さ(byte)
 */
TFS_DECLARE(tfs_size_t) tfs_sbuf_getcapacity(tfs_sbuf_t *sbuf);

/**
 * 文字列バッファsbuf に追加された文字列の総長さ(byte)を取得する。
 * (note)
 * 	tfs_sbuf_getcapacity から取得される値とは異なることがあります。
 * 	また、'\0'分はカウントされません。(strlenと同じです)
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @return tfs_size_t 追加された文字列の総長さ(byte)
 */
TFS_DECLARE(tfs_size_t) tfs_sbuf_getlength(tfs_sbuf_t *sbuf);

/**
 * 文字列バッファsbuf に最後に追加された文字列の後ろににstr が示す文字列を
 * 追加する。
 * str がNULLの時には何もしません。
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @param str const char * 追加する文字列
 */
TFS_DECLARE(void) tfs_sbuf_append(tfs_sbuf_t *sbuf, const char *str);

/**
 * 文字列バッファsbuf の最後に追加された文字列の後ろににstr が示す文字列を
 * len バイトだけ追加する。str がNULLやlen が0 の時には何もしません。
 * (note) tfs_sbuf_append との違い
 * 	この関数は、文字列長さが呼び出し側で既に判明している場合(例えば
 * 	固定値を入れるなど)にtfs_sbuf_appendよりもstrlenを計算しない分だけ
 * 	高速になります。(僅かですが。)
 * 	それ以外の処理はtfs_sbuf_append と同じです。
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @param str const char * 追加する文字列
 */
TFS_DECLARE(void) tfs_sbuf_appendByte(tfs_sbuf_t *sbuf,
                                      tfs_size_t len, const char *str);

/**
 * 文字列バッファsbuf の最後に追加された文字列の後ろににch が示す文字を追加する.
 * ch が'\0' だと追加しません.
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @param str const char * 追加する文字列
 */
TFS_DECLARE(void) tfs_sbuf_appendChar(tfs_sbuf_t *sbuf, char ch);

/**
 * 文字列バッファsbuf のstart が示すインデックス位置に文字列str を追加する。
 * start位置にあった文字はstrlen(str) 分だけ後ろに移動されます。
 * (例)
 * 	sbuf: "01234abcd"の時
 * 	tfs_sbuf_insert(sbuf, 3, "xxxxx") -> "012xxxxx34abcd"
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @param start tfs_size_t インデックス位置
 * @param str const char * 追加する文字列
 * @return tfs_status_t 処理ステータス
 * 	tfs_sbuf_getlength() の結果よりもstartが大きいとき、または
 * 	startが負の値の時には異常終了します。
 */
TFS_DECLARE(tfs_status_t) tfs_sbuf_insert(tfs_sbuf_t *sbuf,
                                          tfs_size_t start, const char *str);

/**
 * 文字列バッファsbuf のstart が示すインデックス位置からインデックス位置
 * (end - 1) までをstr が示す文字列で置換する
 * str が 置換長さよりも長い場合には切り取って挿入します。
 * また、str が 置換長さよりも短い場合には、strlen(str) までをコピーします。
 * end がバッファ長さよりも大きい場合にはバッファを自動拡張します。
 * (例)
 * 	sbuf: "01234abcd"の時
 * 	tfs_sbuf_replace(sbuf, 2, 5, "xxxxx") -> "01xxxabcd"
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @param start tfs_size_t 置き換え先頭インデックス位置
 * @param end tfs_size_t   置き換え終端位置のバイト数
 * @param str const char * 置き換える文字列
 * @return tfs_status_t 処理ステータス
 * 	tfs_sbuf_getlength() の結果よりもstartが大きいとき、または
 * 	startが負の値の時には異常終了します。
 */
TFS_DECLARE(tfs_status_t) tfs_sbuf_replace(tfs_sbuf_t *sbuf, tfs_size_t start,
                                           tfs_size_t end, const char *str);

/**
 * 文字列バッファsbuf から格納されている全ての文字列を取得する。
 * (note)
 * 	取得される文字列は、新しい領域にコピーされません。
 * 	バッファが持っていたポインタを返却するだけです。
 * 	そのため、tfs_sbuf_toString で取得した文字列をコピーせずに
 * 	tfs_sbuf_clear を実施すると、取得文字列は空になってしまいます。
 * 	tfs_sbuf_clear との組み合わせでは注意が必要です。
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @return char * 取得した文字列
 */
TFS_DECLARE(char *) tfs_sbuf_toString(tfs_sbuf_t *sbuf);

/**
 * 文字列バッファsbuf に格納された文字列を全て消去する。
 * ただし、確保されたバッファ領域はクリアされません。
 * (note)
 * 	tfs_sbuf_toString 関数との組み合わせでは注意が必要です。
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 */
TFS_DECLARE(void) tfs_sbuf_clear(tfs_sbuf_t *sbuf);

/**
 * sbuf にアロケートされていたメモリ領域を回収する.
 * この関数呼出し後、sbuf に触ってはなりません.
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 */
TFS_DECLARE(void) tfs_sbuf_destroy(tfs_sbuf_t *sbuf);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_SBUF_H */

