/**
 * $Id$
 *
 * 汎用ユーティリティ (XML DOM parser : XMLのDOM パーサ実装)
 * (note)
 *   現時点でここで定義された一連のAPIではネームスペースを扱えません.
 */
#ifndef INCLUDE_TFS_XML_H
#define INCLUDE_TFS_XML_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * XML DOM パーサをコンテキスト構造体 [ 不完全型の宣言 ]
 */
typedef struct tfs_xml_parser	tfs_xml_parser;

/**
 * XML Document (XMLドキュメントツリーのルート) を表す構造体 [ 不完全型の宣言 ]
 */
typedef struct tfs_xml_doc	tfs_xml_doc;

/**
 * Text ノードを表す構造体 [ 不完全型の宣言 ]
 */
typedef struct tfs_xml_text	tfs_xml_text;

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
typedef struct tfs_xml_elem	tfs_xml_elem;
typedef struct tfs_xml_attr	tfs_xml_attr;

/**
 * XML Element (XMLドキュメントの要素) を表す構造体
 */
struct tfs_xml_elem {
	/* エレメント名称 */
	const char *name;

	/* xml:lang */
	const char *lang;

	/* Textノード / CDATA Section */
	tfs_xml_text *text;

	/* 属性 */
	tfs_xml_attr *attr;

	/* 同階層にあるエレメントへのポインタ(sibling) */
	tfs_xml_elem *next;

	/* child エレメントへのポインタ */
	tfs_xml_elem *first_child;

	/*
	 * 以下はパース時作業用. DOMノードの構成とは関係ありません
	 */

	/* parent エレメントへのポインタ */
	tfs_xml_elem *parent;

	/* 最後に接続されたChildノードへのポインタ */
	tfs_xml_elem *last_child;

	/* このエレメントをパースしたparser へのポインタ
	 * (note) 利用するモジュールはこのメンバへのアクセスは禁止 */
	tfs_xml_parser *parser;
};

/**
 * XML 属性を表す構造体
 */
struct tfs_xml_attr {
	const char *name;	/* 属性名称 */
	const char *value;	/* 値 */

	tfs_xml_attr *next;	/* 次の属性へのポインタ */
};

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * あるDOM ノードを処理するためのXML パーサの生成.
 *
 * (note) parser と DOM ノードとの関係について
 *   parser と あるDOMノードとの関係は1対1となります. 
 *   従って、あるDOMノードをパースするのに使用したparser オブジェクトは
 *   別のDOMノード解析処理に利用できません. 一種の制約事項です.
 *
 * (note) DOMノードのライフサイクルについて
 *   parser と DOMノードのライフサイクルは厳密に一致します.
 *   そのため、パース終了後parserを破棄した場合、出力されれていたDOMノードは
 *   全て不定値となります. DOMノードをコピーすることはできませんので、
 *   基本的には、DOMノードにアクセスする限り、parser を破棄してはなりません.
 *
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * (Reentrant)
 *   この関数はスレッドセーフです.他の関数も、parser をスレッド間で共有しない限り
 *   スレッドセーフとなります.
 *
 * @param pool tfs_pool_t * 外部メモリアロケーターへのポインタ
 * @param parser tfs_xml_parser ** 生成したXML DOMパーサへのポインタ
 */
TFS_DECLARE(void) tfs_xml_parser_create(tfs_pool_t *pool, tfs_xml_parser **parser);

/**
 * 指定されたファイルパスpath のXMLファイルをパースしてDOMノードを構築する.
 * (note)
 *   この関数では、tfs_xml_parser_create() で生成したXMLパーサを利用できません.
 *   本関数から戻されたXMLパーサを使ってDOM ノードは処理されます.
 *
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * @param pool tfs_pool_t * 外部メモリアロケーターへのポインタ
 * @param path const char * XMLファイルのフルパス
 * @param parser tfs_xml_parser ** doc をパースするXML DOMパーサへのポインタ
 * @param doc tfs_xml_doc ** パース結果のXML ドキュメントへのポインタ
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_file(tfs_pool_t *pool, const char *path,
                                             tfs_xml_parser **parser,
                                             tfs_xml_doc **doc);

/**
 * サイズlen のデータ data のXMLデータをパースしてDOMノードを構築する.
 * (note)
 *   この関数では、tfs_xml_parser_create() で生成したXMLパーサを利用できません.
 *   本関数から戻されたXMLパーサを使ってDOM ノードは処理されます.
 *
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * @param pool tfs_pool_t * 外部メモリアロケーターへのポインタ
 * @param data const char * XMLデータ
 * @param len tfs_size_t data の長さ
 * @param parser tfs_xml_parser ** doc をパースするXML DOMパーサへのポインタ
 * @param doc tfs_xml_doc ** パース結果のXML ドキュメントへのポインタ
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_data(tfs_pool_t *pool, const char *data,
                                             tfs_size_t len,
                                             tfs_xml_parser **parser,
                                             tfs_xml_doc **doc);

/**
 * 組み立て途中のサイズlen のデータ data のXMLデータを XML DOMパーサparser に
 * 渡してパースを行う.
 * この関数は全てのXMLデータを渡し終えるまでlen と dataにセットして呼んで下さい.
 * 渡し終えたら、後述する関数tfs_xml_parse_done() を実行して下さい.
 * (note)
 *   parser は関数 tfs_xml_parser_create() で生成したものを使います.
 *
 * @param parser tfs_xml_parser * XML DOMパーサへのポインタ
 * @param data const char * XMLデータ
 * @param len tfs_size_t data の長さ
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_feed(tfs_xml_parser *parser,
                                             const char *data,
                                             tfs_size_t len);

/**
 * tfs_xml_parse_feed で蓄積されたXMLを全てパースしてDOMノードを構築する.
 * (note)
 *   parser は関数 tfs_xml_parser_create() で生成したものを使います.
 *
 * @param parser tfs_xml_parser * XML DOMパーサへのポインタ
 * @param doc tfs_xml_doc ** パース結果のXML ドキュメントへのポインタ
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_done(tfs_xml_parser *parser,
                                             tfs_xml_doc **doc);

/**
 * XML のパース中に発生したエラーメッセージを取得する.
 *
 * @param parser tfs_xml_parser * XML DOMパーサへのポインタ
 * @param rv tfs_status_t エラー発生時のエラーコード
 * @return char * エラーメッセージへのポインタ.
 *                この文字列はfree() してはなりません.
 */
TFS_DECLARE(char *) tfs_xml_geterror(tfs_xml_parser *parser, tfs_status_t rv);

/**
 * Documentノードの元最初のノード(rootノード) を取得する.
 *
 * @param doc tfs_xml_doc *
 * @return tfs_xml_elem * root ノードへのポインタ
 */
TFS_DECLARE(tfs_xml_elem *) tfs_xml_get_rootnode(const tfs_xml_doc *doc);

/**
 * XML エレメントelem のTextノード値(CDATAセクションを含む) を取得して返却する.
 * (note)
 *   p が与えられた場合、戻り値のライフサイクルはp に一致します.
 *   p がNULLの場合、戻り値のライフサイクルはparser のそれと一致します.
 *
 * @param p tfs_pool_t * 作業用プールへのポインタ
 * @param elem tfs_xml_elem * エレメントオブジェクト
 * @param strip_white int XMLのWhiteスペース文字を除去するかどうか.
 * @return char * 取得したCDATA へのポインタ. ライフサイクルはelem と同じなので
 *                必要であれば呼び出し元でコピーして利用して下さい.
 */
TFS_DECLARE(char *) tfs_xml_get_cdata(tfs_xml_elem *elem, int strip_white, tfs_pool_t *p);

/**
 * XML エレメントelem のChild からtagname と一致するエレメントを取得する.
 *
 * @param elem tfs_xml_elem * エレメントオブジェクト
 * @param tagname const char * エレメント名称
 * @return tfs_xml_elem * 取得したエレメントへのポインタ
 */
TFS_DECLARE(tfs_xml_elem *) tfs_xml_find_child(tfs_xml_elem *elem,
                                               const char *tagname);

/**
 * XML DOMパーサparser の破棄.
 * parser が保持していた情報及び生成した全てのXML Document に関する
 * メモリ領域を開放します.
 * 本関数呼出し後、parser に触ってはなりません.
 *
 * @param parser tfs_xml_parser * XML DOMパーサへのポインタ
 */
TFS_DECLARE(void) tfs_xml_parser_destroy(tfs_xml_parser *parser);

/**
 * 文字列s をテキストノード値にする際、利用できない文字列をQuoteする.
 *
 * [ 置換文字一覧 ]
 *   * '<'  -->  '&lt;'
 *   * '>'  -->  '&gt;'
 *   * '&'  -->  '&amp;'
 *
 * @param pool tfs_pool_t * メモリアロケーター
 * @param s const char * 対象文字列
 * @return char * Quoteされた文字列へのポインタ. 変換する必要が無い時でもs を
 *                複製して渡します.
 */
TFS_DECLARE(char *) tfs_xml_quote_string(tfs_pool_t *pool, const char *s);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_XML_H */

