/**
 * $Id$
 *
 * TeamFileプラグインSDK用ヘッダファイル(プラグインローダ)
 */
#define TFSP_CORE_PRIVATE

#include "tfsp.h"
#include "tfsp_loader.h"
#include "tfs_string.h"
#include "tfs_varray.h"
#include "tfs_pools.h"
#include "tfs_xml.h"
#include "tfs_errlog.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#if HAVE_DIRENT_H
#include <dirent.h>
#endif
#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#if HAVE_DLFCN_H
#include <dlfcn.h>
#endif

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * プラグインモジュールコンテキスト取得マクロ
 */
#define PM_CTX(pm)	((pm)->ctx)

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
typedef struct tfsp_loader_config     tfsp_loader_config;

/**
 * プラグインローダ用コンフィグ情報を表す構造体
 */
struct tfsp_loader_config {
	char *name; /* SDK の名前 */
	char *version; /* SDK のバージョン番号 */
	char *summary; /* 短い説明文 */
	char *description; /* 詳細な説明文 */

	/* サポートインターフェース一覧 */
	tfs_varray_t *supported_if;

	/* 作成者情報 */
	tfsp_author_info *author;
};

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * ローディングされたプラグインの状態を管理する構造体 [ 不完全型の定義 ]
 */
struct tfsp_ctx_loader {
	/* プラグインインストールルートパス(絶対パス) */
	char *rootpath;

	/* プラグインローダ用コンフィグファイルパス */
	char *confpath;

	/* プラグイン情報からなる配列 */
	tfs_varray_t *plugins;

	/* プラグインローダコンフィグ */
	tfsp_loader_config *conf;

	/* メモリアロケーター */
	tfs_pool_t *pool;
};

/**
 * プラグインモジュールコンテキスト構造体 [ 不完全型の定義 ]
 */
struct tfsp_plugin_modulectx {
	/* ローディングした共有ライブラリのハンドル */
	void *handle;

	/* dlsym したインターフェースへのハンドル */
	void *interface;

	/* プラグイン用グローバルコンテキストへのポインタ */
	tfsp_ctx_global *gctx;

	/* プラグイン用プロセスレベルコンテキストへのポインタ */
	tfsp_ctx_child *cctx;

	/* ローディング処理中などに発生したプラグインのエラー状態 */
	tfs_error_t *err;
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static tfs_error_t * _read_loader_config(tfs_pool_t *p, const char *confpath,
                                         tfsp_loader_config **conf);
static tfs_error_t * _scan_pluginsdir(tfsp_ctx_loader *loader,
                                      const char *name, tfs_varray_t **plugins);
static tfs_error_t * _read_plugin_config(tfs_pool_t *p, const char *pluginpath,
                                         tfsp_plugin_module **pm);
static tfs_error_t * _validate_plugin_config(tfs_pool_t *p, tfsp_plugin_module *pm);
static tfs_error_t * _load_plugin(tfs_pool_t *p, tfsp_plugin_module *pm);
static tfs_error_t * _unload_plugin(tfs_pool_t *p, tfsp_plugin_module *pm);
static tfs_error_t * _get_plugin_handler(tfs_pool_t *p, void *handle, void **plugin_h);
static const tfsp_ihdr_base * _ge_plugin_baseif(tfsp_plugin_module *pm);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたディレクトリ絶対パス rootpath 以下にあるプラグインライブラリの
 * ロード/アンロードを行うプラグインローダ(コンテキスト)の生成.
 *
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_create(const char *rootpath, tfsp_ctx_loader **loader)
{
	tfs_error_t *err = NULL;
	struct stat dinfo = { 0 };

	/* コンテキストの生成(loader は malloc で生成しなければならない) */
	*loader = malloc(sizeof(tfsp_ctx_loader));
	(*loader)->plugins  = NULL;

	/* コンフィグルートパスが与えられなかった場合 */
	if (IS_EMPTY(rootpath)) {
		/* $plugin_install_dir/plugins */
		rootpath = TFSP_INSTALL_ROOT"/plugins";
	}

	/* メモリアロケーターの生成 */
	tfs_pool_create(&(*loader)->pool);

	/* rootpath はディレクトリか? */
	if (stat(rootpath, &dinfo) == -1) {
		err = tfs_error_create((*loader)->pool, LOG_ERR, TFSP_OS_ERRNO(errno), TFS_EDESC1(
								"Failed to check plugin's root path (path = %s)", rootpath));
		return err;
	}
	else if (!S_ISDIR(dinfo.st_mode)) {
		/* ディレクトリではなかった */
		err = tfs_error_create((*loader)->pool, LOG_ERR, TFSP_ENOPLUGINROOTDIR, TFS_EDESC1(
								"The specified file path is not directory. (path = %s)", rootpath));
		return err;
	}
	(*loader)->rootpath = tfs_pstrdup((*loader)->pool, rootpath);

	/* プラグインローダ用コンフィグファイルの読み込み */
	(*loader)->confpath = tfs_make_full_path((*loader)->pool, rootpath, TFSP_LOADER_CONFFILE, NULL);
	if ((err = _read_loader_config((*loader)->pool, (*loader)->confpath, &(*loader)->conf)) != NULL) {
		return tfs_error_push(err, LOG_ERR, TFSP_ELOADERCONF_READ,
							TFS_EDESC0("Failed to read plugin's loader config."));
	}

	return NULL;
}

/**
 * プラグインローダ loader が管理するインストールされた有効なプラグインを
 * 全てローディングして使用可能にする.
 *
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_loadAll(tfsp_ctx_loader *loader,
                                               tfs_error_t **plugin_err)
{
	tfs_status_t rv;
	tfs_error_t *err = NULL;
	tfs_varray_t *new_plugins = NULL, *scan_plugins = NULL;
	tfsp_plugin_module *pm;
	tfs_int32_t size, i;
	tfs_pool_t *p = (loader != NULL) ? loader->pool : NULL;
	tfs_pool_t *pglobal = NULL;

	*plugin_err = NULL;	/* 初期化 */

	if (loader == NULL || IS_EMPTY(loader->rootpath)) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	/*
	 * プラグインディレクトリの中をスキャンして
	 * プラグイン情報を読む
	 */
	err = _scan_pluginsdir(loader, NULL, &scan_plugins);
	if (err != NULL) {
		return err;
	}

	/*
	 * プラグインをローディングする
	 */
	size = tfs_varray_size(scan_plugins);
	for (i = 0; i < size; i++) {

		/* プラグイン情報を取得 */
		pm = tfs_varray_get(scan_plugins, i);

		/* エラー状態だった? */
		if (PM_CTX(pm)->err != NULL) {
			/* *plugin_err に登録する */
			*plugin_err = (*plugin_err == NULL) ? PM_CTX(pm)->err :
							tfs_error_append(p, *plugin_err, PM_CTX(pm)->err);
			continue;	/* 次へ */
		}

		/* pm のロード (共有ライブラリ & アクティブの場合のみ) */
		err = NULL;
		if (pm->fileinfo->ftype == TFSP_IFFTYPE_SHAREDOBJ && pm->state->active) {
			err = _load_plugin(p, pm);
		}

		if (err == NULL && pm->fileinfo->ftype == TFSP_IFFTYPE_SHAREDOBJ) {
			err = _get_plugin_handler(p, PM_CTX(pm)->handle, &PM_CTX(pm)->interface);
			if (err == NULL) {
				/* ベースハンドラの取得 */
				const tfsp_ihdr_base *bs = _ge_plugin_baseif(pm);

				if (bs == NULL) continue;

				pglobal = NULL;
				/* グローバルコンテキストを生成
		 		 * (note)
				 * gctx は親に依存せず、cterminate で消したいからmallocでアロケートします.
				 * gctx->p からアロケートすると破棄する順番を意識しないとならないのでやりません */
				PM_CTX(pm)->gctx = malloc(sizeof(tfsp_ctx_global));
				memset(PM_CTX(pm)->gctx, 0, sizeof(tfsp_ctx_global));

				tfs_pool_create(&pglobal);	/* グローバルレベルプールの生成 */
				PM_CTX(pm)->gctx->p             = pglobal;
				PM_CTX(pm)->gctx->pluginpath    = tfs_pstrdup(pglobal, pm->pluginpath);
				PM_CTX(pm)->gctx->pluginconfdir =
					tfs_make_full_path(pglobal, pm->pluginpath, TFSP_PLUGIN_CONFIG_DIR, NULL);

				/* プラグイン初期化インターフェースの呼び出し */
				rv = (*bs->init)(PM_CTX(pm)->gctx);
				if (rv != TFSP_SUCCESS) {
					PM_CTX(pm)->err = tfs_error_create(pglobal, LOG_ERR, rv, TFS_EDESC1(
						"Failed to call \"init\" handler. (code = %d)", rv));
					*plugin_err = (*plugin_err == NULL) ? PM_CTX(pm)->err :
									tfs_error_append(pglobal, *plugin_err, PM_CTX(pm)->err);

					/* 失敗したのでモジュールをアンロードする */
					err = _unload_plugin(pglobal, pm);
					if (err != NULL) {
						*plugin_err = (*plugin_err == NULL) ? err :
										tfs_error_append(pglobal, *plugin_err, err);
					}
					continue;	/* 次のプラグインへ */
				}

				/* プラグインを登録する */
				if (new_plugins == NULL) {
					tfs_varray_create(p, size, &new_plugins);
				}
				tfs_varray_add(new_plugins, pm);
				tfs_log_notice(TFS_EDESC1("The loaded plugin (sharedobj) : %s", pm->name));
			}
		}
		else if (pm->fileinfo->ftype == TFSP_IFFTYPE_CGISCRIPT) {
				/* プラグインを登録する */
				if (new_plugins == NULL) {
					tfs_varray_create(p, size, &new_plugins);
				}
				tfs_varray_add(new_plugins, pm);
				tfs_log_notice(TFS_EDESC1("The loaded plugin (script) : %s", pm->name));
		}
		else {
			/* *plugin_err に登録する */
			*plugin_err = (*plugin_err == NULL) ? err :
							tfs_error_append(p, *plugin_err, err);
		}
	}
	tfs_varray_destroy(scan_plugins);	/* これはもう不要 */

	/* 結局ロードされたプラグインはあったの? */
	if (tfs_varray_size(new_plugins) > 0) {
		loader->plugins = new_plugins;	/* 新しいプラグインで置き換え */
		err = NULL;
		/* ロードしたプラグインの個数を表示 */
		tfs_log_notice(TFS_EDESC1("The number of loaded plugin : %d",
					tfs_varray_size(new_plugins)));
	}
	else {
		tfs_varray_destroy(new_plugins);
	}

	return NULL;
}

/**
 * プラグインディレクトリを再スキャンし、有効なプラグインをロードする.
 *
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_reloadAll(tfsp_ctx_loader *loader)
{
	tfs_error_t *err, *plugin_err = NULL;

	/* このプラグインローダからロードされた全てのプラグインをアンロードする */
	err = tfsp_loader_unloadAll(loader, &plugin_err);
	if (plugin_err != NULL) {
		/* コンソール出力する */
		tfs_error_dumpall(plugin_err, NULL, tfs_error_dump_stderr);
		tfs_error_destroy(plugin_err);	/* エラーは不要 */
	}
	if (err != NULL) {
		return err;
	}

	/* プラグインをロードする */
	err = tfsp_loader_loadAll(loader, &plugin_err);
	if (plugin_err != NULL) {
		/* コンソール出力する */
		tfs_error_dumpall(plugin_err, NULL, tfs_error_dump_stderr);
		tfs_error_destroy(plugin_err);	/* エラーは不要 */
	}

	return err;
}

/**
 * プラグインコンテキスト loader で管理されたプラグインをアンロードして
 * 使用できないようにする.
 *
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_unloadAll(tfsp_ctx_loader *loader,
                                                 tfs_error_t **plugin_err)
{
	tfs_pool_t *p;
	const tfsp_ihdr_base *bs;
	tfsp_plugin_module *pm;
	tfs_int32_t size, i;
	tfs_error_t *err = NULL;

	*plugin_err = NULL;	/* 初期化 */

	if (loader == NULL) {
		return tfs_error_create(NULL, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	if (loader->plugins == NULL) {
		return NULL;
	}

	/* プラグインをアンロードする */
	size = tfs_varray_size(loader->plugins);
	for (i = 0; i < size; i++) {
		pm = tfs_varray_get(loader->plugins, i);

		bs = _ge_plugin_baseif(pm);
		if (bs == NULL) continue;

		/* terminate のコール */
		(*bs->terminate)(PM_CTX(pm)->gctx);
		PM_CTX(pm)->interface = NULL;	/* インターフェースをNULL化 */

		/* Child レベルプールを破棄 */
		if (PM_CTX(pm)->gctx->p != NULL) {
			tfs_pool_destroy(PM_CTX(pm)->gctx->p);
			PM_CTX(pm)->gctx->p = NULL;
		}
		/* gctx を破棄する */
		free(PM_CTX(pm)->gctx);
		PM_CTX(pm)->gctx = NULL;

		/* モジュールのアンロード */
		err = _unload_plugin(p, pm);
		if (err != NULL) {
			if (*plugin_err == NULL) {
				*plugin_err = err;
			}
			else {
				*plugin_err = tfs_error_append(p, *plugin_err, err);
			}
			continue;
		}
	}

	/* プラグイン配列を破棄する */
	tfs_varray_destroy(loader->plugins);
	loader->plugins = NULL;

	return NULL;
}

/**
 * プラグインコンテキスト loader で管理されたプラグインの中から
 * タイプ ptype、名前 pname のプラグインをアンロードして使用できないようにする.
 *
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_unload(tfsp_ctx_loader *loader,
                                              tfsp_type_plugin ptype, const char *pname)
{
	/* ##### TODO 将来拡張用 */
	return NULL;
}

/**
 * プラグインのChildレベル初期化処理を実施する.
 *
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_cinit(tfsp_ctx_loader *loader,
                                                 tfs_error_t **plugin_err)
{
	tfs_pool_t *pchild = NULL;
	tfs_status_t rv;
	const tfsp_ihdr_base *bs;
	tfsp_plugin_module *pm;
	tfs_int32_t size, i;
	tfs_error_t *err = NULL;

	*plugin_err = NULL;	/* 初期化 */

	if (loader == NULL) {
		return tfs_error_create(NULL, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
					"The parameter was missing."));
	}

	if (loader->plugins == NULL) {
		return NULL;
	}

	/* 全てのプラグインモジュールを調べる */
	size = tfs_varray_size(loader->plugins);
	for (i = 0; i < size; i++) {
		pm = tfs_varray_get(loader->plugins, i);

		bs = _ge_plugin_baseif(pm);
		if (bs == NULL) continue;

		tfs_pool_create(&pchild);	/* Childレベルプールの生成 */

		/* Child コンテキストを生成.
		 * (note)
		 *   cctx は親に依存せず、cterminate で消したいからmallocでアロケートします.
		 *   pchild からアロケートすると破棄する順番を意識しないとならないのでやりません */
		PM_CTX(pm)->cctx = malloc(sizeof(tfsp_ctx_child));
		memset(PM_CTX(pm)->cctx, 0 , sizeof(tfsp_ctx_child));

		PM_CTX(pm)->cctx->global = PM_CTX(pm)->gctx;
		PM_CTX(pm)->cctx->p = pchild;

		/* cinit のコール */
		rv = (*bs->cinit)(PM_CTX(pm)->cctx);
		if (rv != TFSP_SUCCESS) {
			err = tfs_error_create(pchild, LOG_ERR, rv, TFS_EDESC1(
						"Failed to call \"cinit\" handler. (code = %d)", rv));
			*plugin_err = (*plugin_err == NULL) ?
				err : tfs_error_append(pchild, *plugin_err, err);
		}
	}

	return NULL;
}


/**
 * プラグインのChildレベル終了処理を実施する.
 *
 */
TFS_DECLARE(void) tfsp_loader_cterminate(tfsp_ctx_loader *loader,
                                                 tfs_error_t **plugin_err)
{
	const tfsp_ihdr_base *bs;
	tfsp_plugin_module *pm;
	tfs_int32_t size, i;

	*plugin_err = NULL;	/* 初期化 */

	if (loader == NULL) {
		return;
	}

	if (loader->plugins == NULL) {
		return;
	}

	/* プラグインのcterminate インターフェースを呼び出す */
	size = tfs_varray_size(loader->plugins);
	for (i = 0; i < size; i++) {
		pm = tfs_varray_get(loader->plugins, i);

		/* インターフェースの取得 */
		bs = _ge_plugin_baseif(pm);
		if (bs == NULL) continue;

		/* cterminate のコール */
		(*bs->cterminate)(PM_CTX(pm)->cctx);

		/* Child レベルプールを破棄 */
		if (PM_CTX(pm)->cctx->p != NULL) {
			tfs_pool_destroy(PM_CTX(pm)->cctx->p);
			PM_CTX(pm)->cctx->p = NULL;
		}

		/* cctx を破棄 */
		free(PM_CTX(pm)->cctx);
		PM_CTX(pm)->cctx = NULL;
	}
}

/**
 * 全てのプラグインをアンロードしてプラグインローダコンテキストを破棄する.
 *
 */
TFS_DECLARE(void) tfsp_loader_destroy(tfsp_ctx_loader *loader)
{
	tfs_error_t *err, *plugin_err = NULL;

	if (loader == NULL) {
		return;
	}

	/* このプラグインローダからロードされた全てのプラグインをアンロードする */
	err = tfsp_loader_unloadAll(loader, &plugin_err);
	if (err != NULL) {
		/* コンソール出力する */
		tfs_error_dumpall(err, NULL, tfs_error_dump_stderr);
		tfs_error_destroy(err);	/* エラーは不要 */
	}

	if (plugin_err != NULL) {
		/* コンソール出力する */
		tfs_error_dumpall(plugin_err, NULL, tfs_error_dump_stderr);
		tfs_error_destroy(plugin_err);	/* エラーは不要 */
	}

	/* プラグインローダ用コンフィグを破棄する */
	if (loader->conf != NULL) {
		tfs_varray_destroy(loader->conf->supported_if);
	}
	loader->conf = NULL;

	if (loader->pool != NULL) {
		/* 作業用プールを破棄する */
		tfs_pool_destroy(loader->pool);
		loader->pool = NULL;
	}
	/* プラグインローダを破棄する(これ移行はloader に触ってはならない) */
	free(loader);

	return;
}

/**
 * プラグインコンテキスト loader で管理されたプラグインの中から
 * タイプ ptype、名前 pname のプラグインインターフェースを取得する.
 * pname がNULLであれば、ptype のプラグインインターフェースの内、
 * プラグインチェインの先頭にあるプラグインインターフェースを取得します.
 * 
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_lookupIf(tfs_pool_t *pool,
                                                tfsp_ctx_loader *loader,
                                                tfsp_type_plugin ptype, const char *pname,
                                                const tfsp_plugin_module **pm)
{
	tfs_pool_t *p = (pool != NULL) ? pool : ((loader != NULL) ? loader->pool : NULL);
	tfs_int32_t size, i;
	tfsp_plugin_module *_pm;

	*pm = NULL;	/* 初期化 */

	if (loader == NULL) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	if (loader->plugins == NULL) {
		return NULL;	/* エラーではありません */
	}

	/* ptype, pname のプラグインライブラリを探す */
	size = tfs_varray_size(loader->plugins);
	for (i = 0; i < size; i++) {
		_pm = tfs_varray_get(loader->plugins, i);

		if (_pm->fileinfo->ftype != TFSP_IFFTYPE_SHAREDOBJ) {
			continue;
		}

		if (ptype == _pm->if_info->itype) {
			/* pname が指定されていた場合 */
			if (IS_FILLED(pname)) {
				if (strcmp(pname, _pm->name) == 0) {
					*pm = _pm;
					break;
				}
			}
			else {
				*pm = _pm;
				break;
			}
		}
	}

	return NULL;
}

/**
 * プラグインコンテキスト loader で管理されたプラグインの中から
 * アクティブなプラグイン情報を取得する.
 * 
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_lookupAllIf(tfs_pool_t *pool,
                                                tfsp_ctx_loader *loader,
                                                tfs_varray_t **plugins)
{
	tfs_pool_t *p = (pool != NULL) ? pool : ((loader != NULL) ? loader->pool : NULL);

	*plugins = NULL;	/* 初期化 */

	if (loader == NULL) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	if (loader->plugins == NULL) {
		return NULL;	/* エラーではありません */
	}

	/* loader->plugins をコピーする */
	*plugins = tfs_varray_clone(pool, loader->plugins);

	return NULL;
}

/**
 * プラグインローダ loader のプラグインルートディレクトリパスを取得する.
 */
TFS_DECLARE(const char *) tfsp_loader_get_rootpath(tfs_pool_t *pool,
                                                   tfsp_ctx_loader *loader)
{
	tfs_pool_t *p;
	if (loader == NULL) return NULL;

	p = (pool != NULL) ? pool : loader->pool;

	/* コピーして渡す(変更を避けるため) */
	return tfs_pstrdup(p, loader->rootpath);
}

/**
 * プラグイングローバルコンテキストを取得する.
 *
 */
TFS_DECLARE(const tfsp_ctx_global *) tfsp_pm_get_gctx(const tfsp_plugin_module *pm)
{
	if (pm == NULL) return NULL;

	return PM_CTX(pm)->gctx;
}

/**
 * プラグインChild レベルコンテキストを取得する.
 *
 */
TFS_DECLARE(const tfsp_ctx_child *) tfsp_pm_get_cctx(const tfsp_plugin_module *pm)
{
	if (pm == NULL) return NULL;

	return PM_CTX(pm)->cctx;
}

/**
 * プラグインモジュールpm が持つインターフェースを取得する.
 *
 */
TFS_DECLARE(void *) tfsp_pm_get_interface(const tfsp_plugin_module *pm)
{
	if (pm == NULL) return NULL;

	return PM_CTX(pm)->interface;
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * プラグインローダコンフィグを読み込み、内容をパースしてconf に入れる.
 *
 * @param p tfs_pool_t * 作業用のプール
 * @param confpath const char * プラグインコンフィグファイルのパス
 * @param conf tfsp_loader_config ** パース結果
 * @return tfs_error_t *
 */
static tfs_error_t * _read_loader_config(tfs_pool_t *p, const char *confpath,
                                         tfsp_loader_config **conf)
{
	tfs_status_t rv;
	struct stat finfo = { 0 };
	tfs_xml_parser *parser = NULL;
	tfs_xml_doc *doc = NULL;
	tfs_xml_elem *root, *elem, *child, *child2;
	char *data;

	if (p == NULL || IS_EMPTY(confpath)) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	*conf = NULL;	/* 初期化 */

	/* ファイルの状態チェック */
	if (stat(confpath, &finfo) == -1) {
		return tfs_error_create(p, LOG_ERR, TFSP_OS_ERRNO(errno), TFS_EDESC1(
								"Failed to stat config file path. (path = %s)", confpath));
	}
	/* 通常ファイルではなかった */
	else if (!S_ISREG(finfo.st_mode)) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOLOADERCONF, TFS_EDESC1(
								"This file path was not regular file. (path = %s)", confpath));
	}

	/* プラグインコンフィグファイルのパース */
	if ((rv = tfs_xml_parse_file(NULL, confpath, &parser, &doc)) != TFS_SUCCESS) {
		char *msg = tfs_xml_geterror(parser, rv);

		if (IS_FILLED(msg)) {
			return tfs_error_create(p, LOG_ERR, TFSP_UTIL_ERRNO(rv), TFS_EDESC2(
									"Failed to parse config XML file. (path = %s) "
									"(Reason : %s)", confpath, msg));
		}
		else {
			return tfs_error_create(p, LOG_ERR, TFSP_UTIL_ERRNO(rv), TFS_EDESC1(
									"Failed to parse config XML file. (path = %s)", confpath));
		}
	}

	if (doc == NULL || (root = tfs_xml_get_rootnode(doc)) == NULL || root->first_child == NULL) {
		if (parser != NULL) {
			tfs_xml_parser_destroy(parser);
		}
		return tfs_error_create(p, LOG_ERR, TFSP_ELOADERCONF_READ, TFS_EDESC1(
								"Failed to get root node from XML document. (path = %s)", confpath));
	}

	/* DOM ノードから設定値を取得
	 * [ DTD ]
	 *
	 * <!ELEMENT tfspglobal (name, version, summary, description?, supported-interfaces, logging, author) >
	 *   <!ELEMENT name        (#PCDATA) >
	 *   <!ELEMENT version     (#PCDATA) >
	 *   <!ELEMENT summary     (#PCDATA) >
	 *   <!ELEMENT description (#PCDATA) >
	 *   <!ELEMENT supported-interfaces (interface+) >
	 *     <!ELEMENT interface (itype, iversion) >
	 *       <!ELEMENT itype     (ccipher | ffilter) >
	 *       <!ELEMENT ccipher   EMPTY >
	 *       <!ELEMENT ffilter   EMPTY >
	 *       <!ELEMENT iversion  (current, revision?, age?) >
	 *       <!ELEMENT current   (#PCDATA) >
	 *       <!ELEMENT revision  (#PCDATA) >
	 *       <!ELEMENT age       (#PCDATA) >
	 *  <!ELEMENT logging (loglevel, logmethod) >
	 *    <!ELEMENT loglevel (debug | info | notice | warn | error | crit | alert | emerg) >
	 *      <!ELEMENT debug  EMPTY >
	 *      <!ELEMENT info   EMPTY >
	 *      <!ELEMENT notice EMPTY >
	 *      <!ELEMENT warn   EMPTY >
	 *      <!ELEMENT error  EMPTY >
	 *      <!ELEMENT crit   EMPTY >
	 *      <!ELEMENT alert  EMPTY >
	 *      <!ELEMENT emerg  EMPTY >
	 *    <!ELEMENT logmethod (aplog | customfile) >
	 *      <!ELEMENT aplog      EMPTY >
	 *      <!ELEMENT customfile EMPTY >
	 *   <!ELEMENT author (vender, copyright, homepageurl?, mailaddress?) >
	 *     <!ELEMENT vender      (#PCDATA) >
	 *     <!ELEMENT copyright   (#PCDATA) >
	 *     <!ELEMENT homepageurl (#PCDATA) >
	 *     <!ELEMENT mailaddress (#PCDATA) > 
	 */
	*conf = tfs_pcalloc(p, sizeof(tfsp_loader_config));
	(*conf)->supported_if = NULL;

	for (elem = root->first_child; elem; elem = elem->next) {
		data = tfs_xml_get_cdata(elem, 0, p);
		if (strcmp(elem->name, "name") == 0) {
			(*conf)->name = data;
		}
		else if (strcmp(elem->name, "version") == 0) {
			(*conf)->version = data;
		}
		else if (strcmp(elem->name, "summary") == 0) {
			(*conf)->summary = data;
		}
		else if (strcmp(elem->name, "description") == 0) {
			(*conf)->description = data;
		}
		/* サポートインターフェース情報 */
		else if (strcmp(elem->name, "supported-interfaces") == 0) {
			tfsp_interface_info *interface;
			tfs_xml_elem *if_elem;

			if ((*conf)->supported_if == NULL) {
				tfs_varray_create(p, 5, &(*conf)->supported_if);
			}

			for (if_elem = elem->first_child; if_elem; if_elem = if_elem->next) {

				if (strcmp(if_elem->name, "interface") == 0) {

					interface = tfs_pcalloc(p, sizeof(tfsp_interface_info));
					interface->itype = TFSP_IFTYPE_UNKNOWN;

					for (child = if_elem->first_child; child; child = child->next) {
						if (strcmp(child->name, "itype") == 0) {
							child2 = child->first_child;
							if (child2 != NULL && strcmp(child2->name, "ccipher") == 0) {
								interface->itype = TFSP_IFTYPE_CCIOHER;
							}
							else if (child2 != NULL && strcmp(child2->name, "ffilter") == 0) {
								interface->itype = TFSP_IFTYPE_FFILTER;
							}
							else {
								/* 無視 */
								interface->itype = TFSP_IFTYPE_UNKNOWN;
							}
						}
						else if (strcmp(child->name, "iversion") == 0) {
							child2 = child->first_child;
							data = tfs_xml_get_cdata(child2, 0, p);

							if (interface->iversion == NULL) {
								interface->iversion = tfs_pcalloc(p, sizeof(tfsp_if_version));
							}

							if (child2 != NULL && strcmp(child2->name, "current") == 0) {
								interface->iversion->current = atoi(data);
							}
							else if (child2 != NULL && strcmp(child2->name, "revision") == 0) {
								interface->iversion->revision = atoi(data);
							}
							else if (child2 != NULL && strcmp(child2->name, "age") == 0) {
								interface->iversion->age = atoi(data);
							}
							else { /* 無視 */
							}
						}
						else { /* 無視 */
						}
					}

					if (interface->itype != TFSP_IFTYPE_UNKNOWN) {
						tfs_varray_add((*conf)->supported_if, interface);
					}
				}
				else { /* 無視 */
				}
			}
		}
		/* 作成者情報 */
		else if (strcmp(elem->name, "author") == 0) {

			if ((*conf)->author == NULL) {
				(*conf)->author = tfs_pcalloc(p, sizeof(tfsp_author_info));
			}

			for (child = elem->first_child; child; child = child->next) {
				data = tfs_xml_get_cdata(child, 0, p);

				if (strcmp(child->name, "vender") == 0) {
					(*conf)->author->vender = data;
				}
				else if (strcmp(child->name, "copyright") == 0) {
					(*conf)->author->copyright = data;
				}
				else if (strcmp(child->name, "homepageurl") == 0) {
					(*conf)->author->homepageurl = data;
				}
				else if (strcmp(child->name, "mailaddress") == 0) {
					(*conf)->author->mailaddress = data;
				}
				else { /* 無視 */
				}
			}
		}
		else { /* 無視 */
		}
	}

	/* パーサを破棄 */
	tfs_xml_parser_destroy(parser);

	return NULL;
}

/**
 * 指定されたname のプラグインのディレクトリをスキャンして
 * アクティブだとマークされたプラグインのプラグイン情報を取得する.
 * 情報取得において発生したプラグインに起因するエラーはplugins の中の
 * プラグイン構造体に記録されます. エラー判定には各プラグイン構造体も参照すること.
 *
 * @param loader tfsp_ctx_loader *
 * @param name const char * name プラグインの名前. NULL で全てを指定したことになる.
 * @param plugins tfs_varray_t ** 取得したプラグイン情報を保持する配列
 *   ここに格納されたプラグインがエラー状態であることもあります. 上位層
 * @return tfs_error_t * エラー情報
 */
static tfs_error_t * _scan_pluginsdir(tfsp_ctx_loader *loader,
                                      const char *name, tfs_varray_t **plugins)
{
	tfs_error_t *err = NULL;
	DIR *dir = NULL;
	struct dirent *de;
	struct stat dinfo = { 0 };
	char *dname, *pluginpath;
	tfsp_plugin_module *pm = NULL;
	tfs_pool_t *p = (loader != NULL) ? loader->pool : NULL;

	*plugins = NULL;	/* 初期化 */

	if (loader == NULL || IS_EMPTY(loader->rootpath)) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	/* rootpath のディレクトリを開く */
	if ((dir = opendir(loader->rootpath)) == NULL) {
		return tfs_error_create(p, LOG_ERR, TFSP_OS_ERRNO(errno), TFS_EDESC1(
					"Failed to open rootpath directory (path = %s)", loader->rootpath));
	}

	/* ディレクトリエントリを読む */
	while ((de = readdir(dir)) != NULL) {
		dname = de->d_name;

		/* "." と ".." は除外する */
		if (dname &&
			(strcmp(dname, ".") == 0 || strcmp(dname, "..") == 0)) {
			continue;
		}

		/* name が指定されていればname 以外は除外する */
		if (name != NULL && strcmp(dname, name) != 0) {
			continue;
		}

		/* 個々のプラグインのトップディレクトリパスを算出 */
		pluginpath = tfs_make_full_path(p, loader->rootpath, dname, NULL);

		/* pluginpath のチェック */
		if (stat(pluginpath, &dinfo) == -1 || !S_ISDIR(dinfo.st_mode)) {
			continue;	/* 失敗した、ディレクトリではなかったらスキップ */
		}

		/* 個々のプラグインディレクトリからの情報取得 */
		err = _read_plugin_config(p, pluginpath, &pm);
		if (err != NULL) {
			/* エラー情報を記録しておく */
			if (pm != NULL) {
				PM_CTX(pm)->err = err;
			}
			else {
				/* 致命的. バグ. もうやめる */
				goto cleanup_resource;
			}
		}

		/* 結果はどうであれプラグイン一覧には追加する */
		if (*plugins == NULL) {
			/* (note) この配列は比較的早い段階で破棄されるので
			 * 独自のメモリアロケーターを使う */
			tfs_varray_create(NULL, 5, plugins);
		}
		tfs_varray_add(*plugins, pm);
	}

cleanup_resource:
	(void) closedir(dir);

	return err;
}

/**
 * pluginpath が示すディレクトリを読み込み、プラグインコンフィグファイルを
 * 取得し、プラグイン固有の設定値を読む. 値の検証も行います.
 *
 * @param p tfs_pool_t * 作業用のプール
 * @param pluginpath const char *
 * @param pm tfsp_plugin_module **
 * @return tfs_error_t * エラー情報へのポインタ
 * 	プラグインそのもののエラーは全て警告レベルにしておきます.
 */
static tfs_error_t * _read_plugin_config(tfs_pool_t *p, const char *pluginpath,
                                         tfsp_plugin_module **pm)
{
	struct stat finfo = { 0 };
	char *confpath;
	tfs_xml_parser *parser = NULL;
	tfs_xml_doc *doc = NULL;
	tfs_xml_elem *root, *elem, *elem2, *elem3, *child;
	tfs_status_t rv;
	char *data;

	/* プラグイン情報インスタンスの生成
	 * (note) 何があってもこれは生成します */
	*pm = tfs_pcalloc(p, sizeof(tfsp_plugin_module));
	(*pm)->next = NULL;
	(*pm)->ctx = tfs_pcalloc(p, sizeof(tfsp_plugin_modulectx));

	/* プラグインコンフィグファイルのチェック */
	confpath = tfs_make_full_path(p, pluginpath, TFSP_PLUGIN_CONFFILE, NULL);
	if (stat(confpath, &finfo) == -1) {
		return tfs_error_create(p, LOG_WARNING, TFSP_EINVALID_PLUGINCONF, TFS_EDESC2(
								"Failed to stat for plugin's config XML file. "
								"(path = %s, code = %d)", confpath, errno));
	}
	else if (!S_ISREG(finfo.st_mode)) {
		/* プラグインコンフィグがなかったらプラグインとは認めない */
		return tfs_error_create(p, LOG_WARNING, TFSP_EINVALID_PLUGINCONF, TFS_EDESC1(
								"This plugin's file path was not regular file. "
								"(path = %s)", confpath));
	}

	/* 各プラグインコンフィグファイルのパース */
	if ((rv = tfs_xml_parse_file(NULL, confpath, &parser, &doc)) != TFS_SUCCESS) {
		char *msg = tfs_xml_geterror(parser, rv);
		tfs_error_t *err = NULL;

		if (IS_FILLED(msg)) {
			err = tfs_error_create(p, LOG_WARNING, TFSP_UTIL_ERRNO(rv), TFS_EDESC2(
									"Failed to parse plugins's XML file. (path = %s) "
									"(Reason : %s)", confpath, msg));
		}
		else {
			err = tfs_error_create(p, LOG_WARNING, TFSP_UTIL_ERRNO(rv), TFS_EDESC1(
									"Failed to parse plugin's XML file. (path = %s)", confpath));
		}

		tfs_xml_parser_destroy(parser); /* パーサを破棄 */
		return err;
	}

	/* 有効なXMLエレメントノードが1つも無かった */
	if (doc == NULL || (root = tfs_xml_get_rootnode(doc)) == NULL || root->first_child == NULL) {
		if (parser != NULL) {
			tfs_xml_parser_destroy(parser); /* パーサを破棄 */
		}
		return tfs_error_create(p, LOG_WARNING, TFSP_EEMPTYELEM_PLUGINCONF, TFS_EDESC1(
								"Failed to get root node from plugin's XML document. "
								"(path = %s)", confpath));
	}

	/* プラグインXMLのDOM ノードから設定値を取得
	 * [ DTD ]
	 * <!ELEMENT tfsplugin (name, version, summary, description?,
	 *                      implemented-interface, author, fileinfo, configpageinfo?, state?) >
	 *   <!ELEMENT name        (#PCDATA) >
	 *   <!ELEMENT version     (#PCDATA) >
	 *   <!ELEMENT summary     (#PCDATA) >
	 *   <!ELEMENT description (#PCDATA) >
	 *   <!ELEMENT implemented-interface (interface) >
	 *     <!ELEMENT interface (itype, iversion) >
	 *       <!ELEMENT itype     (ccipher | ffilter) >
	 *         <!ELEMENT ccipher   EMPTY >
	 *         <!ELEMENT ffilter   EMPTY >
	 *       <!ELEMENT iversion  (current, revision?, age?) >
	 *         <!ELEMENT current   (#PCDATA) >
	 *         <!ELEMENT revision  (#PCDATA) >
	 *         <!ELEMENT age       (#PCDATA) >
	 *
	 *   <!ELEMENT author (vender, copyright, homepageurl?, mailaddress?) >
	 *     <!ELEMENT vender      (#PCDATA) >
	 *     <!ELEMENT copyright   (#PCDATA) >
	 *     <!ELEMENT homepageurl (#PCDATA) >
	 *     <!ELEMENT mailaddress (#PCDATA) >
	 *
	 *   <!ELEMENT fileinfo (filepath, filetype, checksum?)>
	 *     <!ELEMENT filepath (#PCDATA) >
	 *     <!ELEMENT filetype (sharedobj | cgiscript) >
	 *     <!ELEMENT sharedobj    EMPTY >
	 *     <!ELEMENT cgiscript    EMPTY > ; (予約エレメント)
	 *     <!ELEMENT checksum (#PCDATA) >
	 *
	 *   <!ELEMENT configpageinfo (globalpage?, locationpage?) >
	 *
	 *     // (記述) /cgi-bin/$name, (URIパターン) /cgi-bin/$name
	 *     <!ELEMENT globalpage   (startupurl,iconpath?, displayname?) >
	 *
	 *     // (記述) /cgi-bin/$name, (URIパターン) $root/.plugin/$name/cgi-bin/$name
	 *     <!ELEMENT locationpage (startupurl,iconpath?, displayname?, screentype?, usertype?) >
	 *       <!ELEMENT startupurl  (#PCDATA) >
	 *       <!ELEMENT iconpath    (#PCDATA) >
	 *       <!ELEMENT displayname (#PCDATA) >
	 *       <!ATTLIST displayname lang CDATA "ja">
	 *       <!ELEMENT screentype  (file | personal | utility) >
	 *       <!ELEMENT usertype    (admin, normal, limited) >
	 *
	 *   <!ELEMENT state (active | inactive) >
	 *     <!ELEMENT active   EMPTY >
	 *     <!ELEMENT inactive EMPTY >
	 */
	for (elem = root->first_child; elem; elem = elem->next) {

		data = tfs_xml_get_cdata(elem, 0, p);
		if (strcmp(elem->name, "name") == 0) {
			(*pm)->name = data;
		}
		else if (strcmp(elem->name, "version") == 0) {
			(*pm)->version = data;
		}
		else if (strcmp(elem->name, "summary") == 0) {
			(*pm)->summary = data;
		}
		else if (strcmp(elem->name, "description") == 0) {
			(*pm)->description = data;
		}
		/* 実装インターフェース情報 */
		else if (strcmp(elem->name, "implemented-interface") == 0) {
			tfs_xml_elem *if_elem, *child2;

			for (if_elem = elem->first_child; if_elem; if_elem = if_elem->next) {

				if (strcmp(if_elem->name, "interface") == 0) {

					if ((*pm)->if_info == NULL) {
						(*pm)->if_info = tfs_pcalloc(p, sizeof(tfsp_interface_info));
					}
					(*pm)->if_info->itype = TFSP_IFTYPE_UNKNOWN;

					for (child = if_elem->first_child; child; child = child->next) {
						if (strcmp(child->name, "itype") == 0) {
							child2 = child->first_child;
							if (child2 != NULL && strcmp(child2->name, "ccipher") == 0) {
								(*pm)->if_info->itype = TFSP_IFTYPE_CCIOHER;
							}
							else if (child2 != NULL && strcmp(child2->name, "ffilter") == 0) {
								(*pm)->if_info->itype = TFSP_IFTYPE_FFILTER;
							}
							else { /* 無視 */
							}
						}
						else if (strcmp(child->name, "iversion") == 0) {
							child2 = child->first_child;
							data = tfs_xml_get_cdata(child2, 0, p);

							if ((*pm)->if_info->iversion == NULL) {
								(*pm)->if_info->iversion = tfs_pcalloc(p, sizeof(tfsp_if_version));
							}

							if (child2 != NULL && strcmp(child2->name, "current") == 0) {
								(*pm)->if_info->iversion->current = atoi(data);
							}
							else if (child2 != NULL && strcmp(child2->name, "revision") == 0) {
								(*pm)->if_info->iversion->revision = atoi(data);
							}
							else if (child2 != NULL && strcmp(child2->name, "age") == 0) {
								(*pm)->if_info->iversion->age = atoi(data);
							}
							else { /* 無視 */
							}
						}
						else { /* 無視 */
						}
					}
				}
				else { /* 無視 */
				}
			}
		}
		/* 作成者情報 */
		else if (strcmp(elem->name, "author") == 0) {

			if ((*pm)->author == NULL) {
				(*pm)->author = tfs_pcalloc(p, sizeof(tfsp_author_info));
			}

			for (child = elem->first_child; child; child = child->next) {
				data = tfs_xml_get_cdata(child, 0, p);

				if (strcmp(child->name, "vender") == 0) {
					(*pm)->author->vender = data;
				}
				else if (strcmp(child->name, "copyright") == 0) {
					(*pm)->author->copyright = data;
				}
				else if (strcmp(child->name, "homepageurl") == 0) {
					(*pm)->author->homepageurl = data;
				}
				else if (strcmp(child->name, "mailaddress") == 0) {
					(*pm)->author->mailaddress = data;
				}
				else { /* 無視 */
				}
			}
		}
		/* ファイル情報 */
		else if (strcmp(elem->name, "fileinfo") == 0) {

			if ((*pm)->fileinfo == NULL) {
				(*pm)->fileinfo = tfs_pcalloc(p, sizeof(tfsp_plugin_fileinfo));
				(*pm)->fileinfo->ftype = TFSP_IFFTYPE_UNKNOWN;
			}

			for (child = elem->first_child; child; child = child->next) {
				data = tfs_xml_get_cdata(child, 0, p);

				if (strcmp(child->name, "filepath") == 0) {
					(*pm)->fileinfo->filepath = data;
				}
				else if (strcmp(child->name, "filetype") == 0) {
					elem2 = child->first_child;

					if (elem2 != NULL && strcmp(elem2->name, "sharedobj") == 0) {
						(*pm)->fileinfo->ftype = TFSP_IFFTYPE_SHAREDOBJ;
					}
					else if (elem2 != NULL && strcmp(elem2->name, "cgiscript") == 0) {
						(*pm)->fileinfo->ftype = TFSP_IFFTYPE_CGISCRIPT;
					}
					else { /* 未分類 */
					}
				}
				else if (strcmp(child->name, "checksum") == 0) {
					(*pm)->fileinfo->checksum = data;
				}
				else { /* 無視 */
				}
			}
		}
		/* コンフィグページ */
		else if (strcmp(elem->name, "configpageinfo") == 0) {

			if ((*pm)->cpageinfo == NULL) {
				(*pm)->cpageinfo = tfs_pcalloc(p, sizeof(tfsp_plugin_cpageinfo));
			}

			for (child = elem->first_child; child; child = child->next) {

				tfsp_plugin_cpage *cpage = NULL;
				data = NULL;

				for (elem2 = child->first_child; elem2; elem2 = elem2->next) {

					if (cpage == NULL) {
						cpage = tfs_pcalloc(p, sizeof(tfsp_plugin_cpage));
						cpage->screentype = TFSP_SCREENTYPE_UNKNOWN;
					}

					data = tfs_xml_get_cdata(elem2, 0, p);
					if (elem2 != NULL && strcmp(elem2->name, "startupurl") == 0) {
						cpage->uri = data;
					}
					else if (elem2 != NULL && strcmp(elem2->name, "iconpath") == 0) {
						cpage->iconpath = data;
					}
					else if (elem2 != NULL && strcmp(elem2->name, "displayname") == 0) {
						if (elem2->attr == NULL || IS_EMPTY(elem2->attr->name) ||
							(strcmp(elem2->attr->name, "xml:lang") == 0 &&
							 strcmp(elem2->attr->value, "ja") == 0)) {
							cpage->displayname = data;
						}
						else {
							cpage->displayname_en = data;
						}
					}
					else if (elem2 != NULL && strcmp(elem2->name, "screentype") == 0) {
						for (elem3 = elem2->first_child; elem3; elem3 = elem3->next) {
							if (elem3 != NULL && strcmp(elem3->name, "file") == 0) {
								cpage->screentype = TFSP_SCREENTYPE_FILE;
							}
							else if (elem3 != NULL && strcmp(elem3->name, "personal") == 0) {
								cpage->screentype = TFSP_SCREENTYPE_PERSONAL;
							}
							else if (elem3 != NULL && strcmp(elem3->name, "utility") == 0) {
								cpage->screentype = TFSP_SCREENTYPE_UTILITY;
							}
						}
					}
					else if ((elem2 != NULL && strcmp(elem2->name, "usertype") == 0)) {
						for (elem3 = elem2->first_child; elem3; elem3 = elem3->next) {
							if (elem3 != NULL && strcmp(elem3->name, "admin") == 0) {
								cpage->use_admin = 1;
							}
							else if (elem3 != NULL && strcmp(elem3->name, "normal") == 0) {
								cpage->use_normal = 1;
							}
							else if (elem3 != NULL && strcmp(elem3->name, "limited") == 0) {
								cpage->use_limited = 1;
							}
							else if (elem3 != NULL && strcmp(elem3->name, "groupleader") == 0) {
								cpage->use_groupleader = 1;
							}
						}
					}
					else { /* 無視 */
					}
				}

				if (strcmp(child->name, "globalpage") == 0) {
					(*pm)->cpageinfo->g_cpuri = cpage;
				}
				else if (strcmp(child->name, "locationpage") == 0) {
					(*pm)->cpageinfo->l_cpuri = cpage;
				}
				else { /* 無視 */
				}
			}
		}
		/* 状態 */
		else if (strcmp(elem->name, "state") == 0) {
			if ((*pm)->state == NULL) {
				(*pm)->state = tfs_pcalloc(p, sizeof(tfsp_plugin_state));
			}

			(*pm)->state->active = 0;
			for (child = elem->first_child; child; child = child->next) {

				if (strcmp(child->name, "active") == 0) {
					(*pm)->state->active = 1;
				}
				else if (strcmp(child->name, "inactive") == 0) {
					(*pm)->state->active = 0;
				}
				else { /* 無視 */
				}
			}
		}
		else { /* 無視 */
		}
	}

	if ((*pm)->state == NULL) {
		(*pm)->state = tfs_pcalloc(p, sizeof(tfsp_plugin_state));
		(*pm)->state->active = 0;
	}

	(*pm)->pluginpath = pluginpath;
	(*pm)->confpath   = confpath;

	/* パーサを破棄 */
	tfs_xml_parser_destroy(parser);

	/*
	 * 読み込んだ値の検証値の検証
	 */
	return _validate_plugin_config(p, *pm);
}

/**
 * pm が示すプラグインコンフィグの検証.
 *
 * @param p tfs_pool_t * 作業用のプール
 * @param pm tfsp_plugin_module * モジュール構造体へのポインタ
 * @return tfs_error_t * エラー構造体へのポインタ. NULLで正常
 */
static tfs_error_t * _validate_plugin_config(tfs_pool_t *p, tfsp_plugin_module *pm)
{
	struct stat finfo = { 0 };

	if (pm == NULL) {
		return tfs_error_create(p, LOG_WARNING, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	/* 名前 */
	if (IS_EMPTY(pm->name)) {
		return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"name\" (tfsplugin) was missing or empty value."));
	}

	/* バージョン */
	if (IS_EMPTY(pm->version)) {
		return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"version\" (tfsplugin) was missing or empty value."));
	}

	/* 短い説明文 */
	if (IS_EMPTY(pm->summary)) {
		return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"summary\" (tfsplugin) was missing or empty value."));
	}

	/* インターフェース情報 */
	if (pm->if_info == NULL) {
		return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"implemented-interface\" (tfsplugin) was missing or empty value."));
	}
	else {
		/* インターフェース種類 */
		if (pm->if_info->itype == TFSP_IFTYPE_UNKNOWN && pm->fileinfo->ftype == TFSP_IFFTYPE_SHAREDOBJ) {
			return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"itype\" (tfsplugin) was missing or empty value "
								"unsupported value."));
		}

		/* インターフェースバージョン */
		if (pm->if_info->iversion == NULL) {
			return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"iversion\" (tfsplugin) was missing or empty value."));
		}
	}

	/* 作成者情報 */
	if (pm->author == NULL) {
		return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"author\" (tfsplugin) was missing or empty value."));
	}
	else {
		/* ベンダー情報 */
		if (IS_EMPTY(pm->author->vender)) {
			return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"vender\" (tfsplugin) was missing or empty value."));
		}

		/* コピーライト */
		if (IS_EMPTY(pm->author->copyright)) {
			return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"copyright\" (tfsplugin) was missing or empty value."));
		}
	}

	/* プラグインファイル情報 */
	if (pm->fileinfo == NULL) {
		return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"fileinfo\" (tfsplugin) was missing or empty value."));
	}
	else {

		/* プラグインライブラリファイル */
		if (IS_EMPTY(pm->fileinfo->filepath)) {
			return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"filepath\" (tfsplugin) was missing or empty value."));
		}

		/* filepath のチェック */
		if (stat(pm->fileinfo->filepath, &finfo) == -1) {
			return tfs_error_create(p, LOG_WARNING, TFSP_OS_ERRNO(errno), TFS_EDESC1(
								"Failed to stat plugin's library file. "
								"(path = %s)", pm->fileinfo->filepath));
		}
		/* 通常ファイルではなかった */
		else if (!S_ISREG(finfo.st_mode)) {
			return tfs_error_create(p, LOG_WARNING, TFSP_ENOLOADERCONF, TFS_EDESC1(
								"This plugin's library file was not regular file. "
								"(path = %s)", pm->fileinfo->filepath));
		}

		/* プラグインファイルの種類 */
		if (pm->fileinfo->ftype == TFSP_IFFTYPE_UNKNOWN) {
			return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"filetype\" (tfsplugin) was missing or empty value or "
								"unsupported value."));
		}

		/* チェックサムによるファイルのチェック */
		if (IS_FILLED(pm->fileinfo->checksum)) {
			/* ##### FIXME 実装を */
		}
	}

	if (pm->cpageinfo != NULL) {
		if (pm->cpageinfo->g_cpuri != NULL) {
			if (IS_EMPTY(pm->cpageinfo->g_cpuri->uri)) {
				return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"startupurl\" (tfsplugin/globalpage) was "
								"missing or empty value."));
			}
		}

		if (pm->cpageinfo->l_cpuri != NULL) {
			if (IS_EMPTY(pm->cpageinfo->l_cpuri->uri)) {
				return tfs_error_create(p, LOG_WARNING, TFSP_ENOVALUE, TFS_EDESC0(
								"The \"startupurl\" (tfsplugin/locationpage) "
								"was missing or empty value."));
			}
		}
	}

	return NULL;
}

/**
 * 指定されたプラグインpm をローディングする.
 *
 * @param p tfs_pool_t * 作業用のプール
 * @param pm tfsp_plugin_module * モジュール構造体へのポインタ
 * @return tfs_error_t * エラー構造体へのポインタ. NULLで正常
 */
static tfs_error_t * _load_plugin(tfs_pool_t *p, tfsp_plugin_module *pm)
{
	tfsp_plugin_fileinfo *fileinfo;

	if (pm == NULL) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}
	fileinfo = pm->fileinfo;

	/* 共有ライブラリ以外はダイナミックローディング可能ではない */
	if (fileinfo->ftype != TFSP_IFFTYPE_SHAREDOBJ) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOTLOADABLE, TFS_EDESC0(
								"The shared-library was not valid dynamic-loading library."));
	}

	/* ライブラリファイル(.so) は指定されていたか? */

	/* プラグインライブラリをダイナミックローディングする */
	PM_CTX(pm)->handle = dlopen(fileinfo->filepath, RTLD_NOW);
	if (PM_CTX(pm)->handle == NULL) {
		const char *msg = dlerror();

		return tfs_error_create(p, LOG_ERR, TFSP_ELOADPLUGIN, TFS_EDESC1(
								"Failed to dlopen shared-library.(Reason : %s)", msg));
	}

	return NULL;
}

/**
 * 指定されたプラグインモジュールpm をメモリ上からアンロードする.
 */
static tfs_error_t * _unload_plugin(tfs_pool_t *p, tfsp_plugin_module *pm)
{
	int ret;

	if (pm == NULL || PM_CTX(pm) == NULL || PM_CTX(pm)->handle == NULL) {
		return NULL;	/* 何もしない */
	}

	/* ダイナミックローディングライブラリをアンロードする */
	ret = dlclose(PM_CTX(pm)->handle);
	if (ret) {
		const char *msg = dlerror();

		return tfs_error_create(p, LOG_ERR, TFSP_EUNLOADPLUGIN, TFS_EDESC1(
								"Failed to dlclose shared-library.(Reason : %s)", msg));
	}
	PM_CTX(pm)->handle = NULL;	/* NULL クリアする */

	return NULL;
}

/**
 * ダイナミックローディングモジュールのハンドルhandle から
 * プラグインハンドラ構造体へのポインタを取得する.
 *
 * @param p tfs_pool_t *
 * @param handle void * ハンドルへのポインタ
 * @param plugin_h void ** 取得したハンドラ構造体へのポインタ
 * @return tfs_error_t * エラー構造体へのポインタ
 */
static tfs_error_t * _get_plugin_handler(tfs_pool_t *p, void *handle, void **plugin_h)
{
	*plugin_h = NULL;

	if (handle == NULL) {
		return tfs_error_create(p, LOG_ERR, TFSP_ENOPARAM, TFS_EDESC0(
								"The parameter was missing."));
	}

	*plugin_h = dlsym(handle, "TFSP_INTERFACE_HANDLER");
	if (*plugin_h == NULL) {
		const char *msg = dlerror();

		return tfs_error_create(p, LOG_ERR, TFSP_ELOADPLUGIN, TFS_EDESC1(
							"Failed to dlsym shared-library.(Reason : %s)", msg));
	}

	return NULL;
}

/**
 * プラグイン基本インターフェースを取得する.
 *
 * @param pm tfsp_plugin_module *
 * @return const tfsp_ihdr_base *
 */
static const tfsp_ihdr_base * _ge_plugin_baseif(tfsp_plugin_module *pm)
{
	tfsp_ihdr_abstract *iab;

	if (pm == NULL || PM_CTX(pm) == NULL) {
		return NULL;
	}
	iab = (tfsp_ihdr_abstract *) PM_CTX(pm)->interface;
	if (iab == NULL) {
		return NULL;
	}

	return iab->bs;
}

#undef TFSP_CORE_PRIVATE


