/**
 * $Id$
 *
 * 汎用ユーティリティ (エラー状態の保持機構)
 */
#include "tfs_errlog.h"
#include "tfs_errno.h"
#include "tfs_pools.h"
#include "tfs_queue.h"
#include "tfs_string.h"

#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#if HAVE_STDARG_H
#include <stdarg.h>
#endif
#if HAVE_TIME_H
#include <time.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * ctime_r 用バッファの長さ
 */
#define CTIME_LEN	26

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
typedef struct tfs_error_elem	tfs_error_elem;

/**
 * エラー状態を保持する構造体 [ 不完全型の定義 ]
 */
struct tfs_error_t {
	/* 複数のエラー情報を保持するQueueへのポインタ */
	tfs_queue_t *errs;

	/* メモリアロケーター */
	tfs_pool_t *pool;
	int allocated_pool;
};

/**
 * 1つのエラー状態を保持する構造体
 */
struct tfs_error_elem {
	int level;	/* ログレベル(LOG_EMERG, LOG_ALERT, ...) */
	tfs_status_t status;	/* エラーコード (see tfs_errno.h) */
	char *funcname;     /* 問題が発生した関数の名称 */
	int lineno;     /* 問題が発生した行番号     */
	const char *desc;   /* 詳細メッセージ */
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_error(void *data);
static const char * _errlevel2str(int lv);

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * エラーメッセージdesc を指定してエラー情報インスタンスを生成する.
 *
 */

TFS_DECLARE(tfs_error_t *) tfs_error_create(tfs_pool_t *pool, int level, int status,
                                            char *funcname, int lineno, const char *fmt, ...)
{
	tfs_pool_t *p = pool;
	tfs_error_t *err = NULL;
	tfs_error_elem *e;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	/* コンテキスト生成 */
	err = malloc(sizeof(tfs_error_t));
	tfs_queue_create(p, &err->errs);	/* キューを生成する */
	err->pool = p;
	err->allocated_pool = allocated_pool;

	/* 与えら得たエラー情報を記録する */
	e = tfs_pcalloc(p, sizeof(tfs_error_elem));
	e->level    = level;
	e->status   = status;
	e->funcname = funcname;
	e->lineno   = lineno;
	if (fmt != NULL) {
		char *buf;
		int n, size = 256;
		va_list args;

		buf = malloc(size + 1);	/* 1つ余分に取っておく(バグ防止) */
		memset(buf, '\0', size + 1);
		while (1) {
			va_start(args, fmt);
			n = vsnprintf(buf, size, fmt, args);
			va_end(args);

			/* buf に十分なサイズがあった場合 */
			if (n > -1 && n < size) {
				e->desc = tfs_pstrndup(pool, buf, n);
				free(buf);
				break;
			}
			/* buf に十分なサイズがなかった */
			else if (n > -1) {
				free(buf);
				size = n + 1;
				buf = malloc(size);
				continue;	/* リトライ */
			}
			else {
				e->desc = NULL;	/* 書き込みに失敗 */
				free(buf);
				break;
			}
		}
	}
	else {
		e->desc = fmt;
	}

	/* キューにエラー情報を格納する */
	tfs_queue_push(err->errs, e);

	/* err をクリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, err, _cleanup_error);

	return err;
}

/**
 * 指定されたエラー情報を記録し、それを指定されたエラーprev と関連付けて返す.
 *
 */
TFS_DECLARE(tfs_error_t *) tfs_error_push(tfs_error_t *prev, int level, int status,
                                          char *funcname, int lineno, const char *fmt, ...)
{
	tfs_error_elem *e;
	tfs_pool_t *p;

	if (prev == NULL) {
		return NULL;	/* 何もしない */
	}
	p = prev->pool;

	/* 与えら得たエラー情報を記録する */
	e = tfs_pcalloc(p, sizeof(tfs_error_elem));
	e->level    = level;
	e->status   = status;
	e->funcname = funcname;
	e->lineno   = lineno;
	if (fmt != NULL) {
		char *buf;
		int n, size = 256;
		va_list args;

		buf = malloc(size + 1);	/* 1つ余分に取っておく(バグ防止) */
		memset(buf, '\0', size + 1);
		while (1) {
			va_start(args, fmt);
			n = vsnprintf(buf, size, fmt, args);
			va_end(args);

			/* buf に十分なサイズがあった場合 */
			if (n > -1 && n < size) {
				e->desc = tfs_pstrndup(p, buf, n);
				free(buf);
				break;
			}
			/* buf に十分なサイズがなかった */
			else if (n > -1) {
				free(buf);
				size = n + 1;
				buf = malloc(size);
				continue;	/* リトライ */
			}
			else {
				e->desc = NULL;	/* 書き込みに失敗 */
				free(buf);
				break;
			}
		}
	}
	else {
		e->desc = fmt;
	}

	/* キューにエラー情報を格納する */
	tfs_queue_push(prev->errs, e);

	return prev;	/* prev を返す */
}

/**
 * 指定されたエラーprev に新しいエラーnew_err を連結して1つのエラー情報にする.
 *
 */
TFS_DECLARE(tfs_error_t *) tfs_error_append(tfs_pool_t *pool, tfs_error_t *prev,
                                            tfs_error_t *new_err)
{
	tfs_error_t *err = NULL;
	tfs_error_elem *e, *new_e;
	tfs_status_t rv;

	if (pool == NULL) return NULL;

	if (prev == NULL && new_err == NULL) {
		return NULL;	/* 何もできない */
	}
	else if (prev == NULL && new_err != NULL) {
		return tfs_error_clone(pool, new_err);
	}
	else if (prev != NULL && new_err == NULL) {
		return tfs_error_clone(pool, prev);
	}
	/* prev != NULL && new_err != NULLの場合 */

	/* prev をコピーする */
	err = tfs_error_clone(pool, prev);

	/* new_error からエラーエレメントを取り出してprev のキューにPushする */
	while ((rv = tfs_queue_pop(new_err->errs, (void **)&e)) != TFS_QUEUE_EMPTY) {
		if (e == NULL) continue;

		new_e = tfs_pcalloc(pool, sizeof(tfs_error_elem));
		new_e->level    = e->level;
		new_e->status   = e->status;
		new_e->funcname = e->funcname;
		new_e->lineno   = e->lineno;
		new_e->desc     = tfs_pstrdup(pool, e->desc);

		tfs_queue_push(err->errs, new_e);
	}

	return err;
}

/**
 * old が示すエラー情報をp のメモリアロケーターを使って複製する.
 *
 */
TFS_DECLARE(tfs_error_t *) tfs_error_clone(tfs_pool_t *pool, tfs_error_t *old)
{
	tfs_error_t *new_err;

	if (old == NULL || pool == NULL) {
		return NULL;	/* クローンできません */
	}

	/* コンテキスト生成 */
	new_err = malloc(sizeof(tfs_error_t));
	new_err->pool = pool;
	new_err->allocated_pool = 0;

	/* キューをクローンする */
	new_err->errs = tfs_queue_clone(pool, old->errs);

	/* new_err をクリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(pool, new_err, _cleanup_error);

	return new_err;
}

/**
 * err が保持する全てのエラー情報をエラーロガーlogger を使って全て出力する.
 * ここ関数がコールされるとerr が蓄えていた全てのエラー情報はクリアされます.
 *
 */
TFS_DECLARE(void) tfs_error_dumpall(tfs_error_t *err, void *data,
                                    tfs_error_logger_t logger)
{
	tfs_error_elem *e = NULL;
	tfs_status_t rv;
	tfs_pool_t *p = err->pool;
	char *line;

	if (err == NULL || logger == NULL || err->errs == NULL) {
		return;	/* 何もしない */
	}

	/* エラーQueueが空になるまでエラー情報を取り出す */
	while ((rv = tfs_queue_pop(err->errs, (void **)&e)) != TFS_QUEUE_EMPTY) {

		/* 出力文字列の組み立て */
		line = tfs_psprintf(p, "%s(%d): (%d) %s",
					e->funcname, e->lineno, e->status, e->desc);

		/* エラーログ出力 */
		(*logger)(data, e->level, line);
	}
}

/**
 * err が保持するエラーステータスを取得する.
 * 複数個のエラーがPushされている場合、一番先頭のエラーコードが返されます.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_error_getstatus(tfs_error_t *err)
{
	tfs_status_t rv;
	tfs_error_elem *e = NULL;

	if (err == NULL) {
		return TFS_SUCCESS;
	}

	/* 先頭の要素を消さずに取り出してみる */
	rv = tfs_queue_peek(err->errs, (void **)&e);
	if (rv == TFS_QUEUE_EMPTY || e == NULL) {
		return TFS_SUCCESS;
	}

	return e->status;
}

/**
 * stderr に出力するエラーロガーの実装(tfs_error_logger_t の実装)
 * エラーレベルとは無関係に出力します. data は使用しません.
 *
 */
TFS_DECLARE(void) tfs_error_dump_stderr(void *data, int level, const char *desc)
{
	if (IS_EMPTY(desc)) {
		return;
	}
	fprintf(stderr, "%s\n", desc);
	fflush(stderr);
}

/**
 * デバッグ用ロガー
 * (note)
 *   基本的にはデバッグ用、良くてインフォメーション表示用ロガーとして使って下さい.
 *
 */
TFS_DECLARE(void) tfs_log_notice(char *funcname, int lineno, const char *fmt, ...)
{
	const char *slevel;
	char sctime[CTIME_LEN] = { 0 };
	va_list args;

	/* エラーログレベルの取得 */
	slevel = _errlevel2str(LOG_NOTICE);
	if (IS_FILLED(slevel) && IS_FILLED(fmt)) {
		time_t t = time(NULL);
		int i;

		/* 現在の時刻を取得する */
		ctime_r(&t, sctime);
		for (i = CTIME_LEN - 1; i >= 0; i--) {
			if (sctime[i] == '\n') {
				sctime[i] = '\0';
			}
		}

		/* ログを出力する */
		va_start(args, fmt);
		/* (note) "- -" はこのロガーでは表示不可能な項目 */
		fprintf(stderr, "[%s] [%s] - - ", sctime, slevel);
		vfprintf(stderr, fmt, args);
		fprintf(stderr, "\n");
		fflush(stderr);
		va_end(args);
	}
}

/**
 * err が消費していた領域を破棄する.
 *
 */
TFS_DECLARE(void) tfs_error_destroy(tfs_error_t *err)
{
	if (err == NULL) return;

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(err->pool, err, _cleanup_error);

	_cleanup_error(err);
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static void _cleanup_error(void *data)
{
	tfs_error_t *err = data;

	if (err == NULL) return;

	if (err->errs != NULL) {
		tfs_queue_destroy(err->errs);
		err->errs = NULL;
	}

	if (err->pool != NULL && err->allocated_pool) {
		tfs_pool_destroy(err->pool);
		err->pool = NULL;
	}
	free(err);
}

/**
 * エラーログで使用するエラーレベル文字列を返却する
 *
 */
static const char * _errlevel2str(int lv)
{
	const char *str = NULL;

	switch (lv) {
		case LOG_EMERG:
			str = "emerg";
			break;
		case LOG_ALERT:
			str = "alert";
			break;
		case LOG_CRIT:
			str = "crit";
			break;
		case LOG_ERR:
			str = "error";
			break;
		case LOG_WARNING:
			str = "warn";
			break;
		case LOG_NOTICE:
			str = "notice";
			break;
		default:
			str = "unknown";
			break;
	}

	return str;
}


