/**
 * $Id$
 *
 * 汎用ユーティリティ (fileio : ファイル操作関連)
 *
 */
#include "tfs_fileio.h"
#include "tfs_string.h"

#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * copy バッファのサイズ(byte)
 */
#define TFS_CP_BUFSIZE	4096

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static tfs_status_t _readfile(int fd, char *buf, ssize_t *rlen);
static tfs_status_t _writefile(int fd, const char *buf, ssize_t *wlen);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * 指定されたファイルディスクリプタfd のファイルからサイズlen 分のデータを
 * buf に読み込む.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_file_read(int fd, char *buf, tfs_ssize_t *len)
{
	return _readfile(fd, buf, len);
}

/**
 * ファイルパスpath のファイルを削除する.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_file_remove(const char *path)
{
	if (IS_EMPTY(path)) {
		return TFS_ENOPARAM;
	}

	/* path の削除 */
	if (unlink(path) == 0) {
		return TFS_SUCCESS;
	}
	else {
		return TFS_OS_ERRNO(errno);
	}
}

/**
 * ファイルパス src が示すファイルをファイルパス dst のパスに移動する.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_file_move(const char *src, const char *dst)
{
	if (IS_EMPTY(src) || IS_EMPTY(dst)) {
		return TFS_ENOPARAM;
	}

	/* ファイルパスのリネーム */
	if (rename(src, dst) == 0) {
		return TFS_SUCCESS;
	}
	else {
		return TFS_OS_ERRNO(errno);
	}
}

/**
 * ファイルパス src が示すファイルの内容をdst が示すファイルパスにコピーする.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_file_copy(const char *src, const char *dst, mode_t mode)
{
	tfs_status_t rv = TFS_SUCCESS;
	int fd_src = -1, fd_dst = -1;
	char buf[TFS_CP_BUFSIZE];
	ssize_t len;

	if (IS_EMPTY(src) || IS_EMPTY(dst)) {
		return TFS_ENOPARAM;
	}

	/* src ファイルを開く */
	if ((fd_src = open(src, TFS_OFLG_READONLY)) == -1) {
		return TFS_OS_ERRNO(errno);
	}

	/* src のパーミッションとdst を同じにする場合 */
	if (mode == TFS_FLMODE_SYNC_SRC) {
		struct stat finfo = { 0 };
		if (fstat(fd_src, &finfo) == -1) {
			return TFS_OS_ERRNO(errno);
		}
		mode = finfo.st_mode;
	}

	/* dst ファイルを開く */
	if ((fd_dst = open(dst, TFS_OFLG_NEWFILE, mode)) == -1) {
		rv = TFS_OS_ERRNO(errno);
		goto cleanup_opened_file;
	}

	/* src からデータを読みながらdst に書き出す */
	while (1) {
		/* 読み込み */
		len = TFS_CP_BUFSIZE;
		rv = _readfile(fd_src, buf, &len);
		if (rv == TFS_FILE_EOF) {
			rv = TFS_SUCCESS;	/* 成功 */
			goto cleanup_opened_file;	/* ファイルの終わり */
		}
		else if (rv != TFS_SUCCESS) {
			goto cleanup_opened_file;
		}

		/* 書き出し */
		rv = _writefile(fd_dst, buf, &len);
		if (rv == TFS_FILE_NOCONTENT || rv == TFS_SUCCESS) {
			rv = TFS_SUCCESS;	/* 成功 */
			goto cleanup_opened_file;
		}
		else if (rv != TFS_SUCCESS) {
			goto cleanup_opened_file;
		}
	}

cleanup_opened_file:
	if (fd_src != -1) {
		close(fd_src);
	}
	if (fd_dst != -1) {
		close(fd_dst);
	}

	/* dst への書き出しに失敗していたので削除しておく */
	if (rv != TFS_SUCCESS) {
		(void) tfs_file_remove(dst);
	}

	return rv;
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * ファイルディスクリプタ fd が示すファイルから *rlen バイトのデータを
 * 読み込み、buf に詰める.
 * なお、*rlen よりも小さいサイズしか読み込めないこともあるので注意.
 * (残りファイルコンテントが*rlen バイトよりも小さい、読み込みに失敗など)
 *
 */
static tfs_status_t _readfile(int fd, char *buf, ssize_t *rlen)
{
	ssize_t len;

	if (fd == -1 || buf == NULL || *rlen == 0) {
		return TFS_EINVALID_PARAM;
	}

	len = read(fd, buf, *rlen);
	if (len == -1) {
		return TFS_OS_ERRNO(errno);
	}
	else if (len == 0) {
		return TFS_FILE_EOF;
	}

	*rlen = len;
	return TFS_SUCCESS;
}

/**
 * ファイルディスクリプタ fd が示すファイルに バッファbuf の*wlen バイト分の
 * データを書き出す.
 */
static tfs_status_t _writefile(int fd, const char *buf, ssize_t *wlen)
{
	ssize_t len, delta;
	int writedata = 0;

	if (fd == -1 || buf == NULL) {
		return TFS_EINVALID_PARAM;
	}

	if (*wlen == 0) {
		return TFS_FILE_NOCONTENT;
	}

	delta = *wlen;
	while (1) {
		len = write(fd, buf, delta);
		if (len == -1) {
			return TFS_OS_ERRNO(errno);
		}
		else if (len == 0) {
			if (!writedata) {
				return TFS_EFILE_WRITE;
			}
			break;
		}

		/* 全部書き終わったか? */
		delta = delta - len;
		if (delta > 0) {
			continue;	/* 全部書き込み終わらなかったのでリトライ */
		}
		else if (delta == 0) {
			break;	/* 書き込み完了 */
		}
		else {
			return TFS_EFILE_WRITE;	/* 有り得ないが一応 */
		}
	}

	return TFS_SUCCESS;
}


