/**
 * $Id$
 *
 * 汎用ユーティリティ (HashMap : ハッシュによるMap インターフェースの実装)
 */
#include "tfs_hashmap.h"
#include "tfs_string.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * ハッシュマップエントリの初期個数
 */
#define HMAP_INIT_MAX	15

/**
 * ハッシュマップエントリへのポインタを格納する配列の要素番号を算出数するマクロ
 */
#define ENTRY_INDEX(hash,max)	(hash) & (max)

/*------------------------------------------------------------------------------
  Define structures and enum
  ----------------------------------------------------------------------------*/
typedef struct tfs_hmap_entry	tfs_hmap_entry;

/**
 * １つのハッシュエントリを表す構造体
 * (note)
 *   next には同じハッシュ値を持つものがつながっていきます
 */
struct tfs_hmap_entry {
	const char *key;	/* キー */
	const void *value;	/* 値   */
	tfs_size_t klen;	/* キーの文字列長さ(パフォーマンス用) */
	tfs_uint32_t hash;	/* ハッシュ値 */

	tfs_hmap_entry *next;	/* 同じハッシュ値を持つ次のハッシュ要素へのポインタ */
};

/**
 * 関数_find_entry のmode値
 */
enum {
	HMAP_ENTRY_GET = 0,	/* 値の取得を行う */
	HMAP_ENTRY_PUT,		/* 値の設定を行う */
	HMAP_ENTRY_REMOVE,	/* 値登録の解除を行う */
};

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * ハッシュマップを表すコンテキスト構造体 [ 不完全型の定義 ]
 * [ 構造 ]
 *   * あるハッシュ値の(key, value) はarray[(ハッシュ値)] のtfs_hmap_entry * に保持.
 *   * 同じハッシュ値(値の衝突) があれば、tfs_hmap_entry のnext メンバにつなげる
 *   * array のサイズは常に(2の2乗)個
 *
 *   array[0] --> tfs_hmap_entry *
 *   array[1] --> tfs_hmap_entry *, tfs_hmap_entry *
 *   ...
 *   array[2^n-1] --> tfs_hmap_entry *
 */
struct tfs_hmap_t {

	/* ハッシュ要素の配列 */
	tfs_hmap_entry **array;

	/* arrayの要素数( = 最大登録可能な数) */
	tfs_uint32_t max;

	/* このハッシュマップに追加された要素の総数 */
	tfs_uint32_t size;

	/* メモリアロケータ */
	tfs_pool_t *pool;

	int allocated_pool;
};

/**
 * ハッシュマップ内の要素列挙に使用するイテレータ [ 不完全型の定義 ]
 */
struct tfs_hmap_index_t {
	tfs_hmap_t *hmap;

	/* 注目しているハッシュエントリ配列のインデックス番号 */
	tfs_uint32_t index;

	/* getvalue() で取得するハッシュエントリへのポインタ */
	tfs_hmap_entry *e;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_hmap(void *data);
static tfs_uint32_t _get_hash(const char *key);
static tfs_hmap_entry ** _alloc_array(tfs_uint32_t size);
static tfs_hmap_entry * _find_entry(tfs_hmap_t *hmap, const char *key,
                                    const void *value, int mode);
static void _expand_array(tfs_hmap_t *hmap);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * ハッシュマップの生成.
 *
 */
TFS_DECLARE(void) tfs_hmap_create(tfs_pool_t *pool, tfs_hmap_t **hmap)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	/* コンテキスト生成 (コンテキストはmallocで) */
	*hmap = malloc(sizeof(tfs_hmap_t));
	(*hmap)->max   = HMAP_INIT_MAX;
	(*hmap)->array = _alloc_array((*hmap)->max);
	(*hmap)->size  = 0;
	(*hmap)->pool  = p;
	(*hmap)->allocated_pool = allocated_pool;

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *hmap, _cleanup_hmap);
}

/**
 * キー文字列 key の値とvalue の関係をハッシュマップhmap にマップする.
 *
 */
TFS_DECLARE(void) tfs_hmap_put(tfs_hmap_t *hmap, const char *key, const void *value)
{
	tfs_hmap_entry *e = NULL;

	if (hmap == NULL || IS_EMPTY(key) || value == NULL) {
		return;
	}

	/* エントリを探す & 無ければ追加する */
	e = _find_entry(hmap, key, value, HMAP_ENTRY_PUT);
	if (e != NULL) {
		/* ハッシュ値の実登録数が最大登録数を上回っているか? */
		if (hmap->size > hmap->max) {
			/* ハッシュ要素配列を拡張する */
			_expand_array(hmap);
		}
	}
}

/**
 * キー文字列 key にマップされた値をハッシュマップhmap から取得する.
 *
 */
TFS_DECLARE(void *) tfs_hmap_get(tfs_hmap_t *hmap, const char *key)
{
	tfs_hmap_entry *e = NULL;

	if (hmap == NULL || IS_EMPTY(key)) {
		return NULL;
	}

	/* エントリを探す */
	if ((e = _find_entry(hmap, key, NULL, HMAP_ENTRY_GET)) != NULL) {
		return (void *)e->value;
	}

	return NULL;
}

/**
 * value がハッシュマップhmap にマップされているかどうか.
 *
 */
TFS_DECLARE(int) tfs_hmap_containsvalue(tfs_hmap_t *hmap, const void *value)
{
	tfs_hmap_entry *e = NULL;
	int i;

	if (hmap == NULL || value == NULL) {
		return 0;
	}

	/* ハッシュマップを虱潰しに探すしかない */
	for (i = 0; i < hmap->max; i++) {
		for (e = hmap->array[i]; e != NULL; e = e->next) {
			if (e->value == value) {
				return 1;	/* 値があった */
			}
		}
	}

	return 0;
}

/**
 * ハッシュマップhmap からキー文字列 key のマッピングを削除する.
 *
 */
TFS_DECLARE(void *) tfs_hmap_remove(tfs_hmap_t *hmap, const char *key)
{
	tfs_hmap_entry *e = NULL;

	if (hmap == NULL || IS_EMPTY(key)) {
		return NULL;
	}

	/* エントリを探す */
	if ((e = _find_entry(hmap, key, NULL, HMAP_ENTRY_REMOVE)) != NULL) {
		return (void *)e->value;
	}

	return NULL;
}

/**
 * ハッシュマップhmap にマップされたマッピングの数を返す.
 *
 */
TFS_DECLARE(tfs_int32_t) tfs_hmap_size(tfs_hmap_t *hmap)
{
	return (hmap != NULL) ? hmap->size : 0;
}

/**
 * ハッシュマップhmap のイテレータを生成して返却する.
 *
 */
TFS_DECLARE(tfs_hmap_index_t *) tfs_hmap_first(tfs_hmap_t *hmap)
{
	tfs_hmap_index_t *iterator;
	tfs_uint32_t i;
	tfs_hmap_entry *e;
	int found = 0;

	if (hmap == NULL || hmap->size == 0) return NULL;

	/* 値のある最初のハッシュエントリを探す */
	for (i = 0; i < hmap->max; i++) {
		for (e = hmap->array[i]; e != NULL; e = e->next) {
			if (e->key != NULL) {	/* 見つかった */
				found = 1;
				break;
			}
		}
		if (found) break;
	}

	/* 何故かひとつも見つからなかった */
	if (!found) {
		return NULL;
	}

	/* イテレーターを生成する */
	iterator = tfs_pcalloc(hmap->pool, sizeof(tfs_hmap_index_t));
	iterator->hmap  = hmap;	/* iterator 生成元を記録する */
	iterator->index = i;
	iterator->e     = e;

	return iterator;
}

/**
 * イテレータiterator が次の要素を指すように移動する.
 *
 */
TFS_DECLARE(tfs_hmap_index_t *) tfs_hmap_next(tfs_hmap_index_t *iterator)
{
	tfs_hmap_entry *e;

	if (iterator == NULL || iterator->hmap == NULL || iterator->e == NULL ||
		iterator->index > iterator->hmap->max) {
		return NULL;
	}

	e = iterator->e;
	/* 次のリストがあった場合 */
	if (e->next != NULL) {
		iterator->e = e->next;	/* 次のリストをセットする */
	}
	/* 次のリストが無かったらハッシュエントリ配列から探す */
	else {
		tfs_hmap_t *hmap = iterator->hmap;

		if (++iterator->index > hmap->max) {
			return NULL;	/* 最後の配列要素まで既に探していた場合 */
		}
		else {
			tfs_uint32_t i;
			int found = 0;
			for (i = iterator->index; i <= hmap->max; i++) {
				for (e = hmap->array[i]; e != NULL; e = e->next) {
					if (e->key != NULL) {	/* 見つけた */
						iterator->index = i;
						iterator->e     = e;
						found = 1;
						break;
					}
				}
				if (found) break;
			}
			if (!found) {
				return NULL;	/* 無かった */
			}
		}
	}

	return iterator;	/* 再利用 */
}

/**
 * イテレータiterator からハッシュマップにマップされたキーと値の組み合わせを
 * 取得する.
 *
 */
TFS_DECLARE(void) tfs_hmap_getvalue(tfs_hmap_index_t *iterator,
                                    const char **key, const void **value)
{
	tfs_hmap_entry *e;

	/* 初期化 */
	*key = NULL;
	*value = NULL;

	if (iterator == NULL || iterator->hmap == NULL || iterator->e == NULL) {
		return;
	}
	e = iterator->e;

	*key = e->key;
	*value = e->value;
}

/**
 * ハッシュマップhmap の要素をクリアする.(マップの解除)
 *
 */
TFS_DECLARE(void) tfs_hmap_clear(tfs_hmap_t *hmap)
{
	int i;
	if (hmap == NULL) return;

	/* (note) hmap->max はクリアしません */
	hmap->size = 0;
	for (i = 0; i < hmap->max; i++) {
		hmap->array[i] = NULL;;
	}
}

/**
 * hmap が表すハッシュマップの破棄.
 *
 */
TFS_DECLARE(void) tfs_hmap_destroy(tfs_hmap_t *hmap)
{
	if (hmap == NULL) return;

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(hmap->pool, hmap, _cleanup_hmap);

	_cleanup_hmap(hmap);
}


/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static void _cleanup_hmap(void *data)
{
	tfs_hmap_t *hmap = data;
	if (hmap == NULL) return;

	tfs_hmap_clear(hmap);	/* クリア */

	/* ハッシュエントリ配列の破棄 */
	free(hmap->array);
	hmap->array = NULL;
	hmap->max   = 0;

	if (hmap->pool != NULL && hmap->allocated_pool) {
		tfs_pool_destroy(hmap->pool);
		hmap->pool = NULL;
	}
	free(hmap);
}

/**
 * 長さkleyの文字列 key のハッシュコードを計算する.
 *
 * @param key const char * 文字列
 * @return tfs_uint32_t ハッシュコード
 *   NULL の場合には0 を返却
 */
static tfs_uint32_t _get_hash(const char *key)
{
	const unsigned char *c;
	tfs_uint32_t hash = 0;

	for (c = key; *c != '\0'; c++) {
		hash = hash * 33 + (tfs_uint32_t) *c;
	}

	return hash;
}

/**
 * tfs_hmap_entry * をsize 個保持できる配列を生成する.
 * (note)
 *   オーバーランを防ぐため、ひとつ余計にアロケートしています。
 *   また、array は頻繁に拡張されるのでプールを使わずにmalloc で
 *   メモリ割り当てを実施しています.
 *
 * @param size tfs_uint32_t 要素数
 * @return tfs_hmap_entry ** 生成した配列へのポインタ
 */
static tfs_hmap_entry ** _alloc_array(tfs_uint32_t size)
{
	tfs_hmap_entry **array = NULL;
	tfs_uint32_t alen = size + 1;

	if (size == 0) {
		return NULL;
	}

	array = malloc(sizeof(tfs_hmap_entry *) * alen);
	memset(array, 0, sizeof(tfs_hmap_entry *) * alen);

	return array;
}

/**
 * キー値key の値を持つハッシュマップエントリへのポインタを検索して取得する.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 * @param key const char * キー文字列(NOT NULL).
 * @param value const void * 値へのポインタ
 * @param mode int 検索モード (HMAP_ENTRY_GET / HMAP_ENTRY_PUT / HMAP_ENTRY_REMOVE)
 * @return tfs_hmap_entry * 探してきたハッシュマップエントリへのポインタ
 */
static tfs_hmap_entry * _find_entry(tfs_hmap_t *hmap, const char *key,
                                    const void *value, int mode)
{
	tfs_hmap_entry *e = NULL, *first = NULL, *prev = NULL;
	tfs_uint32_t hash;
	tfs_size_t klen;
	tfs_pool_t *p = hmap->pool;
	int index;

	if (hmap == NULL || IS_EMPTY(key) || hmap->array == NULL) {
		return NULL;
	}

	/* ハッシュ値を計算する */
	hash = _get_hash(key);

	/* キーの長さを算出 */
	klen = strlen(key);

	/* ハッシュマップエントリのリストを探す */
	index = ENTRY_INDEX(hash, hmap->max);
	for (first = e = hmap->array[index]; e != NULL; e = e->next) {
		/* ハッシュ値の比較 -> キーの長さ比較 -> 文字列比較 という順番で実施.
		 * これで、無用な文字列比較をしないようにしている */
		if (e->hash == hash && e->klen == klen && memcmp(e->key, key, klen) == 0) {
			break;
		}
		prev = e;
	}

	/* 値の取得モードだった場合 */
	if (mode == HMAP_ENTRY_GET) {
		if (e == NULL) {
			return NULL;
		}
	}
	/* 値の設定モードだった場合 */
	else if (mode == HMAP_ENTRY_PUT) {
		if (e != NULL) {
			/* 値を上書きする */
			e->value = value;
		}
		else {
			tfs_hmap_entry *last;

			/* 新しいエントリの生成 */
			e = tfs_pcalloc(p, sizeof(tfs_hmap_entry));
			e->key   = tfs_pstrdup(p, key);	/* キーは複製します */
			e->value = value;
			e->klen  = klen;
			e->hash  = hash;
			e->next  = NULL;

			/* ハッシュマップエントリ配列に記録されていたの? */
			if (first != NULL) {
				for (last = first; last->next != NULL; last = last->next);
				last->next = e;
			}
			else {
				hmap->array[index] = e;
			}
			hmap->size++;	/* 総要素数をカウントアップ */
		}
	}
	/* 値の削除モードだった場合 */
	else if (mode == HMAP_ENTRY_REMOVE) {
		if (e == NULL) {
			return NULL;
		}

		/* 値の登録を解除する */
		if (prev == NULL) {
			hmap->array[index] = NULL;
		}
		else {
			prev->next = e->next;
		}
		hmap->size--;
	}
	else {
		return NULL;	/* 何もできない */
	}

	return e;
}

/**
 * ハッシュマップhmap のハッシュ要素格納用配列を拡張する.
 */
static void _expand_array(tfs_hmap_t *hmap)
{
	tfs_uint32_t new_max;
	tfs_hmap_entry **new_array, *e, *prev, *last;;
	int i, index;

	if (hmap == NULL) return;

	new_max = hmap->max * 2 + 1;	/* 2のn乗 - 1 */
	new_array = _alloc_array(new_max);

	/* もう一度ハッシュ値を計算し直して登録する */
	for (i = 0; i < hmap->max; i++) {
		e = hmap->array[i];
		while (e != NULL) {
			index = ENTRY_INDEX(e->hash, new_max);	/* 再計算 */
			if (new_array[index] == NULL) {
				new_array[index] = e;
			}
			else {
				for (last = new_array[index]; last->next != NULL; last = last->next);
				last->next = e;
			}
			prev = e;	/* 一時退避 */
			e = e->next;	/* 次へ進める */
			prev->next = NULL;	/* new_array に登録された要素のnext へのリンクを切る */
		}
	}

	/* 古い配列を破棄して、新しい配列に入れ換える */
	free(hmap->array);

	hmap->max   = new_max;
	hmap->array = new_array;
}

