/**
 * $Id$
 *
 * 汎用ユーティリティ (HashSet : ハッシュによるSet インターフェースの実装)
 */
#include "tfs_hashset.h"
#include "tfs_hashmap.h"
#include "tfs_string.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * ハッシュマップの値に詰めるダミーメンバ
 */
#define HSET_DUMMY_VAL	"x"

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * ハッシュセットを表すコンテキスト構造体 [ 不完全型の定義 ]
 */
struct tfs_hset_t {

	/* 文字列集合を保持するハッシュマップへのポインタ */
	tfs_hmap_t *hmap;

	/* メモリアロケータ */
	tfs_pool_t *pool;
	int allocated_pool;

};

/**
 * ハッシュセット内の要素列挙に使用するイテレータ [ 不完全型の定義 ]
 */
struct tfs_hset_index_t {
	tfs_hset_t *hset;

	tfs_hmap_index_t *hmap_iterator;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_hset(void *data);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * ハッシュセットの生成
 *
 */
TFS_DECLARE(void) tfs_hset_create(tfs_pool_t *pool, tfs_hset_t **hset)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	/* コンテキスト生成 (コンテキストはmallocで) */
	*hset = malloc(sizeof(tfs_hset_t));
	tfs_hmap_create(p, &(*hset)->hmap);	/* ハッシュマップを生成 */
	(*hset)->pool = p;
	(*hset)->allocated_pool = allocated_pool;

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *hset, _cleanup_hset);
}

/**
 * 要素element がハッシュセットhset になかった場合、セットに追加する.
 *
 */
TFS_DECLARE(int) tfs_hset_add(tfs_hset_t *hset, const char *element)
{
	char *str;

	if (hset == NULL || element == NULL) {
		return 1;
	}

	/* 探してみる */
	str = tfs_hmap_get(hset->hmap, element);
	if (IS_EMPTY(str)) {
		/* ハッシュマップに追加する */
		tfs_hmap_put(hset->hmap, element, HSET_DUMMY_VAL);
		return 1;
	}

	return 0;	/* 持っていなかった */
}

/**
 * 文字列 element がハッシュセットhset に含まれているかどうか.
 *
 */
TFS_DECLARE(int) tfs_hset_contains(tfs_hset_t *hset, const char *element)
{
	char *str;

	if (hset == NULL || element == NULL) {
		return 0;
	}

	str = tfs_hmap_get(hset->hmap, element);
	if (IS_EMPTY(str)) {
		return 0;
	}
	else {
		return 1;
	}
}

/**
 * 指定された文字列element をハッシュセットhset から除去する.
 *
 */
TFS_DECLARE(int) tfs_hset_remove(tfs_hset_t *hset, const char *element)
{
	char *str;

	if (hset == NULL || element == NULL) {
		return 1;
	}

	/* ハッシュマップエントリから削除する */
	str = tfs_hmap_remove(hset->hmap, element);
	if (IS_EMPTY(str)) {
		return 0;	/* 持っていなかった */
	}
	else {
		return 1;	/* 持っていた */
	}

}

/**
 * ハッシュセットhset が保持する全要素数を返す.
 *
 */
TFS_DECLARE(tfs_int32_t) tfs_hset_size(tfs_hset_t *hset)
{
	return (hset != NULL) ? tfs_hmap_size(hset->hmap) : 0;
}

/**
 * ハッシュセットhset のイテレータを生成して返却する.
 *
 */
TFS_DECLARE(tfs_hset_index_t *) tfs_hset_first(tfs_hset_t *hset)
{
	tfs_hset_index_t *iterator;

	if (hset == NULL) return NULL;

	iterator = tfs_pcalloc(hset->pool, sizeof(tfs_hset_index_t));
	iterator->hset = hset;
	iterator->hmap_iterator = tfs_hmap_first(hset->hmap);

	return iterator;
}

/**
 * イテレータiterator が次の要素を指すように移動する.
 *
 */
TFS_DECLARE(tfs_hset_index_t *) tfs_hset_next(tfs_hset_index_t *iterator)
{
	if (iterator == NULL || iterator->hset == NULL || iterator->hmap_iterator == NULL) {
		return NULL;
	}

	iterator->hmap_iterator = tfs_hmap_next(iterator->hmap_iterator);
	return iterator;	/* 再利用 */
}

/**
 * イテレータiterator からハッシュセットにマップされた文字列を取得する.
 *
 */
TFS_DECLARE(void) tfs_hset_getvalue(tfs_hset_index_t *iterator, const char **element)
{
	const void *dummy = NULL;

	*element = NULL;	/* 初期化 */

	if (iterator == NULL || iterator->hmap_iterator == NULL) {
		return;
	}
	tfs_hmap_getvalue(iterator->hmap_iterator, element, &dummy);
}

/**
 * ハッシュセットhset の要素をクリアする.
 *
 */
TFS_DECLARE(void) tfs_hset_clear(tfs_hset_t *hset)
{
	if (hset == NULL || hset->hmap == NULL) {
		return;
	}
	tfs_hmap_clear(hset->hmap);
}

/**
 * hset が示すハッシュセットの破棄.
 *
 */
TFS_DECLARE(void) tfs_hset_destroy(tfs_hset_t *hset)
{
	if (hset == NULL) return;

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(hset->pool, hset, _cleanup_hset);

	_cleanup_hset(hset);
}


/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static void _cleanup_hset(void *data)
{
	tfs_hset_t *hset = data;
	if (hset == NULL) return;

	tfs_hset_clear(hset);
	tfs_hmap_destroy(hset->hmap);	/* ハッシュマップを破棄 */

	if (hset->pool != NULL && hset->allocated_pool) {
		tfs_pool_destroy(hset->pool);
		hset->pool = NULL;
	}
	free(hset);
}

