/**
 * $Id$
 *
 * 汎用ユーティリティ (LinkedList : 可変長リスト)
 */
#include "tfs_linkedlist.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure
  ----------------------------------------------------------------------------*/
typedef struct tfs_linkedlist_elem	tfs_linkedlist_elem;
/**
 * ひとつのリスト要素を示す構造体
 */
struct tfs_linkedlist_elem {
	void *element;

	tfs_linkedlist_elem *next;
};


/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * 可変長リストを表すコンテキスト構造体 [ 不完全型の定義 ]
 */
struct tfs_linkedlist_t {
	/* 要素のリスト */
	tfs_linkedlist_elem *e;

	/* 要素の総数 */
	tfs_int32_t size;

	/* メモリアロケーター */
	tfs_pool_t *pool;
	int allocated_pool;
};

/**
 * 可変長リストのイテレータを表す構造体 [ 不完全型の定義 ]
 */
struct tfs_linkedlist_index_t {
	tfs_linkedlist_t *llist;
	tfs_int32_t position;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_linkedlist(void *data);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * 可変長リストを生成する.
 *
 */
TFS_DECLARE(void) tfs_linkedlist_create(tfs_pool_t *pool, tfs_linkedlist_t **llist)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	/* コンテキスト生成 (コンテキストはmallocで) */
	*llist = malloc(sizeof(tfs_linkedlist_t));
	(*llist)->e    = NULL;
	(*llist)->size = 0;	/* 要素数はなし */
	(*llist)->pool = p;
	(*llist)->allocated_pool = allocated_pool;

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *llist, _cleanup_linkedlist);
}

/**
 * 指定された要素element を可変長リストllist の末尾に追加する.
 *
 */
TFS_DECLARE(void) tfs_linkedlist_add(tfs_linkedlist_t *llist, void *element)
{
	tfs_linkedlist_elem *e;

	if (llist == NULL || element == NULL) {
		return;
	}

	/* 新しいエレメントの挿入 */
	if (llist->e == NULL) {
		llist->e = e = tfs_pcalloc(llist->pool, sizeof(tfs_linkedlist_elem));
	}
	else {
		for (e = llist->e; e->next; e = e->next);
		e->next = tfs_pcalloc(llist->pool, sizeof(tfs_linkedlist_elem));
		e = e->next;
	}

	llist->size++;
	e->element = element;
	e->next = NULL;
}

/**
 * 可変長リストllist の中の位置index にある要素をelement で置き換える.
 * index が範囲外であれば何もしません.
 *
 */
TFS_DECLARE(void) tfs_linkedlist_set(tfs_linkedlist_t *llist,
                                     tfs_int32_t index, void *element)
{
	tfs_linkedlist_elem *e;
	tfs_int32_t i;

	if (llist == NULL || element == NULL ||
		index >= llist->size || llist->e == NULL) {
		return;
	}

	/* index が 0 の時は特別扱い */
	if (index == 0) {
		llist->e->element = element;
	}
	else {
		for (i = 0, e = llist->e; i < index && e; i++, e = e->next);
		if (e != NULL) {
			e->element = element;
		}
	}
}

/**
 * 可変長リストllist の位置index にあるオブジェクトを削除し、それを返す.
 *
 */
TFS_DECLARE(void *) tfs_linkedlist_remove(tfs_linkedlist_t *llist, tfs_int32_t index)
{
	tfs_linkedlist_elem *e, *prev;
	tfs_int32_t i;
	void *element = NULL;

	if (llist == NULL || index >= llist->size || llist->e == NULL) {
		return NULL;
	}

	/* index が 0 の時は特別扱い */
	if (index == 0) {
		element = llist->e->element;
		llist->e = llist->e->next;
		llist->size--;
	}
	else {
		e = llist->e;
		for (i = 0; i < index; i++) {
			if (e == NULL) break;

			prev = e;
			e = e->next;
		}

		if (e != NULL) {
			prev->next = e->next;
			element = e->element;
			llist->size--;
		}
	}

	return element;
}

/**
 * llist が示す可変長リストをpool のメモリ領域にコピーする.
 *
 */
TFS_DECLARE(tfs_linkedlist_t *) tfs_linkedlist_clone(tfs_pool_t *pool, tfs_linkedlist_t *llist)
{
	tfs_linkedlist_t *new_llist = NULL;
	tfs_linkedlist_elem *e, *new_e = NULL;

	if (llist == NULL || pool == NULL) {
		return NULL;
	}

	new_llist = malloc(sizeof(tfs_linkedlist_t));
	new_llist->e = NULL;
	new_llist->size = llist->size;
	new_llist->pool = pool;
	new_llist->allocated_pool = 0;

	/* 可変長リストの要素をコピーする(シャローコピー) */
	for (e = llist->e; e; e = e->next) {
		if (new_llist->e == NULL) {
			new_llist->e = new_e = tfs_pcalloc(pool, sizeof(tfs_linkedlist_elem));
		}
		else {
			new_e->next = tfs_pcalloc(pool, sizeof(tfs_linkedlist_elem));
			new_e = new_e->next;
		}
		new_e->element = e->element;	/* シャローコピー */
		new_e->next = NULL;
	}

	/* new_llist もクリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(pool, new_llist, _cleanup_linkedlist);

	return new_llist;
}

/**
 * 可変長リストllist の総要素数を取得する.
 *
 */
TFS_DECLARE(tfs_int32_t) tfs_linkedlist_size(tfs_linkedlist_t *llist)
{
	return (llist != NULL) ? llist->size : 0;
}

/**
 * 可変長リストllist の位置index にあるオブジェクトを取得する.
 *
 */
TFS_DECLARE(void *) tfs_linkedlist_get(tfs_linkedlist_t *llist, tfs_int32_t index)
{
	tfs_linkedlist_elem *e;
	tfs_int32_t i;

	if (llist == NULL || index >= llist->size || llist->e == NULL) {
		return NULL;
	}

	/* index が 0 の時は特別扱い */
	if (index == 0) {
		e = llist->e;
	}
	else {
		for (i = 0, e = llist->e; i < index && e; i++, e = e->next);
	}
	return (e != NULL) ? e->element : NULL;
}

/**
 * 可変長リストllist のイテレータを生成して返却する.
 *
 */
TFS_DECLARE(tfs_linkedlist_index_t *) tfs_linkedlist_first(tfs_linkedlist_t *llist)
{
	tfs_linkedlist_index_t *iterator;

	if (llist == NULL || llist->e == NULL) {
		return NULL;
	}

	iterator = tfs_pcalloc(llist->pool, sizeof(tfs_linkedlist_index_t));
	iterator->llist    = llist;
	iterator->position = 0;

	return iterator;
}

/**
 * イテレータiterator が次の要素を指すように移動する.
 *
 */
TFS_DECLARE(tfs_linkedlist_index_t *) tfs_linkedlist_next(tfs_linkedlist_index_t *iterator)
{
	if (iterator == NULL || iterator->llist == NULL) {
		return NULL;
	}
	iterator->position++;
	if (iterator->position >= iterator->llist->size) {
		return NULL;
	}

	return iterator;	/* 実際にはiterator のオブジェクトは作り直さない */
}

/**
 * イテレータの値を取得する.
 *
 */
TFS_DECLARE(void *) tfs_linkedlist_getvalue(tfs_linkedlist_index_t *iterator)
{
	if (iterator == NULL || iterator->llist == NULL) {
		return NULL;
	}

	return tfs_linkedlist_get(iterator->llist, iterator->position);
}

/**
 * 可変長リストllist の要素を全てクリアする.
 *
 */
TFS_DECLARE(void) tfs_linkedlist_clear(tfs_linkedlist_t *llist)
{
	if (llist == NULL || llist->e == NULL) {
		return;
	}

	llist->e    = NULL;
	llist->size = 0;
}

/**
 * 可変長リストllist がアロケートしていた領域を破棄する.
 *
 */
TFS_DECLARE(void) tfs_linkedlist_destroy(tfs_linkedlist_t *llist)
{
	if (llist == NULL) return;

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(llist->pool, llist, _cleanup_linkedlist);

	_cleanup_linkedlist(llist);
}


/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static void _cleanup_linkedlist(void *data)
{
	tfs_linkedlist_t *llist = data;
	if (llist == NULL) return;

	/* 要素をクリア */
	tfs_linkedlist_clear(llist);

	if (llist->pool != NULL && llist->allocated_pool) {
		tfs_pool_destroy(llist->pool);
		llist->pool = NULL;
	}
	free(llist);
}

