/**
 * $Id$
 *
 * 汎用ユーティリティ (Allocated Memory Cache : アロケート済みメモリ領域のキャッシュ)
 *
 */
#include "tfs_mcache.h"
#include "tfs_varray.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
#define MCACHE_DEFAULT_LEN	10

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * ロケート済みメモリ領域保持する構造体の定義 [ 不完全型の定義 ]
 */
struct tfs_mcache_t {
	tfs_varray_t *varray;	/* ポインタは可変長配列で保持します */

	/* メモリアロケーター */
	tfs_pool_t *pool;
	int allocated_pool;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_mcache(void *data);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * アロケート済みメモリキャッシュの生成.
 *
 */
TFS_DECLARE(void) tfs_mcache_create(tfs_pool_t *pool,
                                    tfs_int32_t size, tfs_mcache_t **mcache)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	*mcache = malloc(sizeof(tfs_mcache_t));
	(*mcache)->pool = p;
	(*mcache)->allocated_pool = allocated_pool;

	if (size == -1) {
		size = MCACHE_DEFAULT_LEN;
	}

	/* 可変長配列の生成 */
	tfs_varray_create(p, size, &(*mcache)->varray);

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *mcache, _cleanup_mcache);
}

/**
 * アロケートしたオブジェクトへのポインタをmcache に登録する.
 *
 */
TFS_DECLARE(void) tfs_mcache_add(tfs_mcache_t *mcache, void *obj)
{
	if (mcache == NULL || mcache->varray == NULL) {
		return;
	}
	tfs_varray_add(mcache->varray, obj);
}

/**
 * アロケートしたオブジェクトへのポインタobj と一致するポインタを
 * mcache から消去する.
 *
 */
TFS_DECLARE(void) tfs_mcache_remove(tfs_mcache_t *mcache, void *obj)
{
	void *elem;
	tfs_size_t len, i;

	if (mcache == NULL || mcache->varray == NULL || obj == NULL) {
		return;	/* 何もしない */
	}

	/* obj のポインタを探す */
	len = tfs_varray_size(mcache->varray);
	for (i = 0; i < len; i++) {
		elem = tfs_varray_get(mcache->varray, i);
		if (elem == obj) {
			free(elem);
			tfs_varray_set(mcache->varray, i, NULL);	/* NULL を設定 */
		}
	}
}

/**
 * アロケート済みメモリキャッシュに登録されているオブジェクトの数を返す.
 *
 */
TFS_DECLARE(tfs_int32_t) tfs_mcache_size(tfs_mcache_t *mcache)
{
	if (mcache == NULL || mcache->varray == NULL) return 0;

	return tfs_varray_size(mcache->varray);
}

/**
 * アロケート済みメモリキャッシュに登録されたオブジェクトの
 * ヒープ領域を全て開放(free)する.
 *
 */
TFS_DECLARE(void) tfs_mcache_cleanup(tfs_mcache_t *mcache)
{
	void *elem;
	tfs_size_t len, i;

	if (mcache == NULL || mcache->varray == NULL) {
		return;	/* 何もしない */
	}

	/* 全要素をfree する */
	len = tfs_varray_size(mcache->varray);
	for (i = 0; i < len; i++) {
		elem = tfs_varray_get(mcache->varray, i);
		if (elem != NULL) {
			free(elem);
			tfs_varray_set(mcache->varray, i, NULL);	/* NULL を設定 */
		}
	}
}

/**
 * アロケート済みメモリキャッシュを破棄する.
 * クリーンアップされていないオブジェクトがあれば、領域の開放を行ってから
 * mcache を破棄します.
 *
 */
TFS_DECLARE(void) tfs_mcache_destroy(tfs_mcache_t *mcache)
{
	if (mcache == NULL) return;

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(mcache->pool, mcache, _cleanup_mcache);

	_cleanup_mcache(mcache);
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static void _cleanup_mcache(void *data)
{
	tfs_mcache_t *mcache = data;
	if (mcache == NULL) return;

	/* キャッシュオブジェクトのクリア */
	tfs_mcache_cleanup(mcache);

	/* 可変長配列の開放 */
	if (mcache->varray != NULL) {
		tfs_varray_destroy(mcache->varray);
		mcache->varray = NULL;
	}

	if (mcache->pool != NULL && mcache->allocated_pool) {
		tfs_pool_destroy(mcache->pool);
		mcache->pool = NULL;
	}
	free(mcache);
}

