/**
 * $Id$
 *
 * 汎用ユーティリティ (Memory Allocator : 汎用ユーティリティ用メモリアロケーター)
 * (note)
 *   ここで定義されている関数、構造体は他のユーティリティ関数群に依存してはなりません.
 *   利用できるのは、tfs.h、tfs_errno.h のみです.
 */
#include "tfs_pools.h"

#if 0
#include <stdio.h>
#endif

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#if HAVE_PTHREAD_H
#include <pthread.h>
#endif	/* HAVE_PTHREAD_H */

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define incomplete type structure and normal structures
  ----------------------------------------------------------------------------*/
typedef struct tfs_pool_elem	tfs_pool_elem;
typedef struct tfs_pool_cleanup tfs_pool_cleanup;

/**
 * メモリアロケーターを表す構造体 [ 不完全型の定義 ]
 */
struct tfs_pool_t {
	/*
	 * 割り当てられたメモリに関する情報を保持する変数の定義
	 */
	tfs_pool_elem *first;
	tfs_pool_elem *last;
	tfs_int32_t total;	/* 統計情報 & 無限ループ防止の番兵 */

	/* クリーンアップハンドラ関数を保持するリスト */
	tfs_pool_cleanup *cleanup;

	/* 処理の同期化を行うためのmutex */
	pthread_mutex_t *mutex;

	/* このプールがアクティブであることを示す */
	volatile int active;
};

/**
 * あるひとつのmalloc されたメモリ領域へのポインタを管理する構造体
 */
struct tfs_pool_elem {
	void *mem;

	tfs_pool_elem *next;	/* 次の要素へのポインタ */
};

/**
 * クリーンアップハンドラを表す構造体
 */
struct tfs_pool_cleanup {
	/* クリーンアップハンドラ関数へのポインタ */
	tfs_pool_cleanup_t cleanup_fn;

	/* cleanup_fn に渡す引数 */
	const void *data;

	/*
	 * ひとつ古いクリーンアップハンドラ構造体へのポインタ
	 * (note) クリーンアップハンドラはFILO でコールされるため、登録順番とは
	 * 逆順で記録していきます.
	 */
	tfs_pool_cleanup *next;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static int _create_pthread_mutex(pthread_mutex_t **mutex);
static void _destroy_pthread_mutex(pthread_mutex_t *mutex);
static void _cleanup_pthread_mutex(void *mutex);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * メモリアロケーターの生成.
 *
 */
TFS_DECLARE(void) tfs_pool_create(tfs_pool_t **pool)
{
	int ret;
	pthread_mutex_t *mutex = NULL;

	*pool = NULL;	/* 初期化 */

	/* mutex の生成 */
	if ((ret = _create_pthread_mutex(&mutex)) != 0) {
		return;
	}

	*pool = malloc(sizeof(tfs_pool_t));
	(*pool)->first  = NULL;	/* 初期化 */
	(*pool)->last   = NULL;
	(*pool)->total  = 0;
	(*pool)->cleanup= NULL;
	(*pool)->mutex  = mutex;
	(*pool)->active = 1;
}

/**
 * メモリアロケーターによって割り当てられていたポインタを全て
 * リセットして再利用可能な状態にする.
 *
 */
TFS_DECLARE(void) tfs_pool_clear(tfs_pool_t *pool)
{
	tfs_pool_release(pool);	/* 今はtfs_pool_release() と同じ */
}

/**
 * メモリアロケーターによって割り当てられたメモリ領域を開放してOSに返す.
 *
 */
TFS_DECLARE(void) tfs_pool_release(tfs_pool_t *pool)
{
	tfs_int32_t i;
	tfs_pool_elem *elem, *prev;

	if (pool == NULL || pool->total == 0) {
		return;	/* 何もしなくていい */
	}

	/* クリーンアップハンドラのコール */
	if (pool->cleanup != NULL) {
		tfs_pool_cleanup *cl = pool->cleanup, *old;

		while (cl != NULL) {
			old = cl;
			cl = cl->next;

			if (old->cleanup_fn != NULL) {
				(*old->cleanup_fn)((void *)old->data);
			}
			free(old);
		}
		pool->cleanup = NULL;
	}

	/* mutex ロック */
	pthread_cleanup_push(_cleanup_pthread_mutex, (void *)pool->mutex);
	(void) pthread_mutex_lock(pool->mutex);

	/* 管理するメモリ領域及び管理領域をfree する */
	elem = pool->first;
	for (i = 0; i < pool->total; i++) {
		if (elem == NULL) break;

		prev = elem;
		elem = elem->next;

		if (prev->mem != NULL) {
			free(prev->mem);
			prev->mem = NULL;
		}
		free(prev);
	}
	pool->first = pool->last = NULL;
	pool->total = 0;
		
	/* mutex アンロック */
	(void) pthread_mutex_unlock(pool->mutex);
	pthread_cleanup_pop(0);
}

/**
 * メモリアロケーターによって割り当てられたメモリの開放と自身の破棄.
 *
 */
TFS_DECLARE(void) tfs_pool_destroy(tfs_pool_t *pool)
{
	if (pool == NULL) return;

	pool->active = 0;	/* アクティブでないことを示す */

	/* プールが持っていたメモリ領域を破棄する */
	tfs_pool_release(pool);

	/* mutex の破棄 */
	_destroy_pthread_mutex(pool->mutex);
	pool->mutex = NULL;

	/* プール自身を破棄 */
	free(pool);
}

/**
 * メモリアロケーターpool からサイズsize バイトのメモリ領域を取得する.
 *
 */
TFS_DECLARE(void *) tfs_palloc(tfs_pool_t *pool, tfs_size_t size)
{
	void *mem;
	tfs_pool_elem *elem;

	if (pool == NULL || size == 0 || !pool->active) {
		return NULL;
	}

	mem = malloc(size);
	if (mem != NULL) {
		/* メモリ領域管理インスタンスの生成 */
		elem = malloc(sizeof(tfs_pool_elem));
		elem->mem  = mem;
		elem->next = NULL;

		/* mutex ロック */
		pthread_cleanup_push(_cleanup_pthread_mutex, (void *)pool->mutex);
		(void) pthread_mutex_lock(pool->mutex);

		/* リストにつなげる */
		if (pool->first == NULL) {
			pool->first = pool->last = elem;
		}
		else {
			/* 末尾をelem にする */
			pool->last->next = elem;
			pool->last = pool->last->next;
		}
		pool->total++;	/* 要素数をカウントアップ */

		/* mutex アンロック */
		(void) pthread_mutex_unlock(pool->mutex);
		pthread_cleanup_pop(0);
	}

	return mem;
}

/**
 * メモリアロケーターpool からサイズsize バイトのメモリ領域を取得して
 * 取得した領域を0クリアする.
 *
 */
TFS_DECLARE(void *) tfs_pcalloc(tfs_pool_t *pool, tfs_size_t size)
{
	void *mem = tfs_palloc(pool, size);
	if (mem != NULL) {
		memset(mem, 0, size);
	}
	return mem;
}

/**
 * pool が示すメモリアロケーターがクリアまたは破棄される直前にコールされる
 * クリーンアップハンドラ関数を登録する.
 *
 */
TFS_DECLARE(void) tfs_pool_cleanup_register(tfs_pool_t *pool, const void *data,
                                            tfs_pool_cleanup_t cleanup_fn)
{
	tfs_pool_cleanup *cl = NULL;

	if (pool == NULL || cleanup_fn == NULL) {
		return;
	}

	/* mutex ロック */
	pthread_cleanup_push(_cleanup_pthread_mutex, (void *)pool->mutex);
	(void) pthread_mutex_lock(pool->mutex);

	if (pool->cleanup == NULL) {
		pool->cleanup = cl = malloc(sizeof(tfs_pool_cleanup));
		cl->cleanup_fn = cleanup_fn;
		cl->data       = data;
		cl->next       = NULL;
	}
	else {
		int found_same = 0;
		for (cl = pool->cleanup; cl; cl = cl->next) {
			if (cl->cleanup_fn == cleanup_fn && cl->data == data) {
				found_same = 1;	/* 同じものを見つけた */
				break;
			}
		}

		if (!found_same) {
			cl = malloc(sizeof(tfs_pool_cleanup));
			cl->cleanup_fn = cleanup_fn;
			cl->data       = data;
			cl->next       = pool->cleanup;	/* cl の後ろにつなげる */
			pool->cleanup  = cl;
		}
	}

	/* mutex アンロック */
	(void) pthread_mutex_unlock(pool->mutex);
	pthread_cleanup_pop(0);
}

/**
 * 引数data とcleanup_f が示すクリーンアップハンドラの登録をキャンセルし、
 * 呼び出されないようにする.
 *
 */
TFS_DECLARE(void) tfs_pool_cleanup_kill(tfs_pool_t *pool, const void *data,
                                        tfs_pool_cleanup_t cleanup_fn)
{
	tfs_pool_cleanup *cl = NULL, *prev;

	if (pool == NULL || pool->cleanup == NULL || cleanup_fn == NULL) {
		return;
	}

	/* mutex ロック */
	pthread_cleanup_push(_cleanup_pthread_mutex, (void *)pool->mutex);
	(void) pthread_mutex_lock(pool->mutex);

	prev = NULL;
	for (cl = pool->cleanup; cl; cl = cl->next) {
		if (cl->cleanup_fn == cleanup_fn && cl->data == data) {
			/*
			 * cleanup_fn, data をNULL化する.
			 * (note)
			 *   cl をリストから取り除いてはなりません. tfs_pool_cleanup_kill が
			 *   tfs_pool_release の中から再帰的にコールされる可能性があるためです.
			 *   例えば、基本ユーティリティが別の基本ユーティリティを使うケースなど.
			 *   この場合、cl を触ると外部はcl でループしているので問題を起こします.
			 */
			cl->cleanup_fn = NULL;
			cl->data       = NULL;
			break;
		}
		prev = cl;
	}

	/* mutex アンロック */
	(void) pthread_mutex_unlock(pool->mutex);
	pthread_cleanup_pop(0);
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * スレッド同期用Mutex を生成する.
 * (note)
 *    Mutex はPTHREAD_MUTEX_NORMAL属性で生成します. 
 *
 * @param mutex pthread_mutex_t ** 生成されたMutex (heap)
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _create_pthread_mutex(pthread_mutex_t **mutex)
{
	int ret;
	pthread_mutexattr_t mattr;

	*mutex = NULL;	/* 初期化 */

	/* mutex属性の初期化 */
	if ((ret = pthread_mutexattr_init(&mattr)) != 0) {
		return 1;
	}

	/* 属性値の設定 */
	if ((ret = pthread_mutexattr_settype(&mattr, PTHREAD_MUTEX_NORMAL)) != 0) {
		(void) pthread_mutexattr_destroy(&mattr);
		return 1;
	}

	/* mutex の生成 */
	*mutex = malloc(sizeof(pthread_mutex_t));
	if ((ret = pthread_mutex_init(*mutex, &mattr)) != 0) {
		(void) pthread_mutexattr_destroy(&mattr);
		return 1;
	}

	/* mutex 属性は不要になったので破棄 */
	(void) pthread_mutexattr_destroy(&mattr);

	return 0;
}

/**
 * mutex が示すMutex を破棄する.
 * この関数コール後、mutex を触ってはなりません.
 *
 * @param mutex pthread_mutex_t	*
 */
static void _destroy_pthread_mutex(pthread_mutex_t *mutex)
{
	if (mutex == NULL) return;

	(void) pthread_mutex_destroy(mutex);
	free(mutex);	/* ヒープから割り当てられたメモリを解放 */
}

/**
 * スレッドにpush されるクリーンアップハンドラ
 *
 * @param mutex void *
 */
static void _cleanup_pthread_mutex(void *mutex)
{
	(void) pthread_mutex_unlock((pthread_mutex_t *)mutex);
}


