/**
 * $Id$
 *
 * 汎用ユーティリティ (Queue : FIFO のQueue)
 */
#include "tfs_queue.h"
#include "tfs_linkedlist.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * Queueを表すコンテキスト構造体 [ 不完全型の定義 ]
 */
struct tfs_queue_t {
	tfs_linkedlist_t *qlist;

	/* メモリアロケーター */
	tfs_pool_t *pool;
	int allocated_pool;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_queue(void *data);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * Queue の生成.
 *
 */
TFS_DECLARE(void) tfs_queue_create(tfs_pool_t *pool, tfs_queue_t **queue)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	/* コンテキスト生成 (コンテキストはmallocで) */
	*queue = malloc(sizeof(tfs_queue_t));
	(*queue)->qlist = NULL;
	(*queue)->pool  = p;
	(*queue)->allocated_pool = allocated_pool;

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *queue, _cleanup_queue);
}

/**
 * Queue に値value を入れる. (PUSH)
 *
 */
TFS_DECLARE(tfs_status_t) tfs_queue_push(tfs_queue_t *queue, void *value)
{
	if (queue == NULL) {
		return TFS_ENOPARAM;
	}

	if (value == NULL) {
		return TFS_ENULLPOINTER;
	}

	/* リストを生成する */
	if (queue->qlist == NULL) {
		tfs_linkedlist_create(queue->pool, &queue->qlist);
	}

	if (queue->qlist != NULL) {
		/* リストにvalue を追加 */
		tfs_linkedlist_add(queue->qlist, value);
	}

	return TFS_SUCCESS;
}

/**
 * Queue から値を取り出す. (POP)
 * 取り出された値はQueueから取り除かれます.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_queue_pop(tfs_queue_t *queue, void **value)
{
	*value = NULL;	/* 初期化 */

	if (queue == NULL || queue->qlist == NULL) {
		return TFS_QUEUE_EMPTY;
	}

	if (tfs_linkedlist_size(queue->qlist) == 0) {
		return TFS_QUEUE_EMPTY;
	}

	/* Queueの先頭要素を取り出して削除する */
	*value = tfs_linkedlist_remove(queue->qlist, 0);

	return TFS_SUCCESS;
}

/**
 * Queue から値を取り出す. (POP)
 * 取り出された値はQueueから"取り除かれません".(tfs_queue_pop との違い)
 *
 */
TFS_DECLARE(tfs_status_t) tfs_queue_peek(tfs_queue_t *queue, void **value)
{
	*value = NULL;	/* 初期化 */

	if (queue == NULL || queue->qlist == NULL) {
		return TFS_QUEUE_EMPTY;
	}

	if (tfs_linkedlist_size(queue->qlist) == 0) {
		return TFS_QUEUE_EMPTY;
	}

	/* Queueの先頭要素を取り出す(削除はしない) */
	*value = tfs_linkedlist_get(queue->qlist, 0);

	return TFS_SUCCESS;
}

/**
 * queue が示すQueueをpool のメモリ領域にコピーする.
 *
 */
TFS_DECLARE(tfs_queue_t *) tfs_queue_clone(tfs_pool_t *pool, tfs_queue_t *queue)
{
	tfs_queue_t *new_queue = NULL;

	if (queue == NULL) {
		return NULL;	/* クローンできない */
	}

	new_queue = malloc(sizeof(tfs_queue_t));
	new_queue->pool = pool;
	new_queue->allocated_pool = 0;

	/* linkedlist のクローンを作成する */
	new_queue->qlist = tfs_linkedlist_clone(pool, queue->qlist);

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(pool, new_queue, _cleanup_queue);

	return new_queue;
}

/**
 * Queue に格納された要素の総数を取得する.
 *
 */
TFS_DECLARE(tfs_int32_t) tfs_queue_size(tfs_queue_t *queue)
{
	if (queue == NULL || queue->qlist == NULL) {
		return 0;
	}

	return tfs_linkedlist_size(queue->qlist);
}

/**
 * queue が示すQueueの要素を全てクリアする.
 *
 */
TFS_DECLARE(void) tfs_queue_clear(tfs_queue_t *queue)
{
	if (queue == NULL || queue->qlist == NULL) {
		return;
	}

	tfs_linkedlist_clear(queue->qlist);
}

/**
 * queue が示すQueueの破棄.
 *
 */
TFS_DECLARE(void) tfs_queue_destroy(tfs_queue_t *queue)
{
	if (queue == NULL) return;

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(queue->pool, queue, _cleanup_queue);

	_cleanup_queue(queue);
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static void _cleanup_queue(void *data)
{
	tfs_queue_t *queue = data;

	if (queue == NULL) return;

	if (queue->qlist != NULL) {
		/* Queue の実装であるListを破棄 */
		tfs_linkedlist_destroy(queue->qlist);
		queue->qlist = NULL;
	}

	/* メモリアロケータを破棄 */
	if (queue->pool != NULL && queue->allocated_pool) {
		tfs_pool_destroy(queue->pool);
		queue->pool = NULL;
	}
	free(queue);	/* 以降、queue に触ってはならない */
}

