/**
 * $Id$
 *
 * 汎用ユーティリティ (StringBuffer : 可変長文字列バッファ)
 *
 */
#include "tfs_sbuf.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
#define SBUF_DEFAULT_CAPACITY	256	/* デフォルトバッファサイズ(byte) */
#define SBUF_DEFAULT_DELTA		128	/* デフォルトバッファ増加量(byte) */
#define SBUF_EXTEND_DELTA_CNT	5	/* delta拡張までのカウント数      */

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * [ 不完全型の定義 ]
 * 可変長文字列バッファを実現する構造体の定義
 */
struct tfs_sbuf_t {
	tfs_size_t capacity;/* 割り当てられた文字列バッファ の長さ(byte) */
	tfs_size_t length;	/* 格納された文字列の長さ */
	tfs_size_t delta;	/* 文字列バッファの拡張サイズ(byte) */
	int extendcnt;	/* 文字列バッファが拡張された回数 */

	/* 追加された文字列を持つバッファ */
	char *buf;

	/* メモリアロケータ */
	tfs_pool_t *pool;
	int allocated_pool;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_sbuf(void *data);
static void _realloc_sbuf(tfs_sbuf_t *sbuf, tfs_size_t len);

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 指定されたsize バイトの文字列を格納可能な文字列バッファsbuf を生成して
 * 返却する。
 *
 */
TFS_DECLARE(void) tfs_sbuf_create(tfs_pool_t *pool, tfs_size_t size, tfs_sbuf_t **sbuf)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	*sbuf = malloc(sizeof(tfs_sbuf_t));

	if (size >= 0) {
		(*sbuf)->capacity = size;
		(*sbuf)->delta    = size / 2;
		if ((*sbuf)->delta < SBUF_DEFAULT_CAPACITY) {
			(*sbuf)->delta = SBUF_DEFAULT_DELTA;
		}
	}
	else {
		(*sbuf)->capacity = SBUF_DEFAULT_CAPACITY;	/* デフォルト値 */
		(*sbuf)->delta    = SBUF_DEFAULT_DELTA;
	}
	(*sbuf)->length    = 0;
	(*sbuf)->extendcnt = 0;

	/* 文字列バッファは'\0'の1バイト分だけ余計に確保しておく */
	(*sbuf)->buf = malloc((*sbuf)->capacity + 1);
	memset((*sbuf)->buf, '\0', (*sbuf)->capacity + 1);
	(*sbuf)->pool = p;
	(*sbuf)->allocated_pool = allocated_pool;

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *sbuf, _cleanup_sbuf);
}

/**
 * 指定されたstr が示す文字列の長さを初期値としての文字列バッファsbuf を
 * 初期化して返却する。
 *
 */
TFS_DECLARE(void) tfs_sbuf_createbyStr(tfs_pool_t *pool, const char *str, tfs_sbuf_t **sbuf)
{
	tfs_size_t size = SBUF_DEFAULT_CAPACITY;

	if (str != NULL) {
		size = strlen(str);
	}

	/* size 分のバッファを確保する */
	tfs_sbuf_create(pool, size, sbuf);

	/* str のコピー */
	if (str != NULL) {
		memcpy((*sbuf)->buf, str, size);
		(*sbuf)->buf[size] = '\0';	/* 終端文字を入れる */
		(*sbuf)->length = size;
	}
	else {
		(*sbuf)->buf[0] = '\0';
		(*sbuf)->length = 0;
	}
}

/**
 * 文字列バッファsbuf に設定された文字列バッファの現在のサイズ(byte)を取得する。
 *
 */
TFS_DECLARE(tfs_size_t) tfs_sbuf_getcapacity(tfs_sbuf_t *sbuf)
{
	return (sbuf != NULL) ? sbuf->capacity : 0;
}

/**
 * 文字列バッファsbuf に追加された文字列の総長さ(byte)を取得する。
 *
 */
TFS_DECLARE(tfs_size_t) tfs_sbuf_getlength(tfs_sbuf_t *sbuf)
{
	return (sbuf != NULL) ? sbuf->length : 0;
}

/**
 * 文字列バッファsbuf に最後に追加された文字列の後ろににstr が示す文字列を
 * 追加する。
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @param str const char * 追加する文字列
 */
TFS_DECLARE(void) tfs_sbuf_append(tfs_sbuf_t *sbuf, const char *str)
{
	if (str == NULL) return;

	/* 文字列 str をsbuf に追加する */
	tfs_sbuf_appendByte(sbuf, strlen(str), str);
}

/**
 * 文字列バッファsbuf の最後に追加された文字列の後ろににstr が示す文字列を
 * len バイトだけ追加する。
 *
 */
TFS_DECLARE(void) tfs_sbuf_appendByte(tfs_sbuf_t *sbuf,
                                      tfs_size_t len, const char *str)
{
	if (len == 0 || str == NULL) return;

	/* バッファ領域の拡張が必要なら拡張させる(+1 多く取る) */
	_realloc_sbuf(sbuf, len + 1);

	/* バッファにコピーする */
	memcpy(sbuf->buf + sbuf->length, str, len);
	sbuf->length += len;
	sbuf->buf[sbuf->length] = '\0';
}

/**
 * 文字列バッファsbuf の最後に追加された文字列の後ろににch が示す文字を追加する.
 * ch が'\0' だと追加しません.
 *
 */
TFS_DECLARE(void) tfs_sbuf_appendChar(tfs_sbuf_t *sbuf, char ch)
{
	if (sbuf == NULL || ch == '\0') return;

	/* バッファ領域の拡張が必要なら拡張させる(+1 多く取る) */
	_realloc_sbuf(sbuf, 1 + 1);

	/* バッファにコピーする */
	sbuf->buf[sbuf->length] = ch;
	sbuf->length++;
	sbuf->buf[sbuf->length] = '\0';
}

/**
 * 文字列バッファsbuf のstart が示すインデックス位置に文字列str を追加する。
 * start位置にあった文字はstrlen(str) 分だけ後ろに移動されます。
 *
 */
TFS_DECLARE(tfs_status_t) tfs_sbuf_insert(tfs_sbuf_t *sbuf,
                                          tfs_size_t start, const char *str)
{
	tfs_size_t len;
	char *pos;

	if (sbuf == NULL) {
		return TFS_ENOPARAM;
	}
	if (str == NULL) {
		return TFS_SUCCESS;	/* 何もしなくていいだろう */
	}

	if (start > sbuf->length) {
		return TFS_EINVALID_PARAM;
	}
	len = strlen(str);

	/* バッファ領域の拡張が必要なら拡張させる */
	_realloc_sbuf(sbuf, len);

	pos = sbuf->buf + start;	/* 追加位置 */
	memmove(pos + len, pos, (sbuf->length - start));
	memcpy(pos, str, len);

	sbuf->length += len;
	sbuf->buf[sbuf->length] = '\0';

	return TFS_SUCCESS;
}

/**
 * 文字列バッファsbuf のstart が示すインデックス位置からインデックス位置
 * (end - 1) までをstr が示す文字列で置換する
 * str が 置換長さよりも長い場合には切り取って挿入します。
 * また、str が 置換長さよりも短い場合には、strlen(str) までをコピーします。
 * end がバッファ長さよりも大きい場合にはバッファを自動拡張します。
 *
 */
TFS_DECLARE(tfs_status_t) tfs_sbuf_replace(tfs_sbuf_t *sbuf, tfs_size_t start,
                                           tfs_size_t end, const char *str)
{
	tfs_size_t len, cpy_len, realoc_len = 0;

	if (sbuf == NULL) {
		return TFS_ENOPARAM;
	}
	if (start > sbuf->length) {
		return TFS_EINVALID_PARAM;
	}

	cpy_len = end - start;
	if (str == NULL || cpy_len <= 0) {
		return TFS_SUCCESS;	/* 何もしなくていいだろう */
	}
	len = strlen(str);
	if (cpy_len > len) {
		cpy_len = len;	/* 最大コピーサイズをlen にする */
	}

	/* バッファ領域の拡張が必要なら拡張させる */
	if (cpy_len > sbuf->length - start) {
		realoc_len = cpy_len - (sbuf->length - start);
		_realloc_sbuf(sbuf, realoc_len);
	}

	/* start 位置から文字列をコピーする */
	memcpy(sbuf->buf + start, str, cpy_len);
	if (realoc_len > 0) {
		sbuf->length += realoc_len;     /* 増加分だけ加算する */
	}
	sbuf->buf[sbuf->length] = '\0';

	return TFS_SUCCESS;
}

/**
 * 文字列バッファsbuf から格納されている全ての文字列を取得する。
 *
 */
TFS_DECLARE(char *) tfs_sbuf_toString(tfs_sbuf_t *sbuf)
{
	return (sbuf != NULL) ? sbuf->buf : NULL;
}

/**
 * 文字列バッファsbuf に格納された文字列を全て消去する。
 * ただし、確保されたバッファ領域はクリアされません。
 *
 */
TFS_DECLARE(void) tfs_sbuf_clear(tfs_sbuf_t *sbuf)
{
	if (sbuf == NULL) return;

	sbuf->length = 0;	/* 0バイトしか記録されていないものとする */
	memset(sbuf->buf, '\0', sbuf->capacity);
}

/**
 * sbuf にアロケートされていたメモリ領域を回収する.
 * この関数呼出し後、sbuf に触ってはなりません.
 *
 */
TFS_DECLARE(void) tfs_sbuf_destroy(tfs_sbuf_t *sbuf)
{
	if (sbuf == NULL) return;

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(sbuf->pool, sbuf, _cleanup_sbuf);

	_cleanup_sbuf(sbuf);
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static void _cleanup_sbuf(void *data)
{
	tfs_sbuf_t *sbuf = data;
	if (sbuf == NULL) return;

	if (sbuf->buf != NULL) {
		free(sbuf->buf);
		sbuf->buf = NULL;
	}

	if (sbuf->pool != NULL && sbuf->allocated_pool) {
		tfs_pool_destroy(sbuf->pool);
		sbuf->pool = NULL;
	}
	free(sbuf);
}

/**
 * 文字列バッファsbuf にlen バイト分の文字列領域を確保できるかどうか検証し、
 * 出来なければ割り当てられるようsbuf を拡張する。
 * (note) バッファ拡張のルール
 * 	(1) 現バッファの総空き容量をlen が上回っていれば len + deltaだけ拡張する
 * 	(2) バッファの拡張がSBUF_EXTEND_DELTA_CNT 回以上発生していたら
 * 	    delta を現在の値の2倍に増やして、バッファの拡張を行う
 *
 * @param sbuf tfs_sbuf_t * 文字列バッファへのポインタ
 * @param len tfs_size_t 割り当てたいバイト長
 */
static void _realloc_sbuf(tfs_sbuf_t *sbuf, tfs_size_t len)
{
	if (len <= 0) return;	/* sbuf のチェックは済んでいるものとします */

	/* 文字列バッファをオーバしていないか? */
	if (sbuf->length + len > sbuf->capacity) {
		char *newbuf;

		/* 文字列バッファ拡張回数のチェック */
		if (sbuf->extendcnt > 0 &&
			sbuf->extendcnt % SBUF_EXTEND_DELTA_CNT == 0) {
			/* 拡張が多いのは恐らくdeltaが小さいため。倍に増やしてみる */
			sbuf->delta = sbuf->delta * 2;
		}

		/* 連続領域を確保するため、新たに確保しなおす */
		sbuf->capacity += len + sbuf->delta;
		newbuf = malloc(sbuf->capacity);
		memcpy(newbuf, sbuf->buf, sbuf->length);	/* 旧バッファをコピー */
		newbuf[sbuf->length] = '\0';
		free(sbuf->buf);	/* 旧バッファは破棄 */
		sbuf->buf = newbuf;	/* バッファの新旧交代 */

		sbuf->extendcnt++;	/* 拡張カウントをインクリメント */
	}
}


