/**
 * $Id$
 *
 * 汎用ユーティリティ (string : 文字列操作関連)
 *
 */
#include "tfs_string.h"
#include "tfs_varray.h"
#include "tfs_sbuf.h"

#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#if HAVE_STDARG_H
#include <stdarg.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
#define ALLOC_MEM(pool,size)	(((pool) == NULL) ? malloc((size)) : tfs_palloc(pool, (size)))

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static char * _strndup(tfs_pool_t *pool, const char *s, size_t n);
static char * _strncatv(tfs_pool_t *pool, size_t n, int usesize, const char *s0, va_list va);
static void _no2slash(char *str);

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * s が示す文字列のメモリ領域をpoolに確保して内容をコピーする. (メモリアロケーター版)
 *
 */
TFS_DECLARE(char *) tfs_pstrdup(tfs_pool_t *pool, const char *s)
{
	if (IS_EMPTY(s) || pool == NULL) return NULL;

	return _strndup(pool, s, strlen(s));
}

/**
 * s が示す文字列のうち、n文字分を複製してヌル終端で返却. (メモリアロケーター版)
 * s の長さよりもn の方が大きい場合には、tfs_pstrdup() と同じ動作となります.
 *
 */
TFS_DECLARE(char *) tfs_pstrndup(tfs_pool_t *pool, const char *s, tfs_size_t n)
{
	if (pool == NULL) return NULL;

	return _strndup(pool, s, (size_t) n);
}

/**
 * 指定された複数の文字列を連結する. (メモリアロケーター版)
 *
 */
TFS_DECLARE(char *) tfs_pstrcat(tfs_pool_t *pool, const char *s0, ...)
{
	va_list va;
	char *dst;

	if (IS_EMPTY(s0) || pool == NULL) return NULL;

	va_start(va, s0);
	dst = _strncatv(pool, 0, 0, s0, va);
	va_end(va);

	return dst;
}

/**
 * 指定された複数の文字列をn文字分だけ連結する.(メモリアロケーター版)
 *
 */
TFS_DECLARE(char *) tfs_pstrncat(tfs_pool_t *pool, tfs_size_t n, const char *s0, ...)
{
	va_list va;
	char *dst;

	if (IS_EMPTY(s0) || n == 0 || pool == NULL) return NULL;

	va_start(va, s0);
	dst = _strncatv(pool, n, 1, s0, va);
	va_end(va);

	return dst;
}

/**
 * printf-style の書式指定文字を利用できる文字列関数.
 * 指定されたフォーマットの文字列を合成して返却します.
 *
 */
TFS_DECLARE(char *) tfs_psprintf(tfs_pool_t *pool, const char *fmt, ...)
{
	va_list args;
	char *str, *buf;
	int n, size = 256;
	char initbuf[256] = { 0 };
	int alloc_heap = 0;

	if (pool == NULL || fmt == NULL) {
		return NULL;
	}

	buf = initbuf;
	while (1) {
		va_start(args, fmt);
		n = vsnprintf(buf, size, fmt, args);
		va_end(args);

		/* buf に十分なサイズがあった場合 */
		if (n > -1 && n < size) {
			str = tfs_pstrndup(pool, buf, n);
			if (alloc_heap) {
				free(buf);
			}
			return str;
		}
		/* buf に十分なサイズがなかった */
		else if (n > -1) {
			if (alloc_heap) {
				free(buf);
			}
			size = n + 1;
			buf = malloc(size);
			alloc_heap = 1;
			continue;	/* リトライ */
		}
		else {
			if (alloc_heap) {
				free(buf);
			}
			return NULL;	/* 書き込みに失敗 */
		}
	}
}

/**
 * 多重スラッシュをシングルスラッシュにする.
 *
 */
TFS_DECLARE(char *) tfs_no2slash(tfs_pool_t *pool, const char *str)
{
	char *dst;

	if (IS_EMPTY(str)) return NULL;

	dst = tfs_pcalloc(pool, strlen(str) + 1);
	_no2slash(dst);

	return dst;
}

/**
 * ファイルパスを組み立てて返却する.
 *
 */
TFS_DECLARE(char *) tfs_make_full_path(tfs_pool_t *pool, const char *root,
                                       const char *f, ...)
{
	char *cp, *ubuf_str;
	tfs_sbuf_t *ubuf = NULL;
	va_list args;

	/* バッファの作成 */
	tfs_sbuf_create(pool, 256, &ubuf);

	/* root, f をセットする */
	if (IS_FILLED(root)) {
		tfs_sbuf_append(ubuf, root);
	}

	if (f != NULL) {		/* (note) IS_FILLED では駄目!! */
		va_start(args, f);

		if (IS_FILLED(f)) {
			/* ２重スラッシュになるが気にしない。後で除去するので */
			tfs_sbuf_appendByte(ubuf, 1, "/");
			tfs_sbuf_append(ubuf, f);
		}

		/* NULL が現れるまでループする */
		while ((cp = va_arg(args, char *)) != NULL) {
			if (IS_FILLED(cp)) {
				/* ２重スラッシュになるが気にしない。後で除去するので */
				tfs_sbuf_appendByte(ubuf, 1, "/");
				tfs_sbuf_append(ubuf, cp);
			}
		}
	}

	/* va_start で初期化したデータを解放 */
	va_end(args);

	/* 2重スラッシュを取り除く */
	ubuf_str = tfs_sbuf_toString(ubuf);
	_no2slash(ubuf_str);

	return ubuf_str;
}


/**
 * s が示す文字列のメモリ領域をヒープに確保し、内容をコピーする.
 *
 */
TFS_DECLARE(char *) tfs_strdup(const char *s)
{
	if (IS_EMPTY(s)) return NULL;

	return _strndup(NULL, s, strlen(s));
}

/**
 * s が示す文字列のうち、n文字分を複製してヌル終端で返却.
 *
 */
TFS_DECLARE(char *) tfs_strndup(const char *s, tfs_size_t n)
{
	return _strndup(NULL, s, (size_t) n);
}

/**
 * 指定された複数の文字列を連結する.
 *
 */
TFS_DECLARE(char *) tfs_strcat(const char *s0, ...)
{
	va_list va;
	char *dst;

	if (IS_EMPTY(s0)) return NULL;

	va_start(va, s0);
	dst = _strncatv(NULL, 0, 0, s0, va);
	va_end(va);

	return dst;
}

/**
 * 指定された複数の文字列をn文字分だけ連結する.
 *
 */
TFS_DECLARE(char *) tfs_strncat(tfs_size_t n, const char *s0, ...)
{
	va_list va;
	char *dst;

	if (IS_EMPTY(s0) || n == 0) return NULL;

	va_start(va, s0);
	dst = _strncatv(NULL, n, 1, s0, va);
	va_end(va);

	return dst;
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 *
 */
static char * _strndup(tfs_pool_t *pool, const char *s, size_t n)
{
	size_t len, cplen;
	char *dst;

	if (IS_EMPTY(s) || n == 0) return NULL;

	len = strlen(s);
	cplen = (len < n) ? len : n;	/* コピーする文字列長さを算出 */

	dst = ALLOC_MEM(pool, cplen + 1);
	memcpy(dst, s, cplen);
	dst[cplen] = '\0';

	return dst;
}

/**
 *
 */
static char * _strncatv(tfs_pool_t *pool, size_t n, int usesize, const char *s0, va_list va)
{
	char *dst, *first = NULL, *s;
	size_t len, total = 0;
	tfs_varray_t *parts = NULL;
	tfs_int32_t plen;

	if (IS_EMPTY(s0) || (usesize && (n == 0))) return NULL;

	/* 可変長文字列を一時的に格納しておく配列を確保 */
	tfs_varray_create(pool, 5, &parts);

	/* 先頭の文字列を入れる */
	total = strlen(s0);
	tfs_varray_add(parts, (char *)s0);

	/* NULL が現れるまでループする */
	while ((s = va_arg(va, char *)) != NULL) {
		len = strlen(s);
		total += len;
		if (usesize) {
			if (total == n) {
				tfs_varray_add(parts, s);
				break;
			}
			else if (total < n) {
				tfs_varray_add(parts, s);
			}
			else {
				/* 切り取ってコピーする */
				total -= len;
				if (n - total > 0) {
					tfs_varray_add(parts, _strndup(pool, s, (n - total)));
					total += n - total;
				}
				break;
			}
		}
		else {
			tfs_varray_add(parts, s);
		}
	}

	/* パーツを全て連結する */
	if ((plen = tfs_varray_size(parts)) > 0) {
		int i;
		first = dst = ALLOC_MEM(pool, total + 1);
		for (i = 0; i < plen; i++) {
			s = tfs_varray_get(parts, i);
			if (s == NULL) continue;

			len = strlen(s);
			memcpy(dst, s, len);
			dst += len;

			if (i == plen - 1 && usesize) {
				free(s);	/* 最後のコピー文字列をfreeする */
			}
		}
		first[total] = '\0';
	}

	tfs_varray_destroy(parts);	/* クリーンアップ */

	return first;
}

/**
 * str に含まれる多重スラッシュをシングルスラッシュにして返却する.
 *
 * @param str char *
 */
static void _no2slash(char *str)
{
	char *d, *s;

	s = d = str;

	while (*s) {
		if ((*d++ = *s) == '/') {
			do {
				++s;
			}
			while (*s == '/');
		}
		else {
			++s;
		}
	}
	*d = '\0';
}

