/**
 * $Id$
 *
 * 汎用ユーティリティ (Array : 可変長配列コンテナ)
 *
 */
#include "tfs_varray.h"

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
#define VARRAY_DEFAULT_NALLOC	10
#define VARRAY_DEFAULT_DELTA	20

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * 可変長配列を表すコンテキスト構造体 [ 不完全型の定義 ]
 */
struct tfs_varray_t {
	int nalloc;	/* 可変長配列として割り当てられている総要素数 */
	int nelts;	/* 可変長配列に格納されているの要素の数 */
	int delta;	/* 次回拡張する際に使用される配列の長さ */

	/* 可変長配列へのポインタ(void *) をlength 個持つ配列 */
	void **elements;

	/* メモリアロケータ */
	tfs_pool_t *pool;
	int allocated_pool;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _cleanup_varray(void *data);
static void _realloc_array(tfs_varray_t *array);

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * 指定された初期要素数size 個の要素をもつポインタ保持用可変長配列を生成する。
 *
 */
TFS_DECLARE(void) tfs_varray_create(tfs_pool_t *pool,
                                    tfs_int32_t size, tfs_varray_t **varray)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	*varray = malloc(sizeof(tfs_varray_t));

	if (size >= 0) {
		(*varray)->nalloc = size;
		/* delta の算出 */
		(*varray)->delta = (*varray)->nalloc * 2;
	}
	else {
		(*varray)->nalloc = VARRAY_DEFAULT_NALLOC;
		(*varray)->delta  = VARRAY_DEFAULT_DELTA;
	}
	/* 可変長配列の枠を生成 */
	(*varray)->elements = malloc(sizeof(void *) * (*varray)->nalloc);
	memset((*varray)->elements, 0, sizeof(void *) * (*varray)->nalloc);
	(*varray)->nelts = 0;
	(*varray)->pool  = p;
	(*varray)->allocated_pool = allocated_pool;

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *varray, _cleanup_varray);
}

/**
 * 可変長配列varray の末尾にelement が示すオブジェクトを追加する。
 *
 */
TFS_DECLARE(void) tfs_varray_add(tfs_varray_t *varray, void *element)
{
	if (varray == NULL) {
		return;
	}

	if (varray->nalloc < varray->nelts + 1) {
		_realloc_array(varray);
	}

	varray->elements[varray->nelts] = element;
	varray->nelts++;
}

/**
 * 可変長配列varray の位置index のオブジェクトをelement で置き換える。
 * index が範囲外であれば何もしません。
 *
 */
TFS_DECLARE(void) tfs_varray_set(tfs_varray_t *varray,
                                 tfs_int32_t index, void *element)
{
	if (varray == NULL || index < 0 || index > varray->nelts) {
		return;
	}

	varray->elements[index] = element;
}

/**
 * 可変長配列varray の位置index にあるオブジェクトを削除し、返却する。
 *
 */
TFS_DECLARE(void *) tfs_varray_remove(tfs_varray_t *varray, tfs_int32_t index)
{
	void *element;
	int i;

	if (varray == NULL || index < 0 || index > varray->nelts) {
		return NULL;
	}

	element = varray->elements[index];	/* 要素の取り出し */
	/* index 要素を除去する */
	for (i = index; i < varray->nelts - 1; i++) {
		varray->elements[i] = varray->elements[i+1];
	}
	varray->elements[varray->nelts - 1] = NULL;
	varray->nelts--;	/* 要素を1つ減らす */

	return element;
}

/**
 * 可変長配列varray の総要素数を取得する。
 *
 */
TFS_DECLARE(tfs_int32_t) tfs_varray_size(tfs_varray_t *varray)
{
	if (varray == NULL) return 0;

	return varray->nelts;
}

/**
 * 可変長配列varray の中の位置index にある要素を取得する。
 * remove と異なり、index 位置の要素は除去されません。
 *
 */
TFS_DECLARE(void *) tfs_varray_get(tfs_varray_t *varray, tfs_int32_t index)
{
	if (varray == NULL || index < 0 || index > varray->nelts) {
		return NULL;
	}

	return varray->elements[index];
}

/**
 * 可変長配列varray の要素を全てクリアする。
 *
 */
TFS_DECLARE(void) tfs_varray_clear(tfs_varray_t *varray)
{
	if (varray == NULL) return;

	varray->nalloc = 0;
	varray->nelts  = 0;
	if (varray->elements != NULL) {
		free(varray->elements);
		varray->elements = NULL;
	}
}

/**
 * 可変長配列varray の要素全てを取り出す。
 *
 */
TFS_DECLARE(void **) tfs_varray_getelements(tfs_varray_t *varray)
{
	return varray->elements;
}

/**
 * varray が示す可変長配列をpool のメモリ領域にコピーする.
 *
 */
TFS_DECLARE(tfs_varray_t *) tfs_varray_clone(tfs_pool_t *pool, tfs_varray_t *varray)
{
	tfs_varray_t *new_varray = NULL;

	if (varray == NULL) {
		return NULL;
	}

	/* 新規作成 && シャローコピー */
	new_varray = malloc(sizeof(tfs_varray_t));

	new_varray->nalloc   = varray->nalloc;
	new_varray->delta    = varray->delta;
	new_varray->elements = malloc(sizeof(void *) * new_varray->nalloc);
	memcpy(new_varray->elements, varray->elements, sizeof(void *) * new_varray->nalloc);
	new_varray->nelts    = varray->nelts;
	new_varray->pool     = pool;
	new_varray->allocated_pool = 0;

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(pool, new_varray, _cleanup_varray);

	return new_varray;
}

/**
 * 可変長配列varray がアロケートしていた領域を破棄する.
 *
 */
TFS_DECLARE(void) tfs_varray_destroy(tfs_varray_t *varray)
{
	if (varray == NULL) return;
	
	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(varray->pool, varray, _cleanup_varray);

	_cleanup_varray(varray);
}

/*------------------------------------------------------------------------------
 * Define private function
 *----------------------------------------------------------------------------*/
static void _cleanup_varray(void *data)
{
	tfs_varray_t *varray = data;
	if (varray == NULL) return;

	tfs_varray_clear(varray);

	if (varray->pool != NULL && varray->allocated_pool) {
		tfs_pool_destroy(varray->pool);
		varray->pool = NULL;
	}
	free(varray);
}

/**
 * 可変長配列varray を拡張する。
 *
 * @param array tfs_varray_t * 可変長配列へのポインタ
 */
static void _realloc_array(tfs_varray_t *varray)
{
	void **data;
	int new_size;

	if (varray == NULL) return;

	/* 連続領域を確保するため、新たに確保しなおす */
	new_size = varray->nalloc + varray->delta;
	data = malloc(sizeof(void *) * new_size);
	memcpy(data, varray->elements, sizeof(void *) * varray->nalloc);

	varray->nalloc   = new_size;
	free(varray->elements);	/* 旧領域の破棄 */
	varray->elements = data;
}

