/**
 * $Id$
 *
 * 汎用ユーティリティ (XML DOM parser : XMLのDOM パーサ実装)
 */
#include "tfs_xml.h"
#include "tfs_sbuf.h"
#include "tfs_string.h"
#include "tfs_pools.h"
#include "tfs_fileio.h"

#if HAVE_EXPAT_H
#include <expat.h>
#else
#error "expat.h is missing. Please install expat-devel library."
#endif
/**
 * 互換性に問題のあるXML_Status の切り替え
 */
#ifndef XML_STATUS_OK
#define XML_STATUS_T int
#define XML_STATUS_OK 1
#define XML_STATUS_ERROR 0
#else
#define XML_STATUS_T	enum XML_Status
#endif	/* TFS_HAVE_XML_STATUS_TYPE */

#if HAVE_STDIO_H
#include <stdio.h>
#endif
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * Expat XML パーサが処理対象とするXMLのエンコーディング
 */
#define TFS_EXPAT_ENCODING	"UTF-8"

/**
 * XML ファイルを1度に読み込むサイズ
 */
#define TFS_READFILE_LEN	512

/*------------------------------------------------------------------------------
  Define structures and enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * XML DOM パーサをコンテキスト構造体 [ 不完全型の定義 ]
 */
struct tfs_xml_parser {

	/* Document ノードオブジェクトへのポインタ */
	tfs_xml_doc *doc;

	/* 発生したエラーステータス */
	tfs_status_t st;

	/* メモリアロケーター */
	tfs_pool_t *pool;

	int allocated_pool;

	/* Expat XML パーサ */
	XML_Parser xp;
	enum XML_Error xp_err;	/* エラー値(for Expat) */
	int xp_errline;

	/*
	 * 以下はパース時作業用. DOMノードの構成とは関係ありません
	 */
	/* 現在解析中のElement へのポインタ */
	tfs_xml_elem *cur_elem;
};

/**
 * XML Document (XMLドキュメントツリーのルート) を表す構造体 [ 不完全型の定義 ]
 */
struct tfs_xml_doc {
	/* root エレメント */
	tfs_xml_elem *root;

	/* このオブジェクトを生成したパーサへのポインタ */
	tfs_xml_parser *parser;
};

/**
 * Text ノードを表す構造体 [ 不完全型の定義 ]
 */
struct tfs_xml_text {
	/* テキストデータを可変長文字列バッファ
	 * (note)
	 *   巨大なテキストデータを扱いたいのなら大きな文字列バッファではなく
	 *   XMLパーサから受け取ったテキストの断片をそのまま持っておく形式
	 *   にした方がいい. そうでないとメモリの再確保が頻発するから. */
	tfs_sbuf_t *sbuf;
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static void _cleanup_parser(void *data);
static void _cleanup_elem(tfs_xml_elem *elem);
static tfs_status_t _parse_feed(tfs_xml_parser *parser,
                                const char *data, tfs_ssize_t len, int isFinal);
static void _start_element(void *userData, const XML_Char *name, const XML_Char **attrs);
static void _end_element(void *userData, const XML_Char *name);
static void _cdata_handler(void *userData, const XML_Char *s, int len);


/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * あるDOM ノードを処理するためのXML パーサの生成.
 *
 */
TFS_DECLARE(void) tfs_xml_parser_create(tfs_pool_t *pool, tfs_xml_parser **parser)
{
	tfs_pool_t *p = pool;
	int allocated_pool = 0;

	if (p == NULL) {
		/* メモリアロケーター生成 */ 
		tfs_pool_create(&p);
		allocated_pool = 1;
	}

	/* XMLパーサコンテキストの生成(コンテキストはmallocで) */
	*parser = malloc(sizeof(tfs_xml_parser));
	(*parser)->st             = TFS_SUCCESS;
	(*parser)->cur_elem       = NULL;
	(*parser)->pool           = p;
	(*parser)->allocated_pool = allocated_pool;

	/* Document を保持する入れ物の作成 */
	(*parser)->doc = tfs_pcalloc(p, sizeof(tfs_xml_doc));
	(*parser)->doc->root   = NULL;
	(*parser)->doc->parser = *parser;

	/* Expat XML パーサインスタンスの生成 */
	(*parser)->xp = XML_ParserCreate(TFS_EXPAT_ENCODING);
	(*parser)->xp_err     = XML_ERROR_NONE;
	(*parser)->xp_errline = 0;

	/* Expat XML パーサの初期化(ハンドラの設定) */
	XML_SetUserData((*parser)->xp, *parser);
	XML_SetElementHandler((*parser)->xp, _start_element, _end_element);
	XML_SetCharacterDataHandler((*parser)->xp, _cdata_handler);

	/* クリーンアップハンドラに登録しておく */
	tfs_pool_cleanup_register(p, *parser, _cleanup_parser);
}

/**
 * 指定されたファイルパスpath のXMLファイルをパースしてDOMノードを構築する.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_file(tfs_pool_t *pool, const char *path,
                                             tfs_xml_parser **parser,
                                             tfs_xml_doc **doc)
{
	int fd;
	char *buf;
	tfs_ssize_t len;
	tfs_status_t rv = TFS_SUCCESS;

	if (IS_EMPTY(path)) {
		return TFS_ENOPARAM;
	}

	/* 既に存在するパーサオブジェクトは利用できません */
	if (parser && *parser) {
		return TFS_EREUSEXMLPARSER;
	}

	*doc = NULL;	/* 初期化 */
	*parser = NULL;

	/* path のファイルを開く */
	if ((fd = open(path, TFS_OFLG_READONLY)) == -1) {
		return TFS_OS_ERRNO(errno);
	}

	/* パーサコンテキストの生成 */
	tfs_xml_parser_create(pool, parser);

	/* ファイルを読みながらパースする */
	len = TFS_READFILE_LEN;
	buf = malloc(len);

	while (1) {
		len = TFS_READFILE_LEN;
		rv = tfs_file_read(fd, buf, &len);
		/* ファイルの終わり */
		if (rv == TFS_FILE_EOF) {
			char end;
			rv = _parse_feed(*parser, &end, 0, 1);
			if (rv != TFS_SUCCESS) {
				goto cleanup_mem;
			}
			break;
		}
		/* 読み込み失敗 */
		else if (rv != TFS_SUCCESS) {
			goto cleanup_mem;
		}

		/* パーサーにデータを渡す */
		rv = _parse_feed(*parser, buf, len, 0);
		if (rv != TFS_SUCCESS) {
			goto cleanup_mem;
		}
	}
	*doc = (*parser)->doc;

cleanup_mem:
	if (buf != NULL) free(buf);
	close(fd);

	return rv;
}

/**
 * サイズlen のデータ data のXMLデータをパースしてDOMノードを構築する.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_data(tfs_pool_t *pool, const char *data,
                                             tfs_size_t len,
                                             tfs_xml_parser **parser,
                                             tfs_xml_doc **doc)
{
	tfs_status_t rv;

	if (data == NULL || len == 0) {
		return TFS_ENOPARAM;
	}

	/* 既に存在するパーサオブジェクトは利用できません */
	if (parser && *parser) {
		return TFS_EREUSEXMLPARSER;
	}

	*doc = NULL;	/* 初期化 */
	*parser = NULL;

	/* パーサコンテキストの生成 */
	tfs_xml_parser_create(pool, parser);

	/* パーサーにデータを渡す */
	rv = _parse_feed(*parser, data, len, 1);
	if (rv != TFS_SUCCESS) {
		return rv;
	}
	*doc = (*parser)->doc;

	return TFS_SUCCESS;
}

/**
 * 組み立て途中のサイズlen のデータ data のXMLデータを XML DOMパーサparser に
 * 渡してパースを行う.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_feed(tfs_xml_parser *parser,
                                             const char *data,
                                             tfs_size_t len)
{
	tfs_status_t rv;

	if (parser == NULL || data == NULL || len == 0) {
		return TFS_ENOPARAM;
	}

	/* パーサーにデータを渡す */
	rv = _parse_feed(parser, data, len, 0);
	if (rv != TFS_SUCCESS) {
		return rv;
	}

	return TFS_SUCCESS;
}

/**
 * tfs_xml_parse_feed で蓄積されたXMLを全てパースしてDOMノードを構築する.
 *
 */
TFS_DECLARE(tfs_status_t) tfs_xml_parse_done(tfs_xml_parser *parser,
                                             tfs_xml_doc **doc)
{
	char end;
	tfs_status_t rv;

	if (parser == NULL) {
		return TFS_ENOPARAM;
	}
	*doc = NULL;	/* 初期化 */

	/* 長さを0にしてパーサーに終了を知らせる */
	rv = _parse_feed(parser, &end, 0, 1);
	if (rv != TFS_SUCCESS) {
		return rv;
	}
	*doc = parser->doc;

	return TFS_SUCCESS;
}

/**
 * XML のパース中に発生したエラーメッセージを取得する.
 *
 */
TFS_DECLARE(char *) tfs_xml_geterror(tfs_xml_parser *parser, tfs_status_t rv)
{
	if (parser == NULL) {
		return NULL;	/* メッセージは取得不能 */
	}

	if (rv == TFS_EPARSEXML && parser->xp_err != XML_ERROR_NONE) {
		const char *msg = XML_ErrorString(parser->xp_err);
		if (IS_FILLED(msg)) {
			return tfs_psprintf(parser->pool, "%s(%d)", msg, parser->xp_errline);
		}
	}

	return NULL;
}

/**
 * Documentノードの元最初のノード(rootノード) を取得する.
 *
 */
TFS_DECLARE(tfs_xml_elem *) tfs_xml_get_rootnode(const tfs_xml_doc *doc)
{
	return (doc != NULL) ? doc->root : NULL;
}

/**
 * XML エレメントelem のTextノード値(CDATAセクションを含む) を取得して返却する.
 *
 */
TFS_DECLARE(char *) tfs_xml_get_cdata(tfs_xml_elem *elem, int strip_white, tfs_pool_t *p)
{
	char *str = NULL;
	tfs_size_t len;
	tfs_pool_t *pool;

	if (elem == NULL || elem->parser == NULL ||
		elem->text == NULL || elem->text->sbuf == NULL) {
		return NULL;
	}

	/* 引数のプールを優先的に使用する */
	pool = (p != NULL) ? p : elem->parser->pool;

	len = tfs_sbuf_getlength(elem->text->sbuf);
	if (len > 0) {
		str = tfs_pstrdup(pool, tfs_sbuf_toString(elem->text->sbuf));

		/* XML White スペース文字の除去 */
		if (strip_white) {
			/* 先頭についているwhite スペースを除去 */
			while (isspace(*str)) {
				++str;
			}
			len = strlen(str);

			/* 
			 * 取得文字列最後から最初までループし終端または空白でなくなる位置を取得
			 * 取得した位置に終端文字を設定
			 */
			while (len-- > 0 && (str[len] == '\0' || isspace(str[len])))
				continue;

			str[len + 1] = '\0';
		}
	}

	return str;
}

/**
 * XML エレメントelem のChild からtagname と一致するエレメントを取得する.
 *
 */
TFS_DECLARE(tfs_xml_elem *) tfs_xml_find_child(tfs_xml_elem *elem,
                                               const char *tagname)
{
	tfs_xml_elem *child = NULL;

	if (elem == NULL || IS_EMPTY(tagname)) {
		return NULL;
	}

	for (child = elem->first_child; child; child = child->next) {
		if (child->name && strcmp(child->name, tagname) == 0) {
			return child;
		}
	}

	return NULL;
}

/**
 * XML DOMパーサparser の破棄.
 *
 */
TFS_DECLARE(void) tfs_xml_parser_destroy(tfs_xml_parser *parser)
{
	if (parser == NULL) {
		return;	/* 何もできない */
	}

	/* クリーンアップハンドラへの登録を解除 */
	tfs_pool_cleanup_kill(parser->pool, parser, _cleanup_parser);

	_cleanup_parser(parser);
}

/**
 * 文字列s をテキストノード値にする際、利用できない文字列をQuoteする.
 *
 */
TFS_DECLARE(char *) tfs_xml_quote_string(tfs_pool_t *pool, const char *s)
{
	char *result = NULL;
	tfs_sbuf_t *sbuf = NULL;
	size_t i, len;

	if (IS_EMPTY(s) || pool == NULL) {
		return NULL;	/* Quote 処理を実施できない */
	}

	len = strlen(s);
	tfs_sbuf_create(NULL, len * 3, &sbuf);

	/* 1文字ずつスキャンして調べる */
	for (i = 0; i < len; i++) {
		switch (s[i]) {
			case '<':
				tfs_sbuf_appendByte(sbuf, 4, "&lt;");
				break;
			case '>':
				tfs_sbuf_appendByte(sbuf, 4, "&gt;");
				break;
			case '&':
				tfs_sbuf_appendByte(sbuf, 5, "&amp;");
				break;
			default:
				tfs_sbuf_appendChar(sbuf, s[i]);
				break;
		}
	}

	result = tfs_pstrdup(pool, tfs_sbuf_toString(sbuf));
	tfs_sbuf_destroy(sbuf);	/* 可変長文字列バッファは不要 */

	return result;	/* メモリ開放は呼び出しもとの責任で */
}


/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
static void _cleanup_parser(void *data)
{
	tfs_xml_parser *parser = data;
	if (parser == NULL) return;

	/* Expat XML パーサの破棄 */
	if (parser->xp) {
		XML_ParserFree(parser->xp);
	}

	/* XMLエレメントを破棄する */
	if (parser->doc != NULL && parser->doc->root != NULL) {
		tfs_xml_elem *elem;
		for (elem = parser->doc->root; elem != NULL; elem = elem->next) {
			_cleanup_elem(elem);
		}
	}

	/* プールを破棄する */
	if (parser->pool != NULL && parser->allocated_pool) {
		tfs_pool_destroy(parser->pool);
		parser->pool = NULL;
	}
	free(parser);
}

static void _cleanup_elem(tfs_xml_elem *elem)
{
	if (elem != NULL) {
		/* Child ノードのエレメントをクリーンアップする */
		_cleanup_elem(elem->first_child);

		/* 自分と同列のエレメントノードをクリーンアップする */
		for (; elem != NULL; elem = elem->next) {
			if (elem->text != NULL) {
				tfs_sbuf_destroy(elem->text->sbuf);
				elem->text->sbuf = NULL;
				elem->text = NULL;
			}
		}
	}
}

static tfs_status_t _parse_feed(tfs_xml_parser *parser,
                                const char *data, tfs_ssize_t len, int isFinal)
{
	XML_STATUS_T xml_rv;

	if (parser == NULL || data == NULL) {
		return TFS_ENOPARAM;
	}

	/* XML パーサにデータを渡す */
	xml_rv = XML_Parse(parser->xp, data, len, isFinal);
	if (xml_rv == XML_STATUS_ERROR) {
		parser->xp_err     = XML_GetErrorCode(parser->xp);
		parser->xp_errline = XML_GetCurrentLineNumber(parser->xp);
		return TFS_EPARSEXML;
	}

	return TFS_SUCCESS;
}

/**
 * 開始タグハンドラ.
 */
static void _start_element(void *userData, const XML_Char *name, const XML_Char **attrs)
{
	tfs_xml_elem *elem, *parent;
	tfs_xml_parser *parser = userData;
	tfs_pool_t *pool;
	tfs_xml_attr *attr = NULL;

	if (parser == NULL || IS_EMPTY(name)) {
		return;		/* 何もできません */
	}
	pool = parser->pool;

	/* XMLエレメントの生成 */
	elem = tfs_pcalloc(pool, sizeof(tfs_xml_elem));
	elem->name        = tfs_pstrdup(pool, name);
	elem->lang        = NULL;
	elem->text        = NULL;
	elem->attr        = NULL;
	elem->next        = NULL;
	elem->first_child = NULL;
	elem->parent      = NULL;
	elem->last_child  = NULL;
	elem->parser      = parser;

	/* 属性の処理 */
	while (*attrs != NULL) {
		if (elem->attr == NULL) {
			elem->attr = attr = tfs_pcalloc(pool, sizeof(tfs_xml_attr));
		}
		else {
			attr->next = tfs_pcalloc(pool, sizeof(tfs_xml_attr));
			attr = attr->next;
		}
		attr->name  = tfs_pstrdup(pool, *attrs++);
		attr->value = tfs_pstrdup(pool, *attrs++);
		attr->next  = NULL;
	}

	parent = parser->cur_elem;	/* _end_element() と協力してcur_elem を必ず親にしている */
	if (parent == NULL) {
		parser->cur_elem = parser->doc->root = elem;
	}
	else {
		elem->parent = parent;
		if (parent->last_child == NULL) {
			parent->first_child = parent->last_child = elem;
		}
		else {
			/* 旧Childにつなげておいて、最後のChildに自身がなる */
			parent->last_child->next = elem;
			parent->last_child = elem;
		}
		/* _end_element() で本当に自分がparent かどうか決定されるので
		 * ここでは気にせずcur_elem を自分にする */
		parser->cur_elem = elem;
	}
}

/**
 * 終了タグハンドラ.
 */
static void _end_element(void *userData, const XML_Char *name)
{
	tfs_xml_parser *parser = userData;

	/* 現在のノードを親ノードにあげる */
	parser->cur_elem = parser->cur_elem->parent;
}

/**
 * CDATA / Textノードハンドラ
 */
static void _cdata_handler(void *userData, const XML_Char *s, int len)
{
	tfs_xml_parser *parser = userData;
	tfs_xml_elem *elem;

	if (parser == NULL || parser->cur_elem == NULL) return;

	elem = parser->cur_elem;
	if (elem->text == NULL) {
		elem->text = tfs_pcalloc(parser->pool, sizeof(tfs_xml_text));
		tfs_sbuf_create(parser->pool, len + 1, &elem->text->sbuf);
	}
	tfs_sbuf_appendByte(elem->text->sbuf, len, s);
}

