/**
 * $Id$
 *
 * TeamFileプラグインSDK用ヘッダファイル(プラグインハンドラの定義)
 * (note)
 *	各プラグインはこのヘッダファイルを必ずincludeして下さい。
 */
#ifndef INCLUDE_TFSP_IHDR_CCIPHER_H
#define INCLUDE_TFSP_IHDR_CCIPHER_H

#include "tfsp_ihdr.h"
#include "tfs_errlog.h"
#include "tfs_pools.h"
#include "tfs_hashset.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure/enum
  ----------------------------------------------------------------------------*/
typedef struct tfsp_ihdr_ccipher    tfsp_ihdr_ccipher;
typedef struct tfsp_ctx_ccipher     tfsp_ctx_ccipher;
typedef struct tfsp_ccipher_file    tfsp_ccipher_file;

/**
 * TeamFile 実体暗号プラグインインターフェース構造体の定義
 */
struct tfsp_ihdr_ccipher {
	/**
	 * TeamFileサーバプラグインベースインターフェースへのポインタ
	 */
	const tfsp_ihdr_base *bs;

	/**
	 * 実体暗号化の準備 インターフェース
	 * [ 役割 ]
	 *   * 実体暗号化に先立ち、暗号化処理の実施に必要なリソースの初期化.
	 *   * 1度初期化したら、「ファイル暗号化の終了」が実行されるまで
	 *     実体暗号化処理を利用可能
	 *
	 * @param p tfs_pool_t * メモリアロケーター. NULL ならば自身で生成する.
	 * @param ctx tfsp_ctx_ccipher * コンテキストへのポインタ
	 *            ctx のアロケートは呼び出し元で実施.
	 *            ctx->child にはcinit の結果を代入.
	 * @param tfs_error_t * エラーステータス (NULL で成功)
	 * 	エラーは最後に捨ててください.
	 */
	tfs_error_t * (*prepare)(tfs_pool_t *p, tfsp_ctx_ccipher *ctx);

	/**
	 * 与えられたファイルの実体暗号化 インターフェース
	 * [ 役割 ]
	 *   * 実体暗号化処理の実施
	 *
	 * @param ctx tfsp_ctx_ccipher * コンテキストへのポインタ
	 * @param file tfsp_ccipher_file *
	 * @return tfs_error_t * エラーステータス (NULL で成功)
	 * 	エラーは最後に捨ててください.
	 */
	tfs_error_t * (*ccipher)(tfsp_ctx_ccipher *ctx, tfsp_ccipher_file *file);

	/**
	 * 実体暗号化対象となるグループIDの一覧を取得するためのインターフェース
	 *
	 * @param ctx tfsp_ctx_ccipher * コンテキストへのポインタ
	 * @param groupid_set tfs_hset_t ** グループIDからなるセット
	 * @return tfs_error_t * エラーステータス (NULL で成功)
	 * 	エラーは最後に捨ててください.
	 */
	tfs_error_t * (*cipheredGroupSet)(tfsp_ctx_ccipher *ctx, tfs_hset_t **groupid_set);

	/**
	 * 実体暗号化対象から指定されたグループIDのグループを除外する.
	 *
	 * @param ctx tfsp_ctx_ccipher * コンテキストへのポインタ
	 * @param groupid const char * グループID
	 * @return tfs_error_t * エラーステータス (NULL で成功)
	 * 	エラーは最後に捨ててください.
	 */
	tfs_error_t * (*excludeTarget)(tfsp_ctx_ccipher *ctx, const char *groupid);

	/**
	 * 実体暗号化の終了処理 インターフェース
	 * [ 役割 ]
	 *   * 実体暗号化に利用されたリソースの開放
	 *
	 * @param ctx tfsp_ctx_ccipher * コンテキストへのポインタ
	 */
	void (*release)(tfsp_ctx_ccipher *ctx);

	/* ダミーメンバ. プラグインは絶対に触らないこと. */
	void *reserved1;
	void *reserved2;
	void *reserved3;
	void *reserved4;
	void *reserved5;
};

/**
 * 実体暗号化処理に利用するコンテキスト構造体の定義
 */
struct tfsp_ctx_ccipher {
	/**
	 * 子プロセスレベルコンテキスト構造体へのポインタ
	 * cinit で生成したものです.
	 */
	const tfsp_ctx_child *child;

	/**
	 * 処理対象となるLocation の名称(Apache から利用する場合のみ)
	 */
	const char *location;

	/**
	 * 任意のデータへのポインタ (see GET_PRIVATE_CTX マクロ)
	 *
	 * プラグインが勝手に使って下さい. それ以外が触ることはありません.
	 */
	void *d;
};

/**
 * 実体暗号化処理を施すファイルに関する情報を保持する構造体の定義
 */
struct tfsp_ccipher_file {

	/* 入力情報 */
	const char *path;			/* 物理ファイルのフルパス */
	const char *name;			/* ファイルの表示名       */
	const char *ctype;			/* ファイルのContent-Type */
	tfsp_type_operation op;		/* オペレーションの種類   */
	tfsp_type_folder srctype;	/* フォルダ種別(src)      */
	tfsp_type_folder dsttype;	/* フォルダ種別(dst)      */
	int keep_inputfile;			/* 入力ファイルをそのまま維持するかどうか */
	const char *groupid;        /* このファイルをアップロードするグループのID */

	/* 出力情報 */
	char *opath;			/* ファイルのフルパス     */
	char *oname;			/* ファイルの表示名       */
	const char *octype;		/* ファイルのContent-Type */
	tfs_int64_t ofsize;		/* ファイルサイズ         */
};

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFSP_IHDR_CCIPHER_H */

