/**
 * $Id$
 *
 * TeamFileプラグインSDK用ヘッダファイル(プラグインローダ)
 * (note)
 *   このヘッダファイルで宣言・定義された変数は、プラグインライブラリを
 *   ローディングする目的で使用されます.
 *   よって、プラグインライブラリ自身がこのヘッダをinclude してはなりません.
 */
#ifndef TFSP_CORE_PRIVATE
# error "Never include <tfsp_loader.h> except TeamFile Core module."
#endif	/* TFSP_CORE_PRIVATE */

#ifndef INCLUDE_TFSP_LOADER_H
#define INCLUDE_TFSP_LOADER_H

#include "tfsp.h"
#include "tfsp_errno.h"
#include "tfsp_ihdr.h"

#include "tfs_errlog.h"
#include "tfs_varray.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * プラグインローダ用コンフィグファイルの名称
 */
#define TFSP_LOADER_CONFFILE	"config.xml"

/**
 * プラグイン用コンフィグファイルの名称
 */
#define TFSP_PLUGIN_CONFFILE	"plugin.xml"

/*------------------------------------------------------------------------------
  Define structures and enum
  ----------------------------------------------------------------------------*/
typedef struct tfsp_plugin_module     tfsp_plugin_module;
typedef struct tfsp_if_version        tfsp_if_version;
typedef struct tfsp_interface_info    tfsp_interface_info;
typedef struct tfsp_author_info       tfsp_author_info;
typedef struct tfsp_plugin_fileinfo	  tfsp_plugin_fileinfo;
typedef struct tfsp_plugin_cpageinfo  tfsp_plugin_cpageinfo;
typedef struct tfsp_plugin_cpage      tfsp_plugin_cpage;
typedef struct tfsp_plugin_state      tfsp_plugin_state;

/**
 * プラグインがサポートするインターフェースの種類を表す列挙型
 */
typedef enum tfsp_if_type	tfsp_if_type;
enum tfsp_if_type {
	TFSP_IFTYPE_UNKNOWN = 0,	/* (未定義) */
	TFSP_IFTYPE_CCIOHER,		/* 実体暗号化 */
	TFSP_IFTYPE_FFILTER,		/* ファイル名フィルタ */
};

/**
 * プラグインの実装を提供するオブジェクトの種類
 */
typedef enum tfsp_if_filetype	tfsp_if_filetype;
enum tfsp_if_filetype {
	TFSP_IFFTYPE_UNKNOWN = 0,	/* 未定義 */
	TFSP_IFFTYPE_SHAREDOBJ,		/* ダイナミックローディング可能な共有ライブラリ */
	TFSP_IFFTYPE_CGISCRIPT,		/* CGIスクリプト */
};

/**
 * プラグインモジュールコンテキスト構造体 [ 不完全型の宣言 ]
 */
typedef struct tfsp_plugin_modulectx  tfsp_plugin_modulectx;

/**
 * 個々のプラグインを表す構造体
 */
struct tfsp_plugin_module {

	char *name;		/* プラグインの名前 */
	char *version;	/* プラグインのバージョン番号 */
	char *summary;	/* プラグインの短い説明文 */
	char *description;	/* プラグインの詳細な説明文 */

	/* 実装インターフェース */
	tfsp_interface_info *if_info;

	/* 作成者情報 */
	tfsp_author_info *author;

	/* ファイル情報 */
	tfsp_plugin_fileinfo *fileinfo;

	/* コンフィグページの情報 */
	tfsp_plugin_cpageinfo *cpageinfo;

	/* 状態 */
	tfsp_plugin_state *state;

	const char *pluginpath;	/* プラグインディレクトリのパス */
	const char *confpath;	/* プラグインコンフィグファイルのパス */

	/* プラグインローダーが専用に利用するコンテキストへのポインタ */
	tfsp_plugin_modulectx *ctx;

	tfsp_plugin_module *next;	/* 関連するプラグインへのポインタ(予約) */
};

/**
 * インターフェースバージョンを表す構造体
 */
struct tfsp_if_version {
	tfs_int32_t current;
	tfs_int32_t revision;
	tfs_int32_t age;
};

/**
 * サポート/実装しているインターフェース情報を表す構造体
 */
struct tfsp_interface_info {
	/* サポートインターフェースの種類 */
	tfsp_if_type itype;

	/* サポートインターフェースのバージョン */
	tfsp_if_version *iversion;
};

/**
 * 作成者情報を表す構造体
 */
struct tfsp_author_info {
	char *vender;
	char *copyright;
	char *homepageurl;
	char *mailaddress;
};

/**
 * プラグインのファイルに関する情報を表す構造体
 */
struct tfsp_plugin_fileinfo {
	/* プラグインファイルのパス */
	char *filepath;

	/* プラグインファイルの種類 */
	tfsp_if_filetype ftype;

	/* プラグインファイルのチェックサム(MD5) */
	char *checksum;
};

/**
 * コンフィグページ全体の情報を表す構造体
 */
struct tfsp_plugin_cpageinfo {
	tfsp_plugin_cpage *g_cpuri;
	tfsp_plugin_cpage *l_cpuri;
};

typedef enum tfsp_screen_type	tfsp_screen_type;
enum tfsp_screen_type {
	TFSP_SCREENTYPE_UNKNOWN = 0,	/* (未定義)       */
	TFSP_SCREENTYPE_FILE,			/* ファイル       */
	TFSP_SCREENTYPE_PERSONAL,		/* 個人設定       */
	TFSP_SCREENTYPE_UTILITY,		/* ユーティリティ */
};

/**
 * あるひとつのコンフィグページ情報を表す構造体
 */
struct tfsp_plugin_cpage {
	char *uri;			/* 設定画面のURI */
	char *iconpath;		/* 表示アイコンのパス名 */
	char *displayname;	/* 画面表示用名称(日本語) */
	tfsp_screen_type screentype;

	int use_admin;	/* 管理者権限で利用できるかどうか */
	int use_normal;	/* 一般ユーザ権限で利用できるかどうか */
	int use_limited;	/* 制限ユーザ権限で利用できるかどうか */
	int use_groupleader;	/* グループリーダ権限で利用できるかどうか */
	char *displayname_en;	/* 画面表示用名称(英語) */
};

/**
 * プラグインの状態を表す構造体
 */
struct tfsp_plugin_state {
	int active;	/* プラグインがアクティブかどうか(1: アクティブ / 0: 非アクティブ) */
};

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * ローディングされたプラグインの状態を管理する構造体 [ 不完全型の宣言 ]
 */
typedef struct tfsp_ctx_loader	tfsp_ctx_loader;

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたディレクトリ絶対パス rootpath 以下にあるプラグインライブラリの
 * ロード/アンロードを行うプラグインローダ(コンテキスト)の生成.
 * rootpath がNULL の場合には、デフォルトのコンフィグパスを使用する.
 *
 * @param rootpath const char * プラグインルートパス
 * @param loader tfsp_ctx_loader ** 生成したプラグインコンテキストへのポインタ
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_create(const char *rootpath, tfsp_ctx_loader **loader);

/**
 * プラグインローダ loader が管理するインストールされた有効なプラグインを
 * 全てローディングして使用可能にする.
 *
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @param plugin_err tfs_error_t ** ローディングした全プラグインのエラー状態.
 *   戻り値がNULLでもここには値が入るかもしれません.
 *   必要であればログに出力するなどの処理を行なってください.
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_loadAll(tfsp_ctx_loader *loader,
                                               tfs_error_t **plugin_err);

/**
 * プラグインディレクトリを再スキャンし、有効なプラグインをロードする.
 *
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_reloadAll(tfsp_ctx_loader *loader);

/**
 * プラグインコンテキスト loader で管理されたプラグインをアンロードして
 * 使用できないようにする.
 *
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @param plugin_err tfs_error_t ** アンロードするプラグインのエラー
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_unloadAll(tfsp_ctx_loader *loader,
                                                 tfs_error_t **plugin_err);

/**
 * プラグインコンテキスト loader で管理されたプラグインの中から
 * タイプ ptype、名前 pname のプラグインをアンロードして使用できないようにする.
 *
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_unload(tfsp_ctx_loader *loader,
                                              tfsp_type_plugin ptype, const char *pname);

/**
 * プラグインのChildレベル初期化処理を実施する.
 *
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @param plugin_err tfs_error_t ** アンロードするプラグインのエラー
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_cinit(tfsp_ctx_loader *loader,
                                                 tfs_error_t **plugin_err);

/**
 * プラグインのChildレベル終了処理を実施する.
 *
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @param plugin_err tfs_error_t ** アンロードするプラグインのエラー
 */
TFS_DECLARE(void) tfsp_loader_cterminate(tfsp_ctx_loader *loader,
                                                 tfs_error_t **plugin_err);

/**
 * 全てのプラグインをアンロードしてプラグインローダコンテキストを破棄する.
 * 本関数呼出し後、loader に触れてはなりません.
 *
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 */
TFS_DECLARE(void) tfsp_loader_destroy(tfsp_ctx_loader *loader);

/**
 * プラグインコンテキスト loader で管理されたプラグインの中から
 * タイプ ptype、名前 pname のプラグイン情報を取得する.
 * pname がNULLであれば、ptype のプラグインインターフェースの内、
 * プラグインチェインの先頭にあるプラグイン情報を取得します.
 * 
 * @param pool tfs_pool_t * メモリアロケータ−. NULL ならばloader のものを利用する
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @param ptype tfsp_type_plugin プラグイン種別
 * @param pname const char * プラグインの名前
 * @param pm ** const tfsp_plugin_module プラグインモジュールへのポインタ
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_lookupIf(tfs_pool_t *pool,
                                                tfsp_ctx_loader *loader,
                                                tfsp_type_plugin ptype, const char *pname,
                                                const tfsp_plugin_module **pm);

/**
 * プラグインコンテキスト loader で管理されたプラグインの中から
 * アクティブなプラグイン情報を取得する.
 * 
 * @param pool tfs_pool_t * メモリアロケータ−. NULL ならばloader のものを利用する
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @param plugins ** tfs_varray_t プラグインモジュールを持つ配列
 * @return tfs_error_t * エラーインスタンスへのポインタ. 使い終わったらdestroyを。
 */
TFS_DECLARE(tfs_error_t *) tfsp_loader_lookupAllIf(tfs_pool_t *pool,
                                                tfsp_ctx_loader *loader,
                                                tfs_varray_t **plugins);

/**
 * プラグインローダ loader のプラグインルートディレクトリパスを取得する.
 *
 * @param pool tfs_pool_t * メモリアロケータ−. NULL ならばloader のものを使う
 * @param loader tfsp_ctx_loader * プラグインローダコンテキストへのポインタ
 * @return const char * プラグインルートディレクトリパス
 */
TFS_DECLARE(const char *) tfsp_loader_get_rootpath(tfs_pool_t *pool,
                                                   tfsp_ctx_loader *loader);

/**
 * プラグイングローバルコンテキストを取得する.
 *
 * @param pm const tfsp_plugin_module *
 * @return const tfsp_ctx_global *
 */
TFS_DECLARE(const tfsp_ctx_global *) tfsp_pm_get_gctx(const tfsp_plugin_module *pm);

/**
 * プラグインChild レベルコンテキストを取得する.
 *
 * @param pm const tfsp_plugin_module *
 * @return const tfsp_ctx_child *
 */
TFS_DECLARE(const tfsp_ctx_child *) tfsp_pm_get_cctx(const tfsp_plugin_module *pm);

/**
 * プラグインモジュールpm が持つインターフェースを取得する.
 *
 * @param pm const tfsp_plugin_module *
 * @return void *
 */
TFS_DECLARE(void *) tfsp_pm_get_interface(const tfsp_plugin_module *pm);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFSP_LOADER_H */


