/**
 * $Id$
 *
 * 汎用ユーティリティ (fileio : ファイル操作関連)
 */
#ifndef INCLUDE_TFS_FILEIO_H
#define INCLUDE_TFS_FILEIO_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_common.h"
#include "tfs_string.h"

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * 特殊なファイルのopen モード
 */
/* OS のデフォルトopen モード (tfs_file_copy に使用可能) */
#define TFS_FLMODE_OS_DEFAULT	(S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH)
/* src ファイルのパーミッションと同じにするためのモード (tfs_file_copy) */
#define TFS_FLMODE_SYNC_SRC	010000

/**
 * open のフラグ定義
 */
/* 新規ファイルをopen する時 */
#define TFS_OFLG_NEWFILE	(O_CREAT|O_WRONLY|O_TRUNC|O_LARGEFILE)

/* 読み込み専用でopen する時 */
#define TFS_OFLG_READONLY	(O_RDONLY|O_LARGEFILE)

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
 * Declare public function
 *----------------------------------------------------------------------------*/
/**
 * 指定されたファイルディスクリプタfd のファイルからサイズlen 分のデータを
 * buf に読み込む.
 *
 * @param fd int ファイルディスクリプタ
 * @param buf char * バッファ. 少なくともlen サイズのバッファは必須.
 * @param *len tfs_ssize_t サイズ(byte). 読み込みサイズもここに返却
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_file_read(int fd, char *buf, tfs_ssize_t *len);

/**
 * ファイルパスpath のファイルを削除する.
 * path のファイルがopenされていても本関数は成功しますし、openされていた
 * ファイルをclose しない限りアクセス可能となります.
 *
 * @param path const char * 削除対象ファイルのフルパス
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_file_remove(const char *path);

/**
 * ファイルパス src が示すファイルをファイルパス dst のパスに移動する.
 * 移動されたファイルのパーミッションはsrc と同じになります.
 * (note)
 *   ファイル名の変更もフルパスの変更も可能です.
 *
 * @param src const char * 移動元ファイルのファイルパス
 * @param dst const char * 移動先ファイルのファイルパス
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_file_move(const char *src, const char *dst);

/**
 * ファイルパス src が示すファイルの内容をdst が示すファイルパスにコピーする.
 *
 * @param src const char * 複製元ファイルのファイルパス
 * @param dst const char * 複製先ファイルのファイルパス
 * @param mode mode_t dst のパーミッション(open モード)
 *             指定可能な値については、stat.h を参照. なお、以下は本ヘッダ内で定義.
 *             TFS_FLMODE_OS_DEFAULT, TFS_FLMODE_SYNC_SRC
 * @return tfs_status_t 処理ステータス
 */
TFS_DECLARE(tfs_status_t) tfs_file_copy(const char *src, const char *dst, mode_t mode);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_FILEIO_H */

