/**
 * $Id$
 *
 * 汎用ユーティリティ (HashMap : ハッシュによるMap インターフェースの実装)
 */
#ifndef INCLUDE_TFS_HASHMAP_H
#define INCLUDE_TFS_HASHMAP_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * ハッシュマップを表すコンテキスト構造体 [ 不完全型 ]
 */
typedef struct tfs_hmap_t	tfs_hmap_t;

/**
 * ハッシュマップ内の要素列挙に使用するイテレータ [ 不完全型 ]
 */
typedef struct tfs_hmap_index_t	tfs_hmap_index_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * ハッシュマップの生成.
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * (Reentrant)
 *   この関数はスレッドセーフです.他の関数も、hmap をスレッド間で共有しない限り
 *   スレッドセーフとなります.
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param hmap tfs_hmap_t ** 生成されたハッシュマップへのポインタ
 */
TFS_DECLARE(void) tfs_hmap_create(tfs_pool_t *pool, tfs_hmap_t **hmap);

/**
 * キー文字列 key の値とvalue の関係をハッシュマップhmap にマップする.
 * (note)
 *   hmap ではvalue のポインタを記録するだけです. コピーはしません.
 *   変更されることを避けたいのであれば呼び出し元にて複製してから
 *   渡して下さい.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 * @param key const char * キー文字列(NOT NULL).
 *                         NULL だと登録しません.
 * @param value const void * 値へのポインタ
 */
TFS_DECLARE(void) tfs_hmap_put(tfs_hmap_t *hmap, const char *key, const void *value);

/**
 * キー文字列 key にマップされた値をハッシュマップhmap から取得する.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 * @param key const char * キー文字列(NOT NULL)
 * @return void * マップされた値へのポインタ. 存在しなければNULLを返却.
 */
TFS_DECLARE(void *) tfs_hmap_get(tfs_hmap_t *hmap, const char *key);

/**
 * value がハッシュマップhmap にマップされているかどうか.
 * (note)
 *   value が指し示すオブジェクトが何であるのか判らないので
 *   この関数はvalue のポインタを単純に比較するだけです.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 * @param value const char * 値へのポインタ
 * @return int 含まれているかどうか(1: 含まれている / 0: 含まれていない)
 */
TFS_DECLARE(int) tfs_hmap_containsvalue(tfs_hmap_t *hmap, const void *value);

/**
 * ハッシュマップhmap からキー文字列 key のマッピングを削除する.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 * @param key const char * キー文字列(NOT NULL)
 * @return void * key にマップされていた(今は削除された)オブジェクトへのポインタ
 */
TFS_DECLARE(void *) tfs_hmap_remove(tfs_hmap_t *hmap, const char *key);

/**
 * ハッシュマップhmap にマップされたマッピングの数を返す.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 * @return tfs_int32_t 数
 */
TFS_DECLARE(tfs_int32_t) tfs_hmap_size(tfs_hmap_t *hmap);

/**
 * ハッシュマップhmap のイテレータを生成して返却する.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 * @return tfs_hmap_index_t * イテレータへのポインタ
 */
TFS_DECLARE(tfs_hmap_index_t *) tfs_hmap_first(tfs_hmap_t *hmap);

/**
 * イテレータiterator が次の要素を指すように移動する.
 *
 * @param iterator tfs_hmap_index_t * イテレータ
 * @return tfs_hmap_index_t * 次のイテレータへのポインタ. NULLで終端を示す
 */
TFS_DECLARE(tfs_hmap_index_t *) tfs_hmap_next(tfs_hmap_index_t *iterator);

/**
 * イテレータiterator からハッシュマップにマップされたキーと値の組み合わせを
 * 取得する.
 * (note)
 *   取得したkey や value の中身を書き換えるとhmap 内部の値も変わって
 *   しまいます. 特にkey に関しては書き換えると tfs_hmap_get() が
 *   正しく動作しなくなります.
 *   書き換えではなく、tfs_hmap_put() で入れ直してください.
 *
 * @param iterator tfs_hmap_index_t * イテレータ
 * @param key const char ** マップされたキー文字列へのポインタ
 * @param value const void ** key と関連付けれた値へのポインタ
 */
TFS_DECLARE(void) tfs_hmap_getvalue(tfs_hmap_index_t *iterator,
                                    const char **key, const void **value);

/**
 * ハッシュマップhmap の要素をクリアする.(マップの解除)
 *
 * ハッシュマップ内に格納されたエレメントオブジェクトに関しては何もしません.
 * 呼び出し元で必要であれば適切な終了処理を行ってからこの関数を呼び出して下さい.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 */
TFS_DECLARE(void) tfs_hmap_clear(tfs_hmap_t *hmap);

/**
 * hmap が表すハッシュマップの破棄.
 *
 * ハッシュマップ内に格納されたエレメントオブジェクトに関しては何もしません.
 * 呼び出し元で必要であれば適切な終了処理を行ってからこの関数を呼び出して下さい.
 *
 * @param hmap tfs_hmap_t * ハッシュマップへのポインタ
 */
TFS_DECLARE(void) tfs_hmap_destroy(tfs_hmap_t *hmap);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_HASHMAP_H */

