/**
 * $Id$
 *
 * 汎用ユーティリティ (LinkedList : 可変長リスト)
 */
#ifndef INCLUDE_TFS_LINKEDLIST_H
#define INCLUDE_TFS_LINKEDLIST_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * 可変長リストを表すコンテキスト構造体 [ 不完全型 ]
 */
typedef struct tfs_linkedlist_t	tfs_linkedlist_t;

/**
 * 可変長リストのイテレータを表す構造体 [ 不完全型 ]
 */
typedef struct tfs_linkedlist_index_t	tfs_linkedlist_index_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 可変長リストを生成する.
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * (Reentrant)
 *   この関数はスレッドセーフです.他の関数も、llist をスレッド間で共有しない限り
 *   スレッドセーフとなります.
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param llist tfs_linkedlist_t ** 生成した可変長リストへのポインタ
 */
TFS_DECLARE(void) tfs_linkedlist_create(tfs_pool_t *pool, tfs_linkedlist_t **llist);

/**
 * 指定された要素element を可変長リストllist の末尾に追加する.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 * @param element void * 追加するオブジェクトへのポインタ
 */
TFS_DECLARE(void) tfs_linkedlist_add(tfs_linkedlist_t *llist, void *element);

/**
 * 可変長リストllist の中の位置index にある要素をelement で置き換える.
 * index が範囲外であれば何もしません.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 * @param index tfs_int32_t 追加位置
 * @param element void * 追加するオブジェクトへのポインタ
 */
TFS_DECLARE(void) tfs_linkedlist_set(tfs_linkedlist_t *llist,
                                     tfs_int32_t index, void *element);

/**
 * 可変長リストllist の位置index にあるオブジェクトを削除し、それを返す.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 * @param index tfs_int32_t 削除位置
 */
TFS_DECLARE(void *) tfs_linkedlist_remove(tfs_linkedlist_t *llist, tfs_int32_t index);

/**
 * llist が示す可変長リストをpool のメモリ領域にコピーする.
 * (note)
 *   可変長リスト に格納された値はコピーされません。ポインタが共用されます.
 *   入れ物がCloneされるだけです.
 *
 * @param pool tfs_pool_t *
 * @param llist tfs_linkedlist_t * コピー元
 * @return tfs_linkedlist_t * 複製した可変長リストへのポインタ
 */
TFS_DECLARE(tfs_linkedlist_t *) tfs_linkedlist_clone(tfs_pool_t *pool, tfs_linkedlist_t *llist);

/**
 * 可変長リストllist の総要素数を取得する.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 */
TFS_DECLARE(tfs_int32_t) tfs_linkedlist_size(tfs_linkedlist_t *llist);

/**
 * 可変長リストllist の位置index にあるオブジェクトを取得する.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 * @param index tfs_int32_t 取得位置
 * @return void * index に格納されていたオブジェクトへのポインタ
 */
TFS_DECLARE(void *) tfs_linkedlist_get(tfs_linkedlist_t *llist, tfs_int32_t index);

/**
 * 可変長リストllist のイテレータを生成して返却する.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 * @return tfs_linkedlist_index_t * 取得したイテレータへのポインタ
 */
TFS_DECLARE(tfs_linkedlist_index_t *) tfs_linkedlist_first(tfs_linkedlist_t *llist);

/**
 * イテレータiterator が次の要素を指すように移動する.
 *
 * @param iterator tfs_linkedlist_index_t * イテレータ
 * @return tfs_linkedlist_index_t * 次のイテレータへのポインタ. NULLで終端を示す
 */
TFS_DECLARE(tfs_linkedlist_index_t *) tfs_linkedlist_next(tfs_linkedlist_index_t *iterator);

/**
 * イテレータの値を取得する.
 *
 * @param iterator tfs_linkedlist_index_t * イテレータ
 * @return void * 取得した値j
 */
TFS_DECLARE(void *) tfs_linkedlist_getvalue(tfs_linkedlist_index_t *iterator);

/**
 * 可変長リストllist の要素を全てクリアする.
 *
 * 可変長リスト内に格納されたエレメントオブジェクトに関しては何もしません.
 * 呼び出し元で必要であれば適切な終了処理を行ってからこの関数を呼び出して下さい.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 */
TFS_DECLARE(void) tfs_linkedlist_clear(tfs_linkedlist_t *llist);

/**
 * 可変長リストllist がアロケートしていた領域を破棄する.
 *
 * 可変リスト内に格納されたエレメントオブジェクトに関しては何もしません.
 * 呼び出し元で必要であれば適切な終了処理を行ってからこの関数を呼び出して下さい.
 *
 * @param llist tfs_linkedlist_t * 可変長リストへのポインタ
 */
TFS_DECLARE(void) tfs_linkedlist_destroy(tfs_linkedlist_t *llist);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_LINKEDLIST_H */

