/**
 * $Id$
 *
 * 汎用ユーティリティ (Allocated Memory Cache : アロケート済みメモリ領域のキャッシュ)
 * (note)
 *   再利用可能なメモリキャッシュの実装ではありません.
 *   単に、malloc によってヒープに割り当てられたオブジェクトへのポインタを
 *   複数記憶しておき、不要になった全てfreeするために利用して下さい.
 *   それ以上の機能や役割は与えられていません.
 */
#ifndef INCLUDE_TFS_MCACHE_H
#define INCLUDE_TFS_MCACHE_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * アロケート済みメモリ領域保持する構造体 [ 不完全型の宣言 ]
 */
typedef struct tfs_mcache_t	tfs_mcache_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * アロケート済みメモリキャッシュの生成.
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param size tfs_int32_t 初期キャッシュ個数 (-1 を入れたら自動割り当てします)
 * @param mcache tfs_mcache_t ** 生成したアロケート済みメモリキャッシュへのポインタ
 */
TFS_DECLARE(void) tfs_mcache_create(tfs_pool_t *pool,
                                    tfs_int32_t size, tfs_mcache_t **mcache);

/**
 * アロケートしたオブジェクトへのポインタをmcache に登録する.
 * (note)
 *   obj の領域が開放されていた場合、tfs_mcache_cleanup で問題を起こします.
 *
 * @param mcache tfs_mcache_t * アロケート済みメモリキャッシュへのポインタ
 * @param obj void * 外部でアロケートされたオブジェクトへのポインタ
 */
TFS_DECLARE(void) tfs_mcache_add(tfs_mcache_t *mcache, void *obj);

/**
 * アロケートしたオブジェクトへのポインタobj と一致するポインタを
 * mcache から消去する.
 * 一致するオブジェクトがなければ何もしません.
 *
 * @param mcache tfs_mcache_t * アロケート済みメモリキャッシュへのポインタ
 * @param obj void * 外部でアロケートされたオブジェクトへのポインタ
 */
TFS_DECLARE(void) tfs_mcache_remove(tfs_mcache_t *mcache, void *obj);

/**
 * アロケート済みメモリキャッシュに登録されているオブジェクトの数を返す.
 *
 * @param mcache tfs_mcache_t * アロケート済みメモリキャッシュへのポインタ
 * @return tfs_int32_t キャッシュされたオブジェクト数
 */
TFS_DECLARE(tfs_int32_t) tfs_mcache_size(tfs_mcache_t *mcache);

/**
 * アロケート済みメモリキャッシュに登録されたオブジェクトの
 * ヒープ領域を全て開放(free)する.
 *
 * @param mcache tfs_mcache_t * アロケート済みメモリキャッシュへのポインタ
 */
TFS_DECLARE(void) tfs_mcache_cleanup(tfs_mcache_t *mcache);

/**
 * アロケート済みメモリキャッシュを破棄する.
 * クリーンアップされていないオブジェクトがあれば、領域の開放を行ってから
 * mcache を破棄します.
 *
 * @param mcache tfs_mcache_t * アロケート済みメモリキャッシュへのポインタ
 */
TFS_DECLARE(void) tfs_mcache_destroy(tfs_mcache_t *mcache);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_MCACHE_H */

