/**
 * $Id$
 *
 * 汎用ユーティリティ (Memory Allocator : 汎用ユーティリティ用メモリアロケーター)
 *
 * (note1)
 *   メモリアロケーションが必要な場合の殆どは (1) malloc で確保
 *   (2) メモリに対して書き込み (3) 不要になったメモリ領域をfree
 *   という一連の流れになります. このとき(2) は各処理固有で汎用化は困難ですが、
 *   (1)のポインタを覚えておいて、(3) の時にそのポインタをfreeすることは容易です.
 *   GC のような汎用機構は難しいですが、この程度の支援でもポインタをどこかに
 *   キャッシュしておくという煩わしさが軽減され、かなり楽になるでしょう.
 *   このMemory Allocator はこれを実現するための機構を持っています.
 *   apr_pool_t の思想をまねています.
 *
 * (note2)
 *   メモリアロケーターはクリアされたメモリ領域を再利用するかもしれません.
 *   ですので、ポインタ値をキャッシュしておいてオブジェクトの同一性を判断する手の
 *   プログラムは要注意です. クリアされたら不定値になるばかりでなく、別の
 *   オブジェクトへのポインタとして再利用されるかもしれないからです.
 */
#ifndef INCLUDE_TFS_POOLS_H
#define INCLUDE_TFS_POOLS_H

/**
 * (note)
 *   tfs_pools.h はtfs.hまたはtfs_errno.h 以外をinclude してはなりません.
 *   このルールを破ると、ヘッダの依存関係がおかしくなります.
 */
#include "tfs.h"
#include "tfs_errno.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * メモリアロケーターを表す構造体 [ 不完全型の宣言 ]
 */
typedef struct tfs_pool_t	tfs_pool_t;

/*------------------------------------------------------------------------------
  Declare type of functions
  ----------------------------------------------------------------------------*/
/**
 * クリーンアップハンドラとして定義可能な関数型の定義
 *
 * @param void * クリーンアップハンドラが受け取る任意のオブジェクトへのポインタ
 */
typedef TFS_DECLARE(void) (*tfs_pool_cleanup_t)(void *);

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * メモリアロケーターの生成.
 * (note)
 *   以降、tfs_pool_release() やtfs_pool_destroy() されない限り、メモリアロケーター
 *   から割り当てられたメモリ領域はOSに返却されなくなります.
 *   また、このメモリアロケーターから割り当てられたメモリは呼び出し元で
 *   勝手にfree() してはなりません. 確実にCoreDumpします.
 *
 * @param pool tfs_pool_t ** 生成したメモリアロケーターへのポインタ
 */
TFS_DECLARE(void) tfs_pool_create(tfs_pool_t **pool);

/**
 * メモリアロケーターによって割り当てられていたポインタを全て
 * リセットして再利用可能な状態にする.
 *
 * [ tfs_pool_release() との違い ]
 *   OSにメモリを返すかどうかです.malloc() 関数はかなりコストの高い
 *   高精度なアルゴリズムを用いているので、tfs_pool_clear() して
 *   再利用する方がより高速になるでしょう.
 *
 * (note)
 *   現実装では、tfs_pool_release() と同じ動きになっています.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 */
TFS_DECLARE(void) tfs_pool_clear(tfs_pool_t *pool);

/**
 * メモリアロケーターによって割り当てられたメモリ領域を開放してOSに返す.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 */
TFS_DECLARE(void) tfs_pool_release(tfs_pool_t *pool);

/**
 * メモリアロケーターによって割り当てられたメモリの開放と自身の破棄.
 * (note)
 *   この関数呼出し後、pool に触れてはなりません.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 */
TFS_DECLARE(void) tfs_pool_destroy(tfs_pool_t *pool);

/**
 * メモリアロケーターpool からサイズsize バイトのメモリ領域を取得する.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param size tfs_size_t 割り当てるメモリサイズ
 */
TFS_DECLARE(void *) tfs_palloc(tfs_pool_t *pool, tfs_size_t size);

/**
 * メモリアロケーターpool からサイズsize バイトのメモリ領域を取得して
 * 取得した領域を0クリアする.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param size tfs_size_t 割り当てるメモリサイズ
 */
TFS_DECLARE(void *) tfs_pcalloc(tfs_pool_t *pool, tfs_size_t size);

/**
 * pool が示すメモリアロケーターがクリアまたは破棄される直前にコールされる
 * クリーンアップハンドラ関数を登録する.
 * (note)
 *   クリーンアップハンドラは、FILO でコールされます.(登録の逆順にコール)
 *   また、同じdata を持ち、同じクリーンアップハンドラ関数の登録は
 *   内部的にキャンセルします.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param data const void * クリーンアップハンドラに渡される引数へのポインタ
 * @param cleanup_fn tfs_pool_cleanup_t クリーンアップハンドラ関数へのポインタ
 */
TFS_DECLARE(void) tfs_pool_cleanup_register(tfs_pool_t *pool, const void *data,
                                            tfs_pool_cleanup_t cleanup_fn);

/**
 * 引数data とcleanup_f が示すクリーンアップハンドラの登録をキャンセルし、
 * 呼び出されないようにする.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param data const void * クリーンアップハンドラに渡される引数へのポインタ
 * @param cleanup_fn tfs_pool_cleanup_t クリーンアップハンドラ関数へのポインタ
 */
TFS_DECLARE(void) tfs_pool_cleanup_kill(tfs_pool_t *pool, const void *data,
                                        tfs_pool_cleanup_t cleanup_fn);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_POOLS_H */

