/**
 * $Id$
 *
 * 汎用ユーティリティ (string : 文字列操作関連)
 */
#ifndef INCLUDE_TFS_STRING_H
#define INCLUDE_TFS_STRING_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#if HAVE_STDARG_H
#include <stdarg.h>
#endif
#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
#ifndef IS_EMPTY
/**
 * NULL または空文字であるかどうかを判定するマクロ
 *
 * @param s char * / const char * 文字列へのポインタ
 */
#define IS_EMPTY(s) (s == NULL || *(s) == '\0')
#endif	/* IS_EMPTY */

#ifndef IS_FILLED
/**
 * 文字列s が充填されている(NULLではない && 空ではない)ことを判定するマクロ
 * (note)
 * 	!IS_EMPTY(s) と同じ意味になります。
 * 	コード上の判りやすさを狙って別名にしているだけです。
 *
 * @param s char * / const char * 文字列へのポインタ
 */
#define IS_FILLED(s) (s != NULL && *(s) != '\0')
#endif	/* IS_FILLED */

/*------------------------------------------------------------------------------
  Declare structure/enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure/enum
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare public functions 1 (メモリアロケーター版文字列ユーティリティ)
  ----------------------------------------------------------------------------*/
/**
 * s が示す文字列のメモリ領域をpoolに確保して内容をコピーする. (メモリアロケーター版)
 *
 * 複製されたデータは必ずヌル終端(\0)となります.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param s const char * ソース文字列
 * @return char * 複製された文字列へのポインタ
 *                領域はメモリアロケーターから割り当てられます. free() 禁止.
 */
TFS_DECLARE(char *) tfs_pstrdup(tfs_pool_t *pool, const char *s);

/**
 * s が示す文字列のうち、n文字分を複製してヌル終端で返却. (メモリアロケーター版)
 * s の長さよりもn の方が大きい場合には、tfs_pstrdup() と同じ動作となります.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param s const char * ソース文字列
 * @param n tfs_size_t 複製する文字の個数(ヌル終端はカウントしません)
 * @return char * 複製した文字列へのポインタ
 *                n == 0やs == NULL の場合には複製されません. free() 禁止.
 */
TFS_DECLARE(char *) tfs_pstrndup(tfs_pool_t *pool, const char *s, tfs_size_t n);

/**
 * 指定された複数の文字列を連結する. (メモリアロケーター版)
 * (note)
 *   連結の終端には必ず NULL を指定して下さい.
 *   (例)
 *       char *s = tfs_pstrcat("abc", "123", "xxxx", NULL);
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param s0 const char * 先頭の文字列. これがNULLだと連結処理はしません.
 * @param ... (可変長パラメータ) 必ず文字列でなければなりません.
 *                               そうでない場合には暴走します.
 * @return char * 連結した文字列へのポインタ
 */
TFS_DECLARE(char *) tfs_pstrcat(tfs_pool_t *pool, const char *s0, ...);

/**
 * 指定された複数の文字列をn文字分だけ連結する.(メモリアロケーター版)
 *
 * 総連結長さよりも n の方が大きい場合には、tfs_pstrcat() と同じ動作となります.
 * 総連結長さよりも n の方が小さい場合には、先頭からn文字分だけ複製します.
 * 返却されるデータの終端は必ずNULL終端となります.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param n tfs_size_t 総連結文字列の文字数
 * @param s0 const char * 先頭の文字列. これがNULLだと連結処理はしません.
 * @return char * 連結した文字列へのポインタ
 */
TFS_DECLARE(char *) tfs_pstrncat(tfs_pool_t *pool, tfs_size_t n, const char *s0, ...);

/**
 * printf-style の書式指定文字を利用できる文字列関数.
 * 指定されたフォーマットの文字列を合成して返却します.
 *
 * @param pool tfs_pool_t * メモリアロケーターへのポインタ
 * @param fmt const char * 書式指定文字列
 * @param ... 可変長文字列 
 * @return char * 組み立てられた文字列へのポインタ
 */
TFS_DECLARE(char *) tfs_psprintf(tfs_pool_t *pool, const char *fmt, ...)
		__attribute__((format(printf,2,3)));

/**
 * 多重スラッシュをシングルスラッシュにする.
 *
 * @param pool tfs_pool_t *
 * @param str const char *
 * @return char * 変換後の文字列
 */
TFS_DECLARE(char *) tfs_no2slash(tfs_pool_t *pool, const char *str);

/**
 * ファイルパスを組み立てて返却する.
 *
 * @param pool tfs_pool_t *
 * @param root const char * ルートパス
 * @param f const char * 次に結合するパス
 * @param ... 可変長パラメータ. 終端は必ずNULL にして下さい.
 * @return char * 組み立てたパス名
 */
TFS_DECLARE(char *) tfs_make_full_path(tfs_pool_t *pool, const char *root,
                                       const char *f, ...);

/*------------------------------------------------------------------------------
  Declare public functions 2 (メモリアロケーターを持たない文字列ユーティリティ)
  ----------------------------------------------------------------------------*/
/**
 * s が示す文字列のメモリ領域をヒープに確保し、内容をコピーする.
 *
 * 複製されたデータは必ずヌル終端(\0)となります.
 * 本関数で複製された文字列データは使い終わったら必ずfree()して下さい.
 *
 * @param s const char * ソース文字列
 * @return char * 複製された文字列へのポインタ
 *                s がNULLであれば複製されません.
 */
TFS_DECLARE(char *) tfs_strdup(const char *s);

/**
 * s が示す文字列のうち、n文字分を複製してヌル終端で返却.
 * s の長さよりもn の方が大きい場合には、tfs_strdup() と同じ動作となります.
 * s の長さよりもn の方が小さい場合、n文字分だけ切り取って複製します.
 *
 * @param s const char * ソース文字列
 * @param n tfs_size_t 複製する文字の個数(ヌル終端はカウントしません)
 * @return char * 複製した文字列へのポインタ
 *                n == 0やs == NULL の場合には複製されません.
 */
TFS_DECLARE(char *) tfs_strndup(const char *s, tfs_size_t n);

/**
 * 指定された複数の文字列を連結する.
 *
 * 連結された文字列はヒープにアロケートされます.
 * 本関数で複製された文字列データは使い終わったら必ずfree()して下さい.
 * (note)
 *   連結の終端には必ず NULL を指定して下さい.
 *   (例)
 *       char *s = tfs_strcat("abc", "123", "xxxx", NULL);
 *
 * @param s0 const char * 先頭の文字列. これがNULLだと連結処理はしません.
 * @param ... (可変長パラメータ) 必ず文字列でなければなりません.
 *                               そうでない場合には暴走します.
 * @return char * 連結した文字列へのポインタ
 */
TFS_DECLARE(char *) tfs_strcat(const char *s0, ...);

/**
 * 指定された複数の文字列をn文字分だけ連結する.
 *
 * 総連結長さよりも n の方が大きい場合には、tfs_strcat() と同じ動作となります.
 * 総連結長さよりも n の方が小さい場合には、先頭からn文字分だけ複製します.
 * 返却されるデータの終端は必ずNULL終端となります.
 * 本関数で複製された文字列データは使い終わったら必ずfree()して下さい.
 *
 * @param n tfs_size_t 総連結文字列の文字数
 * @param s0 const char * 先頭の文字列. これがNULLだと連結処理はしません.
 * @return char * 連結した文字列へのポインタ
 */
TFS_DECLARE(char *) tfs_strncat(tfs_size_t n, const char *s0, ...);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_STRING_H */

