/**
 * $Id$
 *
 * 汎用ユーティリティ (Array : 可変長配列コンテナ)
 */
#ifndef INCLUDE_TFS_VARRAY_H
#define INCLUDE_TFS_VARRAY_H

#include "tfs.h"
#include "tfs_errno.h"
#include "tfs_pools.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * 可変長配列を表すコンテキスト構造体 [ 不完全型 ]
 */
typedef struct tfs_varray_t	tfs_varray_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 指定された初期要素数size 個の要素をもつポインタ保持用可変長配列を生成する.
 * size は初期サイズです。要素数が足りなくなれば自動的に追加されます。
 * (note)
 *   pool が指定された場合、pool がメモリアロケーターとなります。
 *   この関数で生成されたインスタンスをdestroyしなくても、pool が破棄された時に
 *   自動的に破棄されます.
 * (Reentrant)
 *   この関数はスレッドセーフです.他の関数も、varray をスレッド間で共有しない限り
 *   スレッドセーフとなります.
 *
 * @param pool tfs_pool_t * メモリプールへのポインタ
 * @param size tfs_int32_t 初期要素数
 * @param varray tfs_varray_t ** 生成した可変長配列へのポインタ
 */
TFS_DECLARE(void) tfs_varray_create(tfs_pool_t *pool,
                                    tfs_int32_t size, tfs_varray_t **varray);

/**
 * 可変長配列varray の末尾にelement が示すオブジェクトを追加する.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 * @param element void * 追加するオブジェクトへのポインタ
 */
TFS_DECLARE(void) tfs_varray_add(tfs_varray_t *varray, void *element);

/**
 * 可変長配列varray の位置index のオブジェクトをelement で置き換える.
 * index が範囲外であれば何もしません.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 * @param index tfs_int32_t 追加位置
 * @param element void * 追加するオブジェクトへのポインタ
 */
TFS_DECLARE(void) tfs_varray_set(tfs_varray_t *varray,
                                 tfs_int32_t index, void *element);

/**
 * 可変長配列varray の位置index にあるオブジェクトを削除し、それを返す.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 * @param index tfs_int32_t 削除位置
 * @return void * 削除位置に格納されていたオブジェクトへのポインタ
 */
TFS_DECLARE(void *) tfs_varray_remove(tfs_varray_t *varray, tfs_int32_t index);

/**
 * 可変長配列varray の総要素数を取得する.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 * @return tfs_int32_t 総要素数
 */
TFS_DECLARE(tfs_int32_t) tfs_varray_size(tfs_varray_t *varray);

/**
 * 可変長配列varray の中の位置index にある要素を取得する.
 * remove と異なり、index 位置の要素は除去されません.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 * @param index tfs_int32_t 取得位置(0から始まります)
 * @return void * indexに格納されていたオブジェクトへのポインタ
 */
TFS_DECLARE(void *) tfs_varray_get(tfs_varray_t *varray, tfs_int32_t index);

/**
 * 可変長配列varray の要素を全てクリアする.
 *
 * 可変長配列内に格納されたエレメントオブジェクトに関しては何もしません.
 * 呼び出し元で必要であれば適切な終了処理を行ってからこの関数を呼び出して下さい.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 * @param index tfs_int32_t 取得位置(0から始まります)
 * @return void * indexに格納されていたオブジェクトへのポインタ
 */
TFS_DECLARE(void) tfs_varray_clear(tfs_varray_t *varray);

/**
 * 可変長配列varray の要素全てを取り出す.
 * (note)
 * 	取り出した要素は、tfs_varray_t に対する追加、削除オペレーションが
 * 	行われると不定値となります。追加、削除オペレーションが実施される
 * 	度に、この関数を使って再取得しなければなりません.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 * @return void ** 可変長配列先頭へのポインタ
 */
TFS_DECLARE(void **) tfs_varray_getelements(tfs_varray_t *varray);

/**
 * varray が示す可変長配列をpool のメモリ領域にコピーする.
 * (note)
 *   可変長配列に格納された値はコピーされません。ポインタが共用されます.
 *   入れ物がCloneされるだけです.
 *
 * @param pool tfs_pool_t *
 * @param varray tfs_varray_t * コピー元
 * @return tfs_varray_t * 複製した可変長リストへのポインタ
 */
TFS_DECLARE(tfs_varray_t *) tfs_varray_clone(tfs_pool_t *pool, tfs_varray_t *varray);

/**
 * 可変長配列varray がアロケートしていた領域を破棄する.
 *
 * @param varray tfs_varray_t * 可変長配列へのポインタ
 */
TFS_DECLARE(void) tfs_varray_destroy(tfs_varray_t *varray);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TFS_VARRAY_H */

