/* 
   Authentication session
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>
   Copyright (C) 1999-2002, Joe Orton <joe@manyfish.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavAuthSession.h 132 2005-06-24 09:09:43Z komat $
*/

#ifndef ONION_HEADER_DAVAUTHSESSION_H
#define ONION_HEADER_DAVAUTHSESSION_H 1

#include <onion/OnionCfg.h>

/* NTLM auth header format copied from the code Daniel Stenberg submitted
 *  to neon mailing list
 * <http://mailman.webdav.org/pipermail/neon/2003-September/001413.html>
 */

/* Flag bits definitions based on http://davenport.sourceforge.net/ntlm.html */

#define NTLMFLAG_NEGOTIATE_UNICODE               (1<<0)
/* Indicates that Unicode strings are supported for use in security buffer
   data. */

#define NTLMFLAG_NEGOTIATE_OEM                   (1<<1)
/* Indicates that OEM strings are supported for use in security buffer data. */

#define NTLMFLAG_REQUEST_TARGET                  (1<<2)
/* Requests that the server's authentication realm be included in the Type 2
   message. */

/* unknown (1<<3) */
#define NTLMFLAG_NEGOTIATE_SIGN                  (1<<4)
/* Specifies that authenticated communication between the client and server
   should carry a digital signature (message integrity). */

#define NTLMFLAG_NEGOTIATE_SEAL                  (1<<5)
/* Specifies that authenticated communication between the client and server
   should be encrypted (message confidentiality). */

#define NTLMFLAG_NEGOTIATE_DATAGRAM_STYLE        (1<<6)
/* unknown purpose */

#define NTLMFLAG_NEGOTIATE_LM_KEY                (1<<7)
/* Indicates that the LAN Manager session key should be used for signing and
   sealing authenticated communications. */

#define NTLMFLAG_NEGOTIATE_NETWARE               (1<<8)
/* unknown purpose */

#define NTLMFLAG_NEGOTIATE_NTLM_KEY              (1<<9)
/* Indicates that NTLM authentication is being used. */

/* unknown (1<<10) */
/* unknown (1<<11) */

#define NTLMFLAG_NEGOTIATE_DOMAIN_SUPPLIED       (1<<12)
/* Sent by the client in the Type 1 message to indicate that a desired
   authentication realm is included in the message. */

#define NTLMFLAG_NEGOTIATE_WORKSTATION_SUPPLIED  (1<<13)
/* Sent by the client in the Type 1 message to indicate that the client
   workstation's name is included in the message. */

#define NTLMFLAG_NEGOTIATE_LOCAL_CALL            (1<<14)
/* Sent by the server to indicate that the server and client are on the same
   machine. Implies that the client may use a pre-established local security
   context rather than responding to the challenge. */

#define NTLMFLAG_NEGOTIATE_ALWAYS_SIGN           (1<<15)
/* Indicates that authenticated communication between the client and server
   should be signed with a "dummy" signature. */

#define NTLMFLAG_TARGET_TYPE_DOMAIN              (1<<16)
/* Sent by the server in the Type 2 message to indicate that the target
   authentication realm is a domain. */

#define NTLMFLAG_TARGET_TYPE_SERVER              (1<<17)
/* Sent by the server in the Type 2 message to indicate that the target
   authentication realm is a server. */

#define NTLMFLAG_TARGET_TYPE_SHARE               (1<<18)
/* Sent by the server in the Type 2 message to indicate that the target
   authentication realm is a share. Presumably, this is for share-level
   authentication. Usage is unclear. */

#define NTLMFLAG_NEGOTIATE_NTLM2_KEY             (1<<19)
/* Indicates that the NTLM2 signing and sealing scheme should be used for
   protecting authenticated communications. */

#define NTLMFLAG_REQUEST_INIT_RESPONSE           (1<<20)
/* unknown purpose */

#define NTLMFLAG_REQUEST_ACCEPT_RESPONSE         (1<<21)
/* unknown purpose */

#define NTLMFLAG_REQUEST_NONNT_SESSION_KEY       (1<<22)
/* unknown purpose */

#define NTLMFLAG_NEGOTIATE_TARGET_INFO           (1<<23)
/* Sent by the server in the Type 2 message to indicate that it is including a
   Target Information block in the message. */

/* unknown (1<24) */
/* unknown (1<25) */
/* unknown (1<26) */
/* unknown (1<27) */
/* unknown (1<28) */

#define NTLMFLAG_NEGOTIATE_128                   (1<<29)
/* Indicates that 128-bit encryption is supported. */

#define NTLMFLAG_NEGOTIATE_KEY_EXCHANGE          (1<<30)
/* unknown purpose */

#define NTLMFLAG_NEGOTIATE_56                    (1<<31)
/* Indicates that 56-bit encryption is supported. */




//forward declarations
class CDavAuthManager;
class CDavRequest;
/** <code>CDavAuthSession</code> manages all authentication activities. 
 * Each <code>CDavWorkSession</code> will create 2 auth sessions upon 
 * construction, one  for server auth and the other for proxy auth. 
 */
OI_BEGIN_CLASSEXPORT

class OI_EXPORT CDavAuthSession
{
 public:
  CDavAuthSession(CDavAuthManager* pMgr,
		  const char* pszRequestHeader,
		  const char* pszResponseHeader,
		  const char* pszResponseInfoHeader,
		  unsigned int unStatusCode,
		  OI_AUTH_CLASS enuClass);
  virtual ~CDavAuthSession();
 private:
  CDavAuthManager* m_pAuthManager;
  const char* m_pszRequestHeader;
  const char* m_pszResponseHeader;
  const char* m_pszResponseInfoHeader;
  OI_STRING_A m_strAuthInfoHeader;
  OI_STRING_A m_strDigest;
  OI_STRING_A m_strStoredDigest;
  OI_STRING_A m_strOpaque;
  OI_STRING_A m_strBasic;
  OI_STRING_A m_strNonce;
  OI_STRING_A m_strCNonce;
  OI_STRING_A m_strRealm;
  OI_STRING_A m_strPasswd;
  OI_STRING_A m_strUsername;
  OI_STRING_A m_strHA1;
  OI_STRING_A m_strNTLM;
  OI_STRING_A m_strNTLMDomain;
  OI_STRING_A m_strNTLMHost;
  unsigned char m_chNTLMNonce[8];
  int m_nStatusCode;
  unsigned int m_unNonceCount;
  unsigned int m_unAttempt;

  bool m_bGotAuthHeader;	//got Auth Header field
  bool m_bGotAuthInfoHeader;	//got Auth Info Header field
  bool m_bGotQop;		//got Auth QOP field
  bool m_bGotAlgorithm;		//got Algorithm field
  bool m_bGotStale;		//got Stale field
  bool m_bGotOpaque;		//got Qpaque field
  bool m_bGotNonce;		//got Nonce field
  bool m_bGotRealm;		//got Realm field
  bool m_bGotScheme;		//got Scheme field
  bool m_bStale;		//session needs refresh
  OI_AUTH_CLASS m_enuClass;
  OI_AUTH_SCHEME m_enuScheme;
  OI_AUTH_STATE m_enuState;
  OI_AUTH_QOP m_enuQop;
 public:
  const char* GetUsername();
  const char* GetPasswd();
  const char* GetAuthHeaderName();
  const char* GetAuthInfoHeaderName();
  OI_AUTH_SCHEME GetAuthScheme();
  OI_AUTH_STATE GetAuthState();
  void SetUsername(const char* pszUsername);
  void SetPasswd(const char* pszPasswd);
  void Reset();
  void SetAuthInfoHeader(const char* pszValue);
  OI_RESULT ParseAuthHeader(const char* pszValue);
  virtual OI_RESULT OnCreateRequest(CDavRequest* pReq);
  virtual OI_RESULT OnPreSendRequest(CDavRequest* pReq);
  virtual OI_RESULT QueryEndRequest(CDavRequest* pReq);
 private:
  OI_RESULT challenge();
  OI_RESULT basicChallenge();
  OI_RESULT digestChallenge();
  OI_RESULT ntlmChallenge();
  OI_RESULT verifyResponse(CDavRequest* pReq, 
			   const char* pszValue);
  bool requestDigest(CDavRequest* pReq);
  bool requestCredential();
  void updateCNonce();
};

OI_END_CLASSEXPORT

#endif /*ONION_HEADER_DAVAUTHSESSION_H*/
