#!/bin/sh
#
# $Id$
#-------------------------------------------------------------------------------
# 自動インポートツール
#
# Copyright (c) BAYBITS LLC All rights reserved.
#-------------------------------------------------------------------------------
# Usage: ./autoimp.sh -d (import 先データベース名) -f (import ファイル名)
#                     -h (ホスト名) -p (ポート番号) -U (PostgreSQLユーザ名)
#                     -L (変更後のロケーション名) -s
#-------------------------------------------------------------------------------

#------------------
# set env
#------------------
if test -d "${TF_HOME}"; then
	MT_FUNCTION_HOME="${TF_HOME}/www/bin"
else
	MT_FUNCTION_HOME="/usr/local/teamfile/www/bin"
fi

MT_FUNCTION="${MT_FUNCTION_HOME}/mt-lib"
if test ! -f ${MT_FUNCTION}; then
	echo "The configuration script \"${MT_FUNCTION}\" was missing. Please check it."
	exit 1
fi

# read functions and configuration
. ${MT_FUNCTION}

#
EXCLUSIVE_LOCKFILE="${TMP_BASE_DIR}/autoimp.lock"

#-------------------------
# for PostgreSQL
#-------------------------
DROBDB="${PG_HOME}/bin/dropdb"
CREATEDB="${PG_HOME}/bin/createdb"
RESTORE="${PG_HOME}/bin/pg_restore"

PGCLIENTENCODING="UTF-8"
PGDATABASE=
LD_LIBRARY_PATH="${PG_HOME}/lib:$LD_LIBRARY_PATH"

export PGCLIENTENCODING PGDATABASE LD_LIBRARY_PATH

#---------------
# set trap
#---------------
trap : HUP
trap "exitscript 0" INT
trap "exitscript 0" QUIT
trap "exitscript 0" TERM
trap "exitscript 0" KILL

#-----------------------------------------------------------------------
# exit script
#-----------------------------------------------------------------------
exitscript() {
	unlockscript "${EXCLUSIVE_LOCKFILE}"
	exit $1
}

#-----------------------------------------------------------------------
# show usage
#-----------------------------------------------------------------------
usage() {
	echo
	echo "$0 -d databaseName -f filepath -h hostname -p port -U username -L location [-s]"
	echo
	echo "   -d: Target datatabase importing data"
	echo "   -f: Input exported database data"
	echo "   -h: HostName or IP address"
	echo "   -p: Port number"
	echo "   -U: PostgreSQL username"
	echo "   -L: Import Location name (dst location)"
	echo "   -s: Silent mode"
	echo
}

#-----------------------------------------------------------------------
# main
#-----------------------------------------------------------------------
# check OS type
getOsType
OSTYPE=$?
if test "${OSTYPE}" = "${OS_UNKNOWN}"; then
	echo "Sorry, this operating system is not supported."
	exit 1
fi

# lock
lockscript "${EXCLUSIVE_LOCKFILE}"
RETVAL=$?
if test "${RETVAL}" != "0"; then
	exit 1
fi

#-------------------------------
# Process argument
#-------------------------------
DBNAME=
IMPFILE=
PGHOST=
PGPORT=
DBUSER=
ORG_LOCATIONNAME=
LOCATIONNAME=
IS_SILENT=0
while getopts d:f:h:p:U:O:L:s OPT;
do
	case $OPT in
		d)
			DBNAME="$OPTARG"
			;;
		f)
			IMPFILE="$OPTARG"
			;;
		h)
			PGHOST="$OPTARG"
			;;
		p)
			PGPORT="$OPTARG"
			;;
		U)
			DBUSER="$OPTARG"
			;;
		L)
			LOCATIONNAME="$OPTARG"
			if test -n "${LOCATIONNAME}"; then
				LOCATIONNAME=`echo ${LOCATIONNAME} | sed -e 's:^[/][/]*::g'`
			fi
			;;
		s)
			IS_SILENT=1
			;;
		\?)
			usage
			exitscript 1
			;;
	esac
done
shift `expr $OPTIND - 1`


if test -z "${DBNAME}" || test -z "${IMPFILE}" || test -z "${PGHOST}" || test -z "${PGPORT}" || test -z "${DBUSER}" || test -z "${LOCATIONNAME}"; then
	echo "The some argument were missing."
	usage
	exitscript 1
fi

if test ! -r "${IMPFILE}"; then
	echo "Could not read \"${IMPFILE}\". Please check it."
	usage
	exitscript 2
fi

#-------------------------------
# notification message
#-------------------------------
if test "${IS_SILENT}" = "0"; then
	echo ""
	$ECHO_N "This operation update database. Are you ready ? (y/n) "$ECHO_C
	read ANS
	if test "${ANS}" != "y"; then
		echo "OK, abort."
		exitscript 0
	fi
fi

#-------------------------------
# Drop old database
#-------------------------------
echo "* Drop old database ..."
RETSTR=`$DROBDB -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} ${DBNAME} 2>&1`
RETVAL=$?
echo ${RETSTR} | grep 'ERROR:[ ]*database.*does not exist' >/dev/null
RETVAL2=$?
if test "${RETVAL}" != "0" && test "${RETVAL2}" != "0"; then
	echo "Failed to drop database \"${DBNAME}\". Please check it."
	exitscript 4
fi

#-------------------------------
# Create database
#-------------------------------
echo "* Create import database ..."
$CREATEDB -O ${DBUSER} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -E UNICODE ${DBNAME}
RETVAL=$?
if test "${RETVAL}" != "0"; then
	echo "Failed to create database \"${DBNAME}\". Please check it."
	exitscript 5
fi


#-------------------------------
# Import data
#-------------------------------
echo "* Import data to database ..."
$RESTORE -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -d ${DBNAME} -Fc ${IMPFILE}
RETVAL=$?
if test "${RETVAL}" != "0"; then
	echo "Failed to import data. Please check it."
	exitscript 6
fi

#-------------------------------
# Import data
#-------------------------------
echo "* Search original location name ..."
ORG_LOCATIONNAME=`${PSQL} -d ${DBNAME} -h ${PGHOST} -p ${PGPORT} -U ${DBUSER} -c "select rs_uri_txt from dav_resource where rs_depth_i=1 limit 1" 2>/dev/null | sed -n -e 's:[ ][ ]*::g' -e '/[ ]*\/[a-zA-Z0-9]/p' | sed -e 's:^[/][/]*::g'`
if test -z "${ORG_LOCATIONNAME}"; then
	echo "Failed to get original location name. Please contact developer."
	exitscript 7
fi

EXTRA_SQL=""
${PSQL} -d ${DBNAME} -h ${PGHOST} -p ${PGPORT} -U ${PGUSER} -c "\d divy_confirmreading" >/dev/null 2>&1
RETVAL=$?
if test "${RETVAL}" = "0"; then
	EXTRA_SQL="UPDATE divy_confirmreading SET cr_uri_txt = '/${LOCATIONNAME}' || substring(cr_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1);"
fi

EXTRA_SQL=""
${PSQL} -d ${DBNAME} -h ${PGHOST} -p ${PGPORT} -U ${PGUSER} -c "\d divy_box" >/dev/null 2>&1
RETVAL=$?
if test "${RETVAL}" = "0"; then
	EXTRA_SQL="UPDATE divy_box SET box_uri_txt = '/${LOCATIONNAME}' || substring(box_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1);"
fi

echo "* Modify import data for Location name ..."
$PSQL -h ${PGHOST} -p ${PGPORT} -d ${DBNAME} -U ${DBUSER} <<_MODIFY_DATA_

\set ON_ERROR_STOP 0
\set LO_TRANSACTION "rollback"

BEGIN;

UPDATE dav_link_collection SET lc_uri_txt = '/${LOCATIONNAME}' || substring(lc_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1);
UPDATE dav_lock SET lk_uri_txt = '/${LOCATIONNAME}' || substring(lk_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1), lk_lockkey_txt = '/${LOCATIONNAME}' || substring(lk_lockkey_txt, char_length('/${ORG_LOCATIONNAME}') + 1);
UPDATE dav_resource SET rs_uri_txt = '/${LOCATIONNAME}' || substring(rs_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1);
UPDATE divy_diskquota SET dkqt_uri_txt = '/${LOCATIONNAME}' || substring(dkqt_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1);
UPDATE divy_mailwatch SET mw_rs_uri_txt = '/${LOCATIONNAME}' || substring(mw_rs_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1);
UPDATE divy_resourcestate SET rt_uri_txt = '/${LOCATIONNAME}' || substring(rt_uri_txt, char_length('/${ORG_LOCATIONNAME}') + 1);
${EXTRA_SQL}

COMMIT;
_MODIFY_DATA_
RETVAL=$?
if test "${RETVAL}" != "0"; then
	echo "Failed to modify import data. Please check it."
	exitscript 8
fi

exitscript 0

