#!/bin/sh
#
# $Id$
#-------------------------------------------------------------------------------
# 定期バッチツール
#
# Copyright (c) 2006,2007 COMPUTER HI-TECH INC., All rights reserved.
#-------------------------------------------------------------------------------
# [ 役割 ]
#
#   * tfenvvars に指定された ${TfDbDbname}(リポジトリDB) と ${TfFsRootPath}
#     (ファイルストレージ) をバックアップ
#   * 上記データベースのREINDE
#   * 現存する全データベースのVACUUM
#   * 上記ストレージパス以下に存在するゴミファイルを掃除
#
# [ 動作条件 ]
#	このスクリプトは、PostgreSQL(/usr/local/teamfile/pgsql) と Webサーバ
#   (/usr/local/teamfile/www) が同じマシンにインストールされていなければ
#   動作しません。
#
# [ バックアップディレクトリ階層について ]
#
#    * $STORAGE_BACKUPMETHOD = "tar":
#    ------------------------------------------------
#    $BACKUP_ROOT/                          : Backup Root directory
#        |--- 0/                            : (The day part of today) % ${ROTATE_RATE}
#        |    |--- (Location Name).bkst.tar : Backup Storage
#        |    |--- (Location Name).bkcf/    : Backup Configuration
#        |    |--- (DB Name).bkdb.dat       : Backup Repository database
#        |    ...
#        |--- 1/
#        |--- 2/
#        ...
#
#    * $STORAGE_BACKUPMETHOD = "rsync":
#    ------------------------------------------------
#    $BACKUP_ROOT/                         : Backup Root directory
#        |--- YYYYMMDD/                    : ${ROTATE_RATE} days ago date.
#        |    |--- (Location Name).bkst/   : Backup Storage
#        |    |--- (Location Name).bkcf/   : Backup Configuration
#        |    |--- (DB Name).bkdb.dat      : Backup Repository database
#        |    ...
#        |--- YYYYMMDD/
#        |--- YYYYMMDD/
#        ...
#
# [ Note ]
#	複数のリポジトリDBとストレージをバックアップしたい場合には、
#	tfenvvars ファイルで指定される${TfDbDbname} と ${TfFsRootPath} に
#	バックアップするストレージパスとリポジトリDBの名称を半角スペースで区切って
#	指定します。その際、以下に注意して下さい。
#
#	* ${TfDbDbname} の要素数と${TfFsRootPath}の要素数は等しくなければならない
#	* ある位置の${TfDbDbname} の要素と${TfFsRootPath}の要素が正しい組合せである
#	(例)
#	    TfDbDbname="db_teamfile db_abcd"
#	    TfFsRootPath="/var/lib/teamfile/STORAGE/teamfile /var/lib/teamfile/STORAGE/abcd"
#
#	--> "db_teamfile" と "/var/lib/teamfile/STORAGE/teamfile" の
#	    組合せが正しいことを保証して下さい。さもないと正常にリストアできません。
#-------------------------------------------------------------------------------

# Define global values
ERR_FLAG=0

# The value of $BACKUP_OPT
# 0: (backup all) / 1: -N (maintenance only) / 2: -nr (no repository backup) / 3: -ns (no storage backup)
OPT_BACKUP_ALL=0
OPT_BACKUP_NO=1
OPT_BACKUP_NOREPOS=2
OPT_BACKUP_NOSTRG=3

BACKUP_OPT=$OPT_BACKUP_ALL

#----------------------
# env setting
#----------------------
if test -d "${TF_HOME}"; then
	MT_FUNCTION_HOME="${TF_HOME}/www/bin"
	MT_FUNCTION="${MT_FUNCTION_HOME}/mt-lib"
else
	if test -f "./mt-lib"; then
		MT_FUNCTION_HOME="."
	else
		MT_FUNCTION_HOME="/usr/local/teamfile/www/bin"
	fi
	MT_FUNCTION="${MT_FUNCTION_HOME}/mt-lib"
fi

if test ! -f ${MT_FUNCTION}; then
	echo "The configuration script \"${MT_FUNCTION}\" was missing. Please check it."
	exit 1
fi

# read functions and configuration
. ${MT_FUNCTION}

# for PostgreSQL
PGHOST="${TfDbHostname}"
PGPORT="${TfDbHostport}"

# for Oracle

# check STORAGE_BACKUPMETHOD
if test -z "${STORAGE_BACKUPMETHOD}"; then
	STORAGE_BACKUPMETHOD="rsync"
elif test "${STORAGE_BACKUPMETHOD}" != "tar" && test "${STORAGE_BACKUPMETHOD}" != "rsync"; then
	STORAGE_BACKUPMETHOD="rsync"
fi

if test "${STORAGE_BACKUPMETHOD}" = "tar"; then
	MMDATE=`date '+%d'`
	BACKUP_PREFIX_DIR="`expr ${MMDATE} % ${ROTATE_RATE}`"
else
	MMDATE=`date '+%Y%m%d'`
	BACKUP_PREFIX_DIR=${MMDATE}
	YESTERDAY_PREFIX_DIR=`date --date "1 days ago ${MMDATE}" +%Y%m%d`
	REMOVE_BACKUP_PREFIX_DIR=`date --date "${ROTATE_RATE} days ago ${MMDATE}" +%Y%m%d`
fi

#--------------------------------------
# Show usage
#--------------------------------------
usage() {
	echo "$0 [-N] | -nr | -ns | -h"
	echo "    -N  : NO backup data (maintenance only)"
	echo "    -nr : NO backup Repository data (maintenance + backup Storage)"
	echo "    -ns : NO backup Storage data (maintenance + backup Repository)"
	echo "    -h  : Show this message"
	echo ""
}

#--------------------------------------
# Start Apache2
#--------------------------------------
start_apache() {

	# MacOSX
	if test "${OSTYPE}" = "${OS_MACOSX102}" || test "${OSTYPE}" = "${OS_MACOSX103}"; then
		SystemStarter -v start TeamFileServer >> ${DAILYBAT_LOGFILE} 2>&1
	# Linux, Solaris
	elif test ${OSTYPE_CD} -lt "${OS_LINUX_RHEL7}"; then
		${APACHECTL} start >> ${DAILYBAT_LOGFILE} 2>&1
	else
		/usr/bin/systemctl start teamfile >> ${DAILYBAT_LOGFILE} 2>&1
	fi
}

#--------------------------------------
# Stop Apache
#--------------------------------------
stop_apache() {

	# MacOSX
	if test "${OSTYPE}" = "${OS_MACOSX102}" || test "${OSTYPE}" = "${OS_MACOSX103}"; then
		SystemStarter -v stop TeamFileServer >> ${DAILYBAT_LOGFILE} 2>&1
	# Linux, Solaris
	elif test ${OSTYPE_CD} -lt "${OS_LINUX_RHEL7}"; then
		${APACHECTL} stop >> ${DAILYBAT_LOGFILE} 2>&1
	else
		/usr/bin/systemctl stop teamfile >> ${DAILYBAT_LOGFILE} 2>&1
	fi
}

#--------------------------------------
# backup and maintenance data.
#
# $1 : The path of storage
# $2 : The name of repository
# $3 : The path of configuration
#--------------------------------------
maintenance_data() {
	st_fullpath="$1"
	db_name="$2"
	cf_fullpath="$3"
	if test "${REPOS_DBTYPE}" = "ora"; then
		db_sid="`echo ${db_name} | cut -d ":" -s -f3`"
		db_pass="`echo ${db_name} | cut -d ":" -s -f2`"
		db_name="`echo ${db_name} | cut -d ":" -s -f1`"
	fi

	ERR_FLAG=0

	# check storage path
	if test ! -d "${st_fullpath}"; then
		echo "# "`date`": The \"${st_fullpath}\" directory is missing." >> ${DAILYBAT_LOGFILE}
		return 0
	fi

	st_parentdir=`echo "X${st_fullpath}" | sed -e '1s/^X//' -e 's%/[^/]*$%%'`
	st_dirname=`echo ${st_fullpath} | sed -e '1s/^X//' -e 's%^.*/%%'`
	#-----------------------------------------------------------------------
	# Backup file storage
	#-----------------------------------------------------------------------
	if test "${BACKUP_OPT}" = $OPT_BACKUP_ALL || test "${BACKUP_OPT}" = $OPT_BACKUP_NOREPOS; then

		echo "# "`date`": Backup storage (\"${st_fullpath}\")" >> ${DAILYBAT_LOGFILE}

		if test "${STORAGE_BACKUPMETHOD}" = "tar"; then
			st_backup="${BACKUP_ROOT}/${BACKUP_PREFIX_DIR}/${st_dirname}.bkst.tar"

			# rename old backup file
			if test -f ${st_backup}; then
				mv ${st_backup} ${st_backup}.old
			fi

			# backup storage
			curdir="`pwd`"
			cd ${st_parentdir}
			tar -cf ${st_backup} ${st_dirname} >> ${DAILYBAT_LOGFILE} 2>&1
			RETVAL=$?
			if test "${RETVAL}" != "0"; then
				echo "# "`date`": Failed to backup \"${st_fullpath}\" directory. (ret ="${RETVAL}")" >> ${DAILYBAT_LOGFILE}
				ERR_FLAG=1
			fi
			cd ${curdir}
		# for "rsync" backup
		else
			st_backup="${BACKUP_ROOT}/${BACKUP_PREFIX_DIR}/${st_dirname}.bkst"
			st_y_backup="${BACKUP_ROOT}/${YESTERDAY_PREFIX_DIR}/${st_dirname}.bkst"

			if test "${st_fullpath}" != "${st_backup}"; then
				#rsync -av --delete ${st_fullpath}/ ${st_backup} >> ${DAILYBAT_LOGFILE}
				rsync -av --link-dest=${st_y_backup} ${st_fullpath}/ ${st_backup}
				RETVAL=$?
				if test "${RETVAL}" != "0"; then
					echo "# "`date`": Failed to backup \"${st_fullpath}\" directory by rsync method. (ret ="${RETVAL}")" >> ${DAILYBAT_LOGFILE}
					ERR_FLAG=1
				fi
			else
				echo "# "`date`": Failed to backup  \"${st_fullpath}\" directory by rsync.(backup src and dst are same !!)" >> ${DAILYBAT_LOGFILE}
				ERR_FLAG=1
			fi
		fi
	fi

	#-----------------------------------------------------------------------
	# Backup TeamFile repository
	#-----------------------------------------------------------------------
	if test "${ERR_FLAG}" = "0" && \
	   (test "${BACKUP_OPT}" = $OPT_BACKUP_ALL || test "${BACKUP_OPT}" = $OPT_BACKUP_NOSTRG); then

		echo "# "`date`": Backup repository database (\"${db_name}\")" >> ${DAILYBAT_LOGFILE}

		db_backup="${BACKUP_ROOT}/${BACKUP_PREFIX_DIR}/${db_name}.bkdb.dat"

		# rename old backup file
		if test -f ${db_backup}; then
			mv ${db_backup} ${db_backup}.old
		fi

		# dump database
		if test "${REPOS_DBTYPE}" = "pg"; then
			${PGDUMP} -Fc -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} ${db_name} > ${db_backup} 2>> ${DAILYBAT_LOGFILE}
		elif test "${REPOS_DBTYPE}" = "ora"; then
			${ORACLE_HOME}/bin/exp USERID=${db_name}/${db_pass}@${db_sid} FILE=${db_backup} CONSISTENT=Y
		fi

		RETVAL=$?
		if test "${RETVAL}" != "0"; then
			echo "# "`date`": Failed to backup \"${db_name}\"database (ret ="${RETVAL}")" >> ${DAILYBAT_LOGFILE}
			ERR_FLAG=1
		fi
		sleep 2
	fi

	#-----------------------------------------------------------------------
	# Backup configuration
	#-----------------------------------------------------------------------
	if test "${ERR_FLAG}" = "0" && test "${BACKUP_OPT}" != $OPT_BACKUP_NO; then

		echo "# "`date`": Backup configuration files" >> ${DAILYBAT_LOGFILE}

		cf_backup="${BACKUP_ROOT}/${BACKUP_PREFIX_DIR}/${st_dirname}.bkcf"
		if test "${cf_fullpath}" != "${cf_backup}"; then
			rsync -av --delete ${cf_fullpath}/ ${cf_backup} >> ${DAILYBAT_LOGFILE}
			RETVAL=$?
			if test "${RETVAL}" != "0"; then
				echo "# "`date`": Failed to backup \"${cf_fullpath}\" directory by rsync method. (ret ="${RETVAL}")" >> ${DAILYBAT_LOGFILE}
				ERR_FLAG=1
			fi
		else
			echo "# "`date`": Failed to backup  \"${cf_fullpath}\" directory by rsync.(backup src and dst are same !!)" >> ${DAILYBAT_LOGFILE}
			ERR_FLAG=1
		fi
	fi

	#----------------------------
	# reindex tables
	#----------------------------
	echo "# "`date`": Rebuild index of tables (\"${db_name}\")" >> ${DAILYBAT_LOGFILE}
	if test "${REPOS_DBTYPE}" = "pg"; then
		# check table
		EXTRA_SQL=""
		EXTRA_SQL2=""
		EXTRA_SQL3=""
		EXTRA_SQL4=""
		EXTRA_SQL5=""
		EXTRA_SQL6=""
		EXTRA_SQL7=""
		EXTRA_SQL8=""
		EXTRA_SQL9=""
		EXTRA_SQL10=""
		EXTRA_SQL11=""

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_diskquota" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL="REINDEX TABLE divy_diskquota;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_trash_info" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL2="REINDEX TABLE divy_trash_info;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_resourcestate" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL3="REINDEX TABLE divy_resourcestate;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_passpolicy" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL4="REINDEX TABLE divy_passpolicy;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_passngword" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL5="REINDEX TABLE divy_passngword;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_passpolicystatus" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL6="REINDEX TABLE divy_passpolicystatus;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_passhistory" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL7="REINDEX TABLE divy_passhistory;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_confirmreading" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL8="REINDEX TABLE divy_confirmreading;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_session" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL9="REINDEX TABLE divy_session;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_box" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL10="REINDEX TABLE divy_box;"
		fi

		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -c "\d divy_uploadpolicy" >/dev/null 2>&1
		RETVAL=$?
		if test "${RETVAL}" = "0"; then
			EXTRA_SQL11="REINDEX TABLE divy_uploadpolicy;"
		fi

		RETVAL=0  # reset
		${PSQL} -d ${db_name} -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} <<_REINDEX_TABLES

		\set ON_ERROR_STOP 0
		\set LO_TRANSACTION "rollback"

		BEGIN;

		REINDEX TABLE dav_dead_property;

		REINDEX TABLE dav_link_collection;

		REINDEX TABLE dav_lock;

		REINDEX TABLE dav_namespace;

		REINDEX TABLE dav_resource;

		REINDEX TABLE divy_clmodule;

		REINDEX TABLE divy_dbkey;

		REINDEX TABLE divy_dbms;

		REINDEX TABLE divy_grp;

		REINDEX TABLE divy_grpmem;

		REINDEX TABLE divy_mailwatch;

		REINDEX TABLE divy_sql;

		REINDEX TABLE divy_sqlcontent;

		REINDEX TABLE divy_sqldepend;

		REINDEX TABLE divy_sqlmem;

		REINDEX TABLE divy_sysmsg;

		REINDEX TABLE divy_usr;

		REINDEX TABLE divy_usraccessdeny;

		REINDEX TABLE divy_usrdiskquota;

		${EXTRA_SQL}

		${EXTRA_SQL2}

		${EXTRA_SQL3}

		${EXTRA_SQL4}

		${EXTRA_SQL5}

		${EXTRA_SQL6}

		${EXTRA_SQL7}

		${EXTRA_SQL8}

		${EXTRA_SQL9}

		${EXTRA_SQL10}

		${EXTRA_SQL11}

		commit;

_REINDEX_TABLES

	elif test "${REPOS_DBTYPE}" = "ora"; then
		echo "WHENEVER SQLERROR EXIT 1;" > ${TMP_SQL_FILE}
		echo "CONNECT ${db_name}/${db_pass}@${db_sid};" >> ${TMP_SQL_FILE}	
		echo "spool ${DAILYBAT_LOGFILE};" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX dav_resource_x1 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX dav_resource_x2 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX dav_resource_x3 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX dav_lock_x0 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX dav_lock_x1 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX divy_usr_x0 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX divy_usr_x1 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX divy_grp_x1 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX divy_sql_x0 REBUILD;" >> ${TMP_SQL_FILE}
		echo "ALTER INDEX divy_diskquota_x0 REBUILD;" >> ${TMP_SQL_FILE}
		echo "spool off;" >> ${TMP_SQL_FILE}
		echo "EXIT 0;" >> ${TMP_SQL_FILE}
		${SQLPLUS_S} @${TMP_SQL_FILE}	
		rm -f ${TMP_SQL_FILE}
	fi	

	RETVAL=$?
	if test "${RETVAL}" != "0"; then
		echo "# "`date`": Failed to rebuild index of \"${db_name}\" tables(ret ="${RETVAL}")" >> ${DAILYBAT_LOGFILE}
	fi

	echo "# "`date`": Cleanup garbage files (\"${st_fullpath}\")" >> ${DAILYBAT_LOGFILE}
	find "${st_fullpath}" -name '[0-9]*.[0-9]*@[a-zA-Z0-9-][a-zA-Z0-9.-]*.[0-9]*.tmp_*' -print >> ${DAILYBAT_LOGFILE}
	find "${st_fullpath}" -name '[0-9]*.[0-9]*@[a-zA-Z0-9-][a-zA-Z0-9.-]*.[0-9]*.tmp_*' -exec rm {} \;

	#----------------------------
	# cleanup old directory
	#----------------------------
	if test "${ERR_FLAG}" = "0"; then
		if test "${STORAGE_BACKUPMETHOD}" = "rsync"; then
			rm -fr "${BACKUP_ROOT}/${REMOVE_BACKUP_PREFIX_DIR}"
		fi
	fi

	#----------------------------
	# cleanup "*.old" file
	#----------------------------
	if test "${is_skip_bk}" = 0; then
		if test "${ERR_FLAG}" = "0"; then
			if test "${STORAGE_BACKUPMETHOD}" = "tar" -a -f ${st_backup}.old; then
				rm -f ${st_backup}.old
			fi

			if test -f ${db_backup}.old; then
				rm -f ${db_backup}.old
			fi
		else
			# recover old file
			if test "${STORAGE_BACKUPMETHOD}" = "tar"; then
				mv ${st_backup}.old ${st_backup}
			fi
			mv ${db_backup}.old ${db_backup}
		fi
	fi
}

#--------------------------------------
# Vacuum database
#
#--------------------------------------
vacuum_db() {
	if test "${REPOS_DBTYPE}" != "pg"; then
		return 0
	fi

	#----------------------------
	# vacuum dabase
	#----------------------------
	echo "# "`date`": Vacuum ALL database" >> ${DAILYBAT_LOGFILE}
	${PGVACUUM} -a -h ${PGHOST} -p ${PGPORT} -U ${PG_ADMIN_USER} -z -f -v >> ${DAILYBAT_LOGFILE} 2>&1
	RETVAL=$?
	if test "${RETVAL}" != "0"; then
		echo "# "`date`": Failed to vacuum ALL database (ret ="${RETVAL}")" >> ${DAILYBAT_LOGFILE}
	fi
}

#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# main
#+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
while getopts Nn:h OPT
do
	case $OPT in
		N)
			if test "${BACKUP_OPT}" = $OPT_BACKUP_NOREPOS || \
			   test "${BACKUP_OPT}" = $OPT_BACKUP_NOSTRG; then
			   usage
			   exit 1
			fi
			BACKUP_OPT=$OPT_BACKUP_NO
			;;
		n)
			if test "${BACKUP_OPT}" = $OPT_BACKUP_NO; then
				usage
				exit 1
			fi

			if test "$OPTARG" = "r"; then
				if test "${BACKUP_OPT}" = $OPT_BACKUP_NOSTRG; then
					usage
					exit 1
				fi
				BACKUP_OPT=$OPT_BACKUP_NOREPOS
			elif test "$OPTARG" = "s"; then
				if test "${BACKUP_OPT}" = $OPT_BACKUP_NOREPOS; then
					usage
					exit 1
				fi
				BACKUP_OPT=$OPT_BACKUP_NOSTRG
			else
				usage
				exit 1
			fi
			;;
		h)
			usage
			exit 0
			;;
		\?)
			usage
			exit 1
			;;
	esac
done
shift `expr $OPTIND - 1`

# get OS type
getOsType
OSTYPE=$?

# create backup directory
if test ! -d "${BACKUP_ROOT}/${BACKUP_PREFIX_DIR}"; then
	mkdir -p ${BACKUP_ROOT}/${BACKUP_PREFIX_DIR}
fi

# make new log
echo "# "`date` > ${DAILYBAT_LOGFILE}

# check combination of $TfDbDbname and $TfFsRootPath is valid
db_count=0
for dummy in ${TfDbDbname}; do
	db_count=`expr ${db_count} + 1`
done

st_count=0
for dummy in ${TfFsRootPath}; do
	st_count=`expr ${st_count} + 1`
done

if test "${db_count}" != "${st_count}"; then
	echo "# "`date`": The combination of \$TfDbDbname and \$TfFsRootPath is not valid. Please check it." >> ${DAILYBAT_LOGFILE}
	exit 1
fi

#----------------------
# check look
#----------------------
lockscript "${DAILYBAT_LOOKFILE}" >> ${DAILYBAT_LOGFILE}
RETVAL=$?
if test "${RETVAL}" != "0"; then
	exit 0
fi

if test "${BACKUP_OPT}" != $OPT_BACKUP_NO; then
	#----------------------
	# Stop Http Server
	#----------------------
	echo "# "`date`": Shutdown teamfile" >> ${DAILYBAT_LOGFILE}
	h_pid=""
	if test -f "/var/run/teamfile.pid"; then
		h_pid=`cat /var/run/teamfile.pid`
	fi

	stop_apache
	if test -n "${h_pid}"; then
		while [ 1 ]; do
			pidlist=`ps -p ${h_pid}| grep httpd`
			if test -z "${pidlist}"; then
				break;
			fi
			sleep 1
			cnt=`expr ${cnt} + 1`
			if test "${cnt}" = "300"; then
				break;
			fi
		done
	fi

	#----------------------
	# Restart Repository DB
	#----------------------
	if test "${REPOS_DBTYPE}" = "pg"; then
		echo "# "`date`": Restart PostgreSQL" >> ${DAILYBAT_LOGFILE}

		# MacOSX
		if test "${OSTYPE}" = "${OS_MACOSX102}" || test "${OSTYPE}" = "${OS_MACOSX103}"; then
			SystemStarter -v restart PostgreSQL >> ${DAILYBAT_LOGFILE} 2>&1
		# Linux, Solaris
		elif test ${OSTYPE_CD} -lt "${OS_LINUX_RHEL7}"; then
			if test -f ${POSTGRESCTL}; then
				${POSTGRESCTL} restart >> ${DAILYBAT_LOGFILE} 2>&1
			fi
		else
			/usr/bin/systemctl restart tfpostgresql >> ${DAILYBAT_LOGFILE} 2>&1
		fi
	fi
fi

IS_VACUUM_DB=0
if test -n "${TfDbDbname}"; then
	IS_VACUUM_DB=1
fi
#--------------------------
# do maintenance (Phase 1)
#--------------------------
for st_target in ${TfFsRootPath}; do
	db_target="`echo ${TfDbDbname} | sed -e 's/[ ]\+[^ ]*//g' -e 's/ //g'`"
	TfDbDbname="`echo ${TfDbDbname} | sed -e 's/'${db_target}'//g' -e 's/^ //g'`"

	maintenance_data ${st_target} ${db_target} "${APACHE_HOME}/conf"
done

#--------------------------
# do maintenance (Phase 2)
#--------------------------
if test "${IS_VACUUM_DB}" = "1" && test "${REPOS_DBTYPE}" = "pg"; then
	vacuum_db
fi

#----------------------------
# Start Http Server
#----------------------------
if test "${BACKUP_OPT}" != $OPT_BACKUP_NO; then
	echo "# "`date`": Start teamfile" >> ${DAILYBAT_LOGFILE}
	# unset PGCLIENTENCODING for EUC_JP postgreSQL database
	unset PGCLIENTENCODING

	# if apache had been running, start again.
	if test -n "${h_pid}"; then
		start_apache
	fi
fi

# unlock
unlockscript "${DAILYBAT_LOOKFILE}" >> ${DAILYBAT_LOGFILE}

exit 0

