#!/bin/sh
#
# $Id$
#-------------------------------------------------------------------------------
# 指定したプロセスIDのメモリイメージを取得するスクリプト.
#
# Create: 2005/12/13 Tue takehara
#-------------------------------------------------------------------------------
# (Input)  : -p $PID -d $CORE_OUTPUTDIR
# (Output) : core.$PID, dumpstat.dat
#-------------------------------------------------------------------------------
OWN_VERSION=`grep '^# [\$]Id\:' $0 | sed -e 's/[#\$\:]//g' | awk '{ print $3 }'`

#---------------------------------------------------------------------
# setup env
#---------------------------------------------------------------------
GDB="/usr/bin/gdb"
PS="/bin/ps"
UNAME="/bin/uname"
PATH="/bin:/sbin:/usr/bin:/usr/sbin"
CORE_BASE_NAME="core"
DUMPSTATUS_FILE="dumpstat.dat"

export PATH

# check gdb program
if test ! -x $GDB; then
	echo "[ERROR] Could not find gdb program. Please install it."
	exit 1
fi

# check OS
OS=`$UNAME -s`
if test "${OS}" != "Linux"; then
	echo "[ERROR] Sorry, this program is supported only Linux."
	exit 1
fi

#---------------------------------------------------------------------
# Process arguments
#---------------------------------------------------------------------
usage() {
	echo "* Generate Core dump script. [version: ${OWN_VERSION}] *"
	echo "* Usage: $0 [-p target_pid] [-d output_dir] | [-h]"
	echo "    -p target_pid : Specify target PID"
	echo "    -d output_dir : Specifiy output directory"
	echo "    -h : Show this message."
}

#---------------------------------------------------------------------
# main
#---------------------------------------------------------------------
TARGET_PID=
OUTPUT_DIR=
while getopts p:d:h OPT
do
	case $OPT in
		p)
			TARGET_PID="$OPTARG"	
			;;
		d)
			OUTPUT_DIR="$OPTARG"
			;;
		h)
			usage
			exit 0
			;;
		\?)
			usage
			exit 1
			;;
	esac
done
shift `expr $OPTIND - 1`

# check directory
if test ! -d "${OUTPUT_DIR}"; then
	echo "[ERROR] The output_dir is not valid directory. Please check it."
	usage
	exit 1
fi

if test -z "${TARGET_PID}"; then
	echo "[ERROR] Target PID is missing."
	usage
	exit 1
fi

# check pid process
$PS -p ${TARGET_PID} >/dev/null 2>&1
RETVAL=$?
if test "${RETVAL}" != "0"; then
	echo "[ERROR] The process \"${TARGET_PID}\" is active ? Maybe die."
	usage
	exit 1
fi

#---------------------------------------------------------------------
# execute gdb
#---------------------------------------------------------------------
CORE_PATH="${OUTPUT_DIR}/${CORE_BASE_NAME}.${TARGET_PID}"
DUMPSTATUS="${OUTPUT_DIR}/${DUMPSTATUS_FILE}"

# cleanup files
rm -f ${CORE_PATH} ${DUMPSTATUS}

# execute command
echo `date` > ${DUMPSTATUS}
echo "* Getting environment status..."

echo "##### ps result #####" >> ${DUMPSTATUS}
$PS -efl --forest >> ${DUMPSTATUS} 2>&1

echo "* Getting process image. Please wait for a while ..."
echo "##### gdb result #####" >> ${DUMPSTATUS}
$GDB <<_GDB_CMD_ >> ${DUMPSTATUS} 2>&1
attach ${TARGET_PID}
gcore ${CORE_PATH}
detach
quit
_GDB_CMD_
RETVAL=$?

if test "${RETVAL}" = "0"; then
	if test -f ${CORE_PATH}; then
		echo "* Success. Please send \"${CORE_PATH}\" and \"${DUMPSTATUS}\" file to developer."
	else
		echo "* Failed. Please check the a execution owner of this script is same as \"${TARGET_PID}\" owner."
	fi
else
	echo "* Failed. Please contact developer's group."
fi

exit 0

