# $Id$
#-------------------------------------------------------------------------------
# TeamFile server common perl library.
#-------------------------------------------------------------------------------
use strict;
use Getopt::Long;
use File::Temp qw(tempfile);

#------------------
# set env
#------------------
my $UPSP_MODE_CHECKUPDATE=1;
my $UPSP_MODE_PREINSTALL=2;
my $UPSP_MODE_INSTALL=3;
my $UPSP_MODE_POSTINSTALL=4;

#-------------------------------------------------------------------------------
# Define fixed values
#-------------------------------------------------------------------------------
my $APACHE_HOME    = '/usr/local/teamfile/www';
my $CONFIG_PATH    = '/usr/local/teamfile/www/conf/conf.d';
my $LOG_PATH       = '/var/log/teamfile';
my $TMP_PATH       = '/var/lib/teamfile/tmp';
my $JOB_WORK_PATH  = '/var/lib/teamfile/jobs';
my $TMP_SCRIPT_DIR = $TMP_PATH;

my $HTTPD_FILE     = $APACHE_HOME . '/conf/httpd.conf';
my $SSL_FILE       = $APACHE_HOME . '/conf/ssl.conf';
my $APACHE_PID     = '/var/run/teamfile.pid';
my $DEF_HTTP_PORT  = '80';
my $DEF_HTTPS_PORT = '443';

my $TFVALS_FILE    = '/usr/local/teamfile/www/bin/tfenvvars.default';

#
# getter functions
#
sub getApacheHome() { $APACHE_HOME }
sub getApachePidPath() { $APACHE_PID }
sub getConfigPath() { $CONFIG_PATH }
sub getLogPath() { $LOG_PATH }
sub getJobsWorkPath { $JOB_WORK_PATH }
sub getTmpScriptPath() { $TMP_SCRIPT_DIR }
sub getHttpConf() { $HTTPD_FILE }
sub getHttpsConf() { $SSL_FILE }

#-------------------------------------------------------------------------------
# Get ip address string
#-------------------------------------------------------------------------------
sub getipaddrs {
	my $ipaddr_str;
	my @ipaddr_array;

	if (open(IFCONFIG_IN, "/sbin/ifconfig -a |")) {

		while (my $line = <IFCONFIG_IN>) {
			chop($line);
			if ($line =~ /inet addr:/ && $line !~ /127\.0\.0\.1/) {
				$line =~ s/^[ ][ ]*inet[ ][ ]*addr://g;
				$line =~ s/[ ][ ]*.*$//g;
				push(@ipaddr_array, $line);
			}
		}

		if (@ipaddr_array) {
			$ipaddr_str = join("/", @ipaddr_array);
		}
		close(IFCONFIG_IN);
	}
	return ($ipaddr_str);
}

#-------------------------------------------------------------------------------
# Trim space
#-------------------------------------------------------------------------------
sub strim {
	my @list = @_;
	for (@list) {
		s/^\s+//;
		s/\s+$//;
	}
	return wantarray ? @list : $list[0];
}

#---------------------------------------------------------------------------
# Read update script's arguments and Callback functions
#---------------------------------------------------------------------------
sub process_upscript_arg {

	my $silent_mode        = 0;
	my $rollback_mode      = 0;
	my $script_mode        = undef;
	my $dl_dir_path        = undef;
	my $up_filelist_path   = undef;
	my $up_script_tmp_path = undef;
	my $opt_h = 0;

	# Get command line options
	my $result = GetOptions(
					"s" => \$silent_mode,
					"m=i" => \$script_mode,
					"R" => \$rollback_mode,
					"d=s" => \$dl_dir_path,
					"u=s" => \$up_filelist_path,
					"p=s" => \$up_script_tmp_path,
					"h" => \$opt_h);

	# check arguments
	if (!$result || $opt_h) {
		&up_usage();
	}

	if (!$script_mode) {
		print("The sicript_mode is EMPTY.\n");
		&up_usage();
	}

	if (! -d $dl_dir_path) {
		print("The path of downloaded module directory is not valid.\n");
		&up_usage();
	}

	if (! -f $up_filelist_path) {
		print("The filelist path is not valid.\n");
		&up_usage();
	}

	my %scntx = (
					silent       => $silent_mode,
					rollback     => $rollback_mode,
					pdownload    => $dl_dir_path,
					pupfilelist  => $up_filelist_path,
					pupscripttmp => $up_script_tmp_path,
				);

	# switch functions
	if ($script_mode == $UPSP_MODE_CHECKUPDATE) {
		$result = &upscript_checkupdate(%scntx);
	}
	elsif ($script_mode == $UPSP_MODE_PREINSTALL) {
		$result = &upscript_preinstall(%scntx);
	}
	elsif ($script_mode == $UPSP_MODE_INSTALL) {
		$result = &upscript_install(%scntx);
	}
	elsif ($script_mode == $UPSP_MODE_POSTINSTALL) {
		$result = &upscript_postinstall(%scntx);
	}

	return $result;
}

#-------------------------------------------------------------------------------
# Show update script's Usage
#-------------------------------------------------------------------------------
sub up_usage {
	print "Usage: xxxscript.sh [-s] [-m mode] [-R] [-d download dir] [-u filelist path] [-p script temporary dir] [-h]\n";
	exit(1);
}

#-------------------------------------------------------------------------------
# Chekc PID status
#-------------------------------------------------------------------------------
sub isRunning {
	my $pidfile = shift || '';

	if (-f $pidfile && open F_PID, "<", $pidfile) {
		my @rec = <F_PID>;
		my $pid = @rec[0];

		close F_PID;
		my $ret = `/bin/ps -p $pid`;
		if ($? == 0) {
			return 1;
		}
	}

	return 0;
}

#-------------------------------------------------------------------------------
# extract value
#-------------------------------------------------------------------------------
sub extractValue {
	my $line = shift || '';
	my $delm = shift || '=';

	if (!$line) {
		return '';
	}

	$line =~ s/\n$//g;
	$line =~ s/^[ ][ ]*//g;
	$line =~ s/[ ][ ]*$//g;
	$line =~ s/"//g;
	$line =~ s/'//g;

	my @keyval = split(/$delm/, $line);

	return @keyval[1];
}

#-------------------------------------------------------------------------------
# Lock
#-------------------------------------------------------------------------------
sub lockScript {
	my $flock = shift || '';

	# Check PID file
	if (-f $flock && open F_PID, "<", $flock) {
		my @rec = <F_PID>;
		close(F_PID);

		my $pid = @rec[0];

		system "/bin/ps -p $pid >/dev/null 2>&1";
		if ($? == 0) {
			return $pid;
		}
		else {
			unlink($flock);
		}
	}

	# Write PID to file.
	if (open F_PID, ">", $flock) {
		print F_PID $$;
		close F_PID;
	}

	return 0;
}

#-------------------------------------------------------------------------------
# Unlock
#-------------------------------------------------------------------------------
sub unlockScript {
	my $flock = shift || '';

	if (-f $flock) {
		unlink $flock;
	}
}

#-------------------------------------------------------------------------------
# Read global configuration
#-------------------------------------------------------------------------------
sub readTfGlobalSettings {
	my %params = ();

	# read $TFVALS_FILE
	if (open F_TFVALS, "<", $TFVALS_FILE) {
		while (<F_TFVALS>) {
			if ($_ =~ /^[ \t]*#/) {
				next;
			}

			if ($_ =~ /^TF_SYSEXEC_USER=/) {
				$params{TF_SYSEXEC_USER} = extractValue($_);
			}
			elsif ($_ =~ /^TF_SYSEXEC_USERPW=/) {
				$params{TF_SYSEXEC_USERPW} = extractValue($_);
			}
			elsif ($_ =~ /^TF_HTTP_PORT=/) {
				$params{TF_HTTP_PORT} = extractValue($_);
			}
			elsif ($_ =~ /^TF_HTTPS_PORT=/) {
				$params{TF_HTTPS_PORT} = extractValue($_);
			}
		}
		close F_TFVALS;
	}

	# execute script for reading values
	my ($tmph, $filename) = tempfile(	TEMPLATE => 'tempXXXXX',
										DIR => $TMP_SCRIPT_DIR,
										SUFFIX => '.sh');
	if (open F_TMP, ">", $filename) {
		print F_TMP '. ' . $TFVALS_FILE . "\n";
		print F_TMP 'echo ${DAILYBAT_LOOKFILE}';
		close F_TMP;
	}
	$params{DAILYBAT_LOOKFILE} = `/bin/sh $filename`;
	unlink $filename;

	if (!$params{TF_HTTP_PORT} && open F_HTTP, "<", $HTTPD_FILE) {
		while (<F_HTTP>) {
			if ($_ =~ /^[ \t]*#/) {
				next;
			}
			if ($_ =~ /[ \t]*Listen[ \t][ \t]*(\d+)/) {
				$params{TF_HTTP_PORT} = $1;
				last;
			}
		}
		close F_HTTP;
	}
	if (!$params{TF_HTTP_PORT}) {
		$params{TF_HTTP_PORT} = $DEF_HTTP_PORT;
	}

	if (!$params{TF_HTTPS_PORT} && open F_HTTPS, "<", $SSL_FILE) {
		while (<F_HTTPS>) {
			if ($_ =~ /^[ \t]*#/) {
				next;
			}
			if ($_ =~ /[ \t]*Listen[ \t][ \t]*(\d+)/) {
				$params{TF_HTTPS_PORT} = $1;
				last;
			}
		}
		close F_HTTPS;
	}
	if (!$params{TF_HTTPS_PORT}) {
		$params{TF_HTTPS_PORT} = $DEF_HTTPS_PORT;
	}

	return %params;
}


#-------------------------------------------------------------------------------
# Read location configuration
#-------------------------------------------------------------------------------
sub getLocationConflist {
	my @filelist = ();

	if (opendir CF_DIR, $CONFIG_PATH) {
		my @list = readdir CF_DIR;
		closedir CF_DIR;

		@list = sort(@list);
		foreach my $f (@list) {
			if ($f =~ /new_[a-zA-Z0-9_-][a-zA-Z0-9_-]*\.conf$/) {
				push @filelist, $f;
			}
		}
	}

	return @filelist;
}


#-------------------------------------------------------------------------------
# Parse location configuration
#-------------------------------------------------------------------------------
sub parseLocationConf {
	my $lcfile      = shift || '';
	my $gparams_ref = shift || undef;

	my %params = ();

	if (!$lcfile || !$gparams_ref) {
		return undef;
	}

	$lcfile = $CONFIG_PATH . '/' . $lcfile;
	if (! -f $lcfile) {
		return undef;
	}

	my %gparams = %{$gparams_ref};

	# set default values
	$params{UseSSL}            = 0;
	$params{UseAutodelete}     = 0;
	$params{UseConfirmReading} = 0;
	$params{BaseUri}           = undef;
	$params{Location}          = undef;

	# read data
	if (open F_LCFILE, "<", $lcfile) {
		while (<F_LCFILE>) {
			if ($_ =~ /^[ \t]*#/) {
				next;
			}

			if ($_ =~ /[ \t]*SSLRequireSSL/) {
				$params{UseSSL} = 1;
			}
			elsif ($_ =~ /[ \t]*TfAutoDelete[ \t][ \t]*(on|On|oN|ON)/) {
				$params{UseAutodelete} = 1;
			}
			elsif ($_ =~ /[ \t]*TfConfirmReading[ \t][ \t]*(on|On|oN|ON)/) {
				$params{UseConfirmReading} = 1;
			}
			elsif ($_ =~ /[ \t]*TfMlHostname[ \t][ \t](.*)/) {
				$params{mailhost} = $1;
			}
			elsif ($_ =~ /[ \t]*TfMlHostport[ \t][ \t](.*)/) {
				$params{mailport} = $1;
			}
			elsif ($_ =~ /^<Location[ ][ ]*\/([a-zA-Z0-9_-][a-zA-Z0-9_-]*)>/) {
				$params{Location} = $1;
			}
		}
		close F_LCFILE;
	}

	if (!$params{Location}) {
		return undef;
	}

	if ($params{UseSSL}) {
		$params{BaseUri} = 'https://127.0.0.1:' . $gparams{TF_HTTPS_PORT} . '/' . $params{Location};
	}
	else {
		$params{BaseUri} = 'http://127.0.0.1:'  . $gparams{TF_HTTP_PORT}  . '/'  . $params{Location};
	}

	return %params;
}

#
# Is it time to Run job ?
#
sub isItTimeToRunJob {
	my $now      = shift || '';
	my $cmp_hour = shift || '';
	my $cmp_min  = shift || '';

	if (!$now) {
		return 0;
	}

	my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = localtime($now);
	if ($hour == $cmp_hour && $min == $cmp_min) {
		return 1;
	}

	return 0;
}


1;

