/**
 * $Id$
 *
 * TeamFileサーバ用ライセンス検証ツール
 */
#include "apr.h"
#include "apr_strings.h"
#include "apr_time.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if APR_HAVE_TIME_H
#include <time.h>
#endif
#if APR_HAVE_UNISTD_H
#include <unistd.h>
#endif

/* TeamFile server header */
#include "util_common.h"

/*------------------------------------------------------------------------------
 * Define fixed values and macros
 *----------------------------------------------------------------------------*/
#define ERRLOG0(f)	(fprintf(stderr, f"\n"))
#define ERRLOG1(f,a1)	(fprintf(stderr, f"\n", a1))
#define ERRLOG2(f,a1, a2)	(fprintf(stderr, f"\n", a1, a2))

/*------------------------------------------------------------------------------
 * Define structure
 *----------------------------------------------------------------------------*/
typedef struct TfsLicense	TfsLicense;
struct TfsLicense {
	int u;
	int ua;
	int fopt;
	int is_p;
	int is_f;
	char *nif;
	char *nip;
	time_t st;
	int duration;
	char *filepath;
};

/*------------------------------------------------------------------------------
 * Declare private functions
 *----------------------------------------------------------------------------*/
static apr_status_t _InitAprContext(apr_pool_t **global);
static TfsLicense * _parse_argument(apr_pool_t *p, int argc, char *argv[]);
static int _validate_argument(TfsLicense *lc);
static void _usage(void);
static int _parse_licensetext(apr_pool_t *p, TfsLicense *lc);

/*------------------------------------------------------------------------------
 * main
 *----------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
	apr_pool_t *context = NULL;
	apr_status_t rv;
	TfsLicense *lc = NULL;

	/* APR環境の初期化 */
	rv = _InitAprContext(&context);
	if (rv != APR_SUCCESS) {
		ERRLOG1("* Failed to initialize APR context (%d)", rv);
		return 1;
	}

	/* 引数の処理 */
	if ((lc = _parse_argument(context, argc, argv)) == NULL) {
		return 1;
	}


	/* ライセンステキストの復号 */
	if (_parse_licensetext(context, lc)) {
		ERRLOG0("* Failed to decode license text");
		return 1;
	}

	return 0;
}

/*------------------------------------------------------------------------------
 * Define private functions
 *----------------------------------------------------------------------------*/
/**
 * APR 環境の初期化
 */
static apr_status_t _InitAprContext(apr_pool_t **global)
{
	apr_status_t rv;

	*global = NULL;

	if ((rv = apr_initialize()) != APR_SUCCESS) {
		return rv;
	}

	/* 終了ハンドラに登録 */
	atexit(apr_terminate);

	/* グローバルプールの作成 */
	if ((rv = apr_pool_create(global, NULL)) != APR_SUCCESS) {
		return rv;
	}

	return APR_SUCCESS;
}

/**
 * 引数の処理
 */
static TfsLicense * _parse_argument(apr_pool_t *p, int argc, char *argv[])
{
	int ch;
	TfsLicense *lc = NULL;

	//argcは与えられた引数の数	
	if (argc <= 1) {
		ERRLOG0("* Too few argument");
		//_usage();は --helpの表示
		_usage();
		return NULL;
	}

	lc = apr_pcalloc(p, sizeof(TfsLicense));
	lc->u    = -1;
	lc->ua   = -1;
	lc->fopt = -1;
	lc->is_p = 0;

	/* 引数処理 */
	while ((ch = getopt(argc, argv, "pf:")) != -1) {
		switch (ch) {

			case 'p':
				lc->is_p = 1;
				break;
			case 'f':
				if (IS_FILLED(optarg)) {
					lc->filepath = apr_pstrdup(p,optarg);
				}
				break;
		}
	}

	/* 引数の検証 */ 
	if (_validate_argument(lc)) {
		ERRLOG0("* Invalid argument.");
		_usage();
		return NULL;
	}
	return lc;
}

/**
 * 引数チェック
 */
static int _validate_argument(TfsLicense *lc)
{
	if (lc == NULL) return 1;

	if (IS_EMPTY(lc->filepath)) {
		ERRLOG0("* The parameter \"-f\" was missing.");
		return 1;
	}

	return 0;
}

/**
 * Usage表示
 */
static void _usage(void)
{
	fprintf(stderr, "Usage: tfslcvalidator -f filepath [-p]\n");
	fprintf(stderr, "       f: The license file path that you want to decode\n");
	fprintf(stderr, "       p: Show decoded license file\n");
	fflush(stderr);
}

static int _parse_licensetext(apr_pool_t *p, TfsLicense *lc)
{

	apr_status_t rv;
	apr_file_t *fd    = NULL;
	apr_finfo_t finfo = { 0 };
	int ret, i, st = 1;
	char *buf, *text, *plaintext = NULL;
	const char *keystr[] = { "TfServerLicenseKey", "TfDbLicenseKey", NULL };

	rv = apr_file_open(&fd, lc->filepath, APR_READ | APR_BINARY, 0, p);
	if (rv != APR_SUCCESS) {
		ERRLOG0("* The specified license file was missing.");
		return 1;
	}

	rv = apr_file_info_get(&finfo, APR_FINFO_SIZE, fd);
	if (rv != APR_SUCCESS) {
		apr_file_close(fd);
		ERRLOG0("* Could not read license file.");
		return 1;
	}
	if (finfo.size == 0) {
		apr_file_close(fd);
		ERRLOG0("* The license file was empty.");
		return 1;
	}

	while ((ret = divy_file_readline(p, fd, &buf)) != DIVY_UTIL_COMMON_ERR) {
		if (IS_EMPTY(buf)) {
			if (ret == DIVY_UTIL_COMMON_EOF) break;
			continue;	/* 空行は読み飛ばし */
		}
		/* コメント行は読み飛ばし */
		if (buf[0] == '#') continue;

		for (i = 0; i < 2; i++) {
			plaintext = NULL;
			if (strncmp(buf, keystr[i], strlen(keystr[i])) == 0) {
				text = (char *)dav_divy_trim_white(p, &buf[strlen(keystr[i])]);
				if (text != NULL && *text != '\0') {
					plaintext = (char *)divy_decipher_text(p, text);
				}
			}
			if (plaintext != NULL) {
				st = 0;
				if (lc->is_p == 1) {
					fprintf(stdout, "%s\n", plaintext);
				}
			}
		}	
	}
	
	/* 適切なライセンスが含まれていなかった場合 */
	if (plaintext == NULL) {
		ERRLOG0("* This file is not license file");	
		st = 1;
	}
	apr_file_close(fd);
	return st;
}

