/*
 * auth.c
 * $Id$
 *
 * TeamFile 認証 (basic only support)
 *
 */

#include "httpd.h"
#include "http_core.h"
#include "http_config.h"
#include "http_protocol.h"
#include "http_request.h"
#include "apr.h"
#include "apr_md5.h"
#include "apr_strings.h"
#include "apr_hash.h"
#include "apr_thread_mutex.h"

#include "mod_dav_tf.h"
#include "mod_dav.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "tf_rdbo_user.h"
#include "tf_env.h"
#include "auth.h"
#include "util.h"
#include "util_ldap.h"
#include "util_common.h"
#include "util_auth.h"

#include "util_md5.h"		 /* パスワードのMD5用 */

#include "mod_auth.h"

/*------------------------------------------------------------------------------
  Define Fixed values and macro
  ----------------------------------------------------------------------------*/

APLOG_USE_MODULE(dav_tf);

/*--------------------------------------------------------------
 * Declare prototype function (private scope functions)
 *-------------------------------------------------------------*/
static const char * _dav_divy_get_password(request_rec *r, const char *sent_pw);
static int _divy_get_auth_pw(request_rec *r, const char **pw);
static authn_status authn_check_teamfile_password(request_rec *r,
                                                  const char* user,
                                                  const char *password)
{
	TRACE(r->pool);

	ERRLOG2(r->pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"user = %s / password = %s", user, password);
//	return AUTH_DENIED;
	_dav_divy_get_password(r, password);
	return AUTH_GRANTED;
}

static authz_status authz_check_authorization(request_rec *r,
                                              const char* require_line,
                                              const void *parsed_require_line)
{
	TRACE(r->pool);

	/* 認証をスルーするURIか調べる スルーするならTRUE */
	if (dav_divy_auth_through_uri(r)) {
		return AUTHZ_GRANTED;
	}

	if (!r->user) {
		return AUTHZ_DENIED_NO_USER;
	}

	return AUTHZ_GRANTED;
}

/**
 * mod_dav_divy.cよりフックされる認証関数
 * conf->divyauthが設定(1)されている場合のみ動作する。
 *
 * @param r request_rec *
 * @return int (HTTP_STATUS)
 */
DIVY_DECLARE(int) dav_divy_authenticate_user(request_rec *r) {

	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	apr_pool_t *cache_p = (r->main) ? r->main->pool : r->pool;
	char 	   *sent_pw = "";
	const char *real_pw = NULL;
	int res;
	int ldapresult = TF_LDAP_TRUE;
	char *sid = NULL;
	int parse_result = DIVY_AUTH_SESSION_OK;
	int issaml = 0;	/* SAMLではない */
	int mnum;

	TRACE(r->pool);

#if 0
	const apr_array_header_t *arr = apr_table_elts(r->headers_in);
	int i;
	for (i=0; i < arr->nelts; i++) {
			apr_table_entry_t *entry = (apr_table_entry_t*)(arr->elts + (arr->elt_size * i));
			ERRLOG3(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"no %d, %s : %s", i, entry->key, entry->val);
	}
#endif

	/* 自分(Divy)が認証を行わない場合はDECLINED */
	if (conf->auth_on != DIVY_AUTH_ON) {
		return DECLINED;
	}

	/* 不要なユーザエージェントは拒絶する */
	if (apr_table_get(r->subprocess_env, DIVY_APENV_FORBIDDEN_USER_AGENT)) 
		return HTTP_FORBIDDEN;

	/* ブラウザからのリクエストならBasic認証ではなくPOST認証とする */
	if (divy_util_auth_request_is_browser(r)) {

		/* 他で挿入されたWWW-Authenticateヘッダを除去 */
		apr_table_unset(r->err_headers_out, "WWW-Authenticate");

#if 0
		/* 認証をスルーするURIか調べる スルーするならTRUE */
		if (dav_divy_auth_through_uri(r)) {
			/* 認証をスルーするということは、認証情報が
			   存在しないことを意味します。r->userがない状態です */
			return OK;
		}
#endif

		/* r->userへユーザIDをセットする */
		if ((parse_result = divy_util_auth_parse_cookie(r, &sent_pw, &sid))
													!= DIVY_AUTH_SESSION_OK) {
			if (parse_result == DIVY_AUTH_SESSION_TIMEOUT) {
				/* タイムアウトしましたのエラー */
				return divy_redirect_login(r, sid, HTTP_REQUEST_TIME_OUT);
			}
			else {
				/* cookieがないもしくは誤りは.loginへリダイレクト */
				return divy_redirect_login(r, sid, HTTP_OK);
			}
		}

		/* ここに来るのはr->userとsent_pwが入っている */

	}
	else {
		/* 
		 * r->userとr->ap_auth_typeをセットする。
		 */
		res = _divy_get_auth_pw(r, (const char **) &sent_pw);
		if (res != OK)
			return HTTP_UNAUTHORIZED;
	}

	/* r->userからサーバ側で該当のユーザのパスワードを取得する 
	 * 
	 * 1. 取得できない場合(user unknown)
	 * 2. 取得した内容とクライアントから送信されてきた内容と異なる場合
	 * 以上の場合は再度問い合わせを行う。
	 */
	real_pw = _dav_divy_get_password(r, sent_pw);

	 /*
	  * LDAPにユーザIDとパスワードのチェックをする
	  */
	ldapresult = divy_util_ldap_check_userid(r, r->user, sent_pw);

	if (ldapresult != TF_LDAP_TRUE) {
		/* パスワードが不一致であった回数を記録する */
		if (divy_rdbo_set_failed_login(r) == TF_LOCKOUT_LOCKOUT) {
			return divy_redirect_login(r, sid, HTTP_FORBIDDEN);
		}
		/* ログには出力しているので正常な失敗として通知 */
		return divy_redirect_login(r, sid, HTTP_UNAUTHORIZED);
	}

	/*
	 * ユーザがデータベースに存在しなかった場合再認証
	 */
	if (IS_EMPTY(real_pw)) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed Password for user ( %s ) not found",
			REPLACE_NULL(r->user));

		return divy_redirect_login(r, sid, HTTP_UNAUTHORIZED);
	}

	/* SAMLの認証で来たのかを調べる */
	issaml = divy_pcache_get_flag(cache_p, DIVY_PCACHE_FLG_USE_SAML);
	/* 
	 * SAMLの認証できたのならパスワードがそもそもわからない為、dbに記録
	 * されているパスワードがそのまま返却されてきます(sent_pw)
	 * なのでここでap_md5を実行する必要はない
	 */
	if (conf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON && issaml == 0) {
		sent_pw = ap_md5(r->pool, (const unsigned char*)sent_pw);
	}

	/*
	 * データベースの内容とクライアントから送られたデータと
	 * 異なる場合再認証
	 */
	if (strcmp(sent_pw, real_pw)) {
		if (divy_util_auth_allow_ignore_login(r, sent_pw) == 0) {

			ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"password mismatch error."
					"[ sent userid => %s / sent password => %s ]",
					REPLACE_NULL(r->user), REPLACE_NULL(sent_pw));

			/* パスワードが不一致であった回数を記録する */
			if (divy_rdbo_set_failed_login(r) == TF_LOCKOUT_LOCKOUT) {
				return divy_redirect_login(r, sid, HTTP_FORBIDDEN);
			}

			return divy_redirect_login(r, sid, HTTP_UNAUTHORIZED);
		}
		/* ここに来るということはパスワードを無視したログインが可能だった */
	}

	/* メインリクエストのみ(r->mainがセットされていない場合) */
	if (!r->main) {

		/* ロックアウトのリフレッシュ TODO BUG */
		if (divy_rdbo_refresh_lockout(r) != 0) {
			return divy_redirect_login(r, sid, HTTP_FORBIDDEN);
		};

	 	/* データベースにアクセスした時間/クライアントをセットする。 */
		if (dav_divy_update_lastaccess(r) != 0) {
			return HTTP_INTERNAL_SERVER_ERROR;
		}

		if (divy_enable_memcache(r) && IS_FILLED(sid)) {
			divy_util_auth_refresh_session(r, sid);
		}

	}

#ifdef DIVY_SUPPORT_LOGACCESS
	/* ロガー用にアクセスユーザIDをセットする */
	divy_logger_set_userid(r, r->user);
#endif	/* DIVY_SUPPORT_LOGACCESS */

	/* サーバ—ポリシーヘッダを送り込む */
	if (conf->tfserverpolicyheader != NULL) {
		apr_table_set(r->err_headers_out,
				DIVY_HEADER_SERVER_POLICY, conf->tfserverpolicyheader);
	}

	/* ブラウザだけCSRFトークンの存在を確認する */
	mnum = divy_get_method_number(r);
	if (divy_util_auth_request_is_browser(r)) {
		if (mnum != M_POST && mnum != M_GET && mnum != M_OPTIONS) {
			const char *csrf_token;
			char *sesspw, *sessid;
			divy_auth_session session = { 0 };
			csrf_token = sesspw = sessid = NULL;

			/* 送信されてきたヘッダのトークン取得 */
			csrf_token = apr_table_get(r->headers_in, "X-CSRF-TOKEN");

			/* セッション内のトークンの取得 */
			session.sid = sid;
			(void)(divy_util_auth_get_memcache_userinfo(r, &session));

			ERRLOG2(r->pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
							"x-csrf-token = %s <=> cookie.uniqueidi = %s",
							csrf_token, session.uniqueid);

			if (IS_EMPTY(csrf_token) || IS_EMPTY(session.uniqueid)) {
				ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DEBUG,
						"Failed to CSRF token error."
						"x-csrf-token = %s <=> cookie.uniqueid = %s",
						csrf_token, session.uniqueid);
				return HTTP_FORBIDDEN;
			}

			if (strcmp(csrf_token, session.uniqueid) != 0) {
				ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DEBUG,
						"Failed to CSRF token error."
						"x-csrf-token = %s <=> cookie.uniqueid = %s",
						csrf_token, session.uniqueid);
					return HTTP_FORBIDDEN;
			}
		}
	}

	return OK;
}

/**
 * Authorization ヘッダをパースしてパスワード文字列を取得する.
 *
 */
DIVY_DECLARE(char *) divy_parse_authorization_header(request_rec *r)
{
	const char *auth_line = apr_table_get(r->headers_in,
						(PROXYREQ_PROXY == r->proxyreq)
						   ? "Proxy-Authorizeation"
						   : "Authorization");
	const char *str;
	char *password = NULL;

	/* Authorization ヘッダがない場合 */
	if (IS_EMPTY(auth_line)) {
		return NULL;
	}

	/* Basic 認証でない場合はNG */
	str = ap_getword(r->pool, &auth_line, ' ');
	if (IS_EMPTY(str) ||
		(IS_FILLED(str) && strcasecmp(str, DIVY_AUTH_TYPE_BASIC) != 0)) {
		return NULL;
	}
	/* 半角スペースまたはタブ文字を飛ばす */
	while (*auth_line == ' ' || *auth_line == '\t') {
		auth_line++;
		if (*auth_line == '\0') {
			return NULL;	/* 飛ばしに失敗 */
		}
	}
	
	/*
	 * auth_lineより[ user:passwordのbase64エンコード ]
	 * をデコードしてパスワードとして返す。
	 * Example YXNkZmFzZGY= --> yonekura
	 */
	str = ap_pbase64decode(r->pool, auth_line);
	password = divy_strchr((char*)str, ':');
	if (password != NULL) {
		password++;
	}

	return password;
}

/**
 * リクエストヘッダを確認して認証タイプを取得する。
 * 
 * @param r request_rec *
 * @param pw const char **
 * @return HTTP_STATUS
 *
 */
static int _divy_get_auth_pw(request_rec *r, const char **pw) {

	const char *auth_line = apr_table_get(r->headers_in,
						(PROXYREQ_PROXY == r->proxyreq)
						   ? "Proxy-Authorizeation"
						   : "Authorization");
	const char *t = NULL;
	const char *auth_param = NULL;
	const char *auth_type = NULL;

	*pw = NULL;	/* 初期化 */

	TRACE(r->pool);

	/* Authorization ヘッダがない場合 HTTP_UNAUTHORIZED */
	if (!auth_line) {
		divy_note_auth_failure(r);
		return HTTP_UNAUTHORIZED;
	}

	ERRLOG1(r->pool, APLOG_DEBUG, DIVY_SST_DEBUG, "auth_line = %s", auth_line);

	/* リクエストの認証タイプを取得 */
	auth_type = ap_getword(r->pool, &auth_line, ' ');
	if (strcasecmp(auth_type, DIVY_AUTH_TYPE_BASIC) == 0) {
		/* Basic認証の場合 */
		auth_param = auth_line;
		while (*auth_param == ' ' || *auth_param == '\t') {
				auth_param++;
		}

		t = ap_pbase64decode(r->pool, auth_param);
		r->user = ap_getword_nulls(r->pool, &t, ':');
		r->ap_auth_type = "Basic";
		*pw = t;
	}
	else if (strcasecmp(auth_type, DIVY_AUTH_TYPE_BEARER) == 0) {
		/* Bearer認証の場合 */
		/* Microsoft Office */
		ERRLOG0(r->pool, APLOG_DEBUG,
			 DIVY_FST_INFO + DIVY_SST_DEBUG, "AUTH TYPE = BEARER");
		r->ap_auth_type = "Bearer";
		*pw = NULL;	/* パスワードはNULL */
	}

	return OK;
}

/**
 * パスワードを取得する。
 * メインリクエストのみデータベースを検索してデータをr->notesテーブル
 * に設定する。
 * サブリクエストはr->main->notesテーブルより取得する為、r->mainより
 * 値を取得するだけとする。
 *
 * @param r request_rec *
 * @param sent_pw const char * ユーザが入力したパスワード
 * @return const char * 取得したパスワード
 */
static const char * _dav_divy_get_password(request_rec *r, const char *sent_pw)
{

	int ret;

	TRACE(r->pool);

	/* メインリクエストの場合のみデータベースよりセットする。
	 * サブリクエストはメインリクエストからすべての情報を
	 * 取得する為、データベースを利用しない。
	 */

	if (!r->main) {
		ret = divy_rdbo_cache_userinfo(r, 0/* usecache */);
		if (ret == DIVY_STCODE_USER_NOT_EXIST) {
			/*
			 * ユーザが存在しなかった場合
			 *
			 * (note)
			 * ユーザが存在しないエラーはLDAPを利用している
			 * 時にだけ起こりうるエラーです
			 */
			if (divy_util_ldap_found_create_user(r, r->user, sent_pw)
				       			!= TF_LDAP_TRUE) {
				return NULL;
			}
			else {
				/* 成功したら再度チェック */
				if ((ret = divy_rdbo_cache_userinfo(r, 1/* no cache */)) != 0) {
					/* 失敗したらあきらめる */
					return NULL;
				}
			}

		} else if (ret != 0) {
			return NULL;
		}
	}

	return divy_get_password(r);
}


DIVY_DECLARE(int) dav_divy_hook_tf_auth_failure(request_rec *r, const char* auth_type)
{

	TRACE(r->pool);

	if (strcasecmp(auth_type, "teamfile"))
		return DECLINED;

	divy_note_auth_failure(r);
	return OK;

}
/*--------------------------------------------------------------
  AUTH-provider Hook structure see auth.c
  --------------------------------------------------------------*/
const authn_provider authn_tf_provider = {
    &authn_check_teamfile_password,
};

const authz_provider authz_tf_provider = {
    &authz_check_authorization,
    NULL,
};



