/**
 * $Id$
 *
 * tf_db_pg9.c
 *
 * Postgres DBアクセスプロバイダ (実装)
 *
 */
/* Apache header files */
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */

#if defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H
#include <unistd.h>
#endif	/* defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H */

/* postgres header files */
#ifndef INCLUDE_PGS_H
#define INCLUDE_PGS_H
#include "libpq-fe.h"
#endif

/*
 * (note)
 * カラムタイプの取得を断念したことにより以下のincludeは不要になりました。
 * 理由の詳細は関数pg_stmt_executeQueryを参照して下さい。
 */
#if 0
#ifndef INCLUDE_POSTGRES_H
#define INCLUDE_POSTGRES_H
#include "postgres.h"
#include "catalog/pg_type.h"
#endif
#endif

/* docmng header files */
#include "tf_db_pg9.h"
#include "util_common.h"
#include "tf_provider.h"

/*--------------------------------------------------------------
  Define fixed value and macros
  --------------------------------------------------------------*/
/**
 * このモジュールが扱うDBMSの種類。
 * 他のDBプロバイダと重複してはならない.
 */
#define PG_DB_PROVIDER_TYPE "pg"

/*
 * エラーメッセージ、ネイティブコード格納領域のサイズ(byte)
 */
#define PG_EMSG_LEN	512

/*
 * l が表す整数とPG_EMSG_LEN を比較して、小さい方の値を返すマクロ
 * l が負の値の場合には、0を返却する
 */
#define PG_GET_EMSG_LEN(l) (l < PG_EMSG_LEN ? (l > 0 ? l : 0) : PG_EMSG_LEN)

/**
 * このDBプロバイダのクライアントエンコーディング
 */
#define PG_CLIENT_ENCODING "UTF-8"

/**
 * 件数指定FETCH をサポートするかどうか
 * 以下のコメントアウトを外すと件数指定のFETCHが動作するようになります。
 */
#define PG_SUPPORT_PARTIAL_FETCH

#ifdef PG_SUPPORT_PARTIAL_FETCH
/**
 * 最大FETCH 件数
 */
#define PG_MAX_FETCH_CNT	"1000"
#endif	/* PG_SUPPORT_PARTIAL_FETCH */

/*--------------------------------------------------------------
  Define structures and enum
  --------------------------------------------------------------*/
/*
 * A context object depending on specific DBMS.
 */
struct __db_dbms_Context {
	PGconn     *pgconn;
	/*
	 * トランザクションごとで発行されるカーソルを保持する。
	 */
	int	   cursorcnt;
};

/*
 * A statement context object depending on specific DBMS.
 */
struct __db_dbms_stmtCntxt {
	/* プリペアされていないオリジナルSQL */
	const char *query;

	/* pg_prepared_part * を持つ配列(定義はtf_db_pg.c) */
	divy_array_t *prepared_parts;
};
/*
 * A resultset context object depending on specific DBMS.
 * Implematation of this object will be supplied by provider module.
 */
struct __db_dbms_rsetCntxt {
	PGresult         *pgres;
	int		  maxtup;
	int		  curtup;
	int		  transref;	/* トランザクションカウント */
#ifdef PG_SUPPORT_PARTIAL_FETCH
	int		  use_cursor;
	char		 *cursorname;	/* カーソル名 */
#endif	/* PG_SUPPORT_PARTIAL_FETCH */
};

/**
 * ある1つのバインド変数にバインドされた値とそのバインド変数の結合先となる
 * SQL文の断片を保持する構造体。
 */
struct __pg9_prepared_part {
/* バインド変数に値が入っていないことを示す値 */
#define PG_BIND_INIT_STR	"\a\b"
	/* バインド変数の値 */
	char *bind;

	/* SQL文の断片。bindの結合先 */
	char *sql;

	/* バインド変数に値があったかどうか(0: なし /1: あり) */
	apr_int32_t has_bind;

	/* バインド変数の型 */
	DbSqlType type;
};
typedef struct __pg9_prepared_part  pg9_prepared_part;

/*--------------------------------------------------------------
  Declare provider functions
  --------------------------------------------------------------*/
/*
 * These functions are used for DbDataSource structure.
 */
static DbConn *pg9_dbds_getDbConn(DbDataSource *dbds, apr_pool_t *pool);
static void pg9_dbds_rebuildConn(DbDataSource *dbds);
static int  pg9_dbds_getCode(const DbDataSource *dbds);
static char * pg9_dbds_getNativeCode(const DbDataSource *dbds);
static char * pg9_dbds_getMsg(const DbDataSource *dbds);

/*
 * These functions are used for DbConn structure.
 */
static void pg9_dbconn_close(DbConn *dbconn);
static int  pg9_dbconn_isClosed(DbConn *dbconn);
static DbPreparedStmt * pg9_dbconn_prepareStatement(DbConn *dbconn, const char *sqltext, apr_pool_t *pool);
static void pg9_dbconn_startTrans(DbConn *dbconn, const unsigned int timeout);
static void pg9_dbconn_commit(DbConn *dbconn);
static void pg9_dbconn_rollback(DbConn *dbconn);
static void pg9_dbconn_setTransIsolation(DbConn *dbconn, const DbIsolationLv level);
static const char * pg9_dbconn_getExecutableSQL(DbConn *dbconn);
static int  pg9_dbconn_getCode(const DbConn *dbconn);
static char * pg9_dbconn_getNativeCode(const DbConn *dbconn);
static char * pg9_dbconn_getMsg(const DbConn *dbconn);

/*
 * These functions are used for DbPreparedStmt structure.
 */
static void pg9_stmt_close(DbPreparedStmt *stmt);
static DbResultSet * pg9_stmt_executeQuery(DbPreparedStmt *stmt, apr_pool_t *pool);
static DbResultSet * pg9_stmt_executeQueryForUpdate(DbPreparedStmt *stmt, apr_pool_t *pool);
static int  pg9_stmt_executeUpdate(DbPreparedStmt *stmt, apr_pool_t *pool);
static void pg9_stmt_setInt(DbPreparedStmt *stmt, const int index, const apr_int32_t value);
static void pg9_stmt_setBigInt(DbPreparedStmt *stmt, const int index, const apr_int64_t value);
static void pg9_stmt_setDouble(DbPreparedStmt *stmt, const int index, const double value);
static void pg9_stmt_setString(DbPreparedStmt *stmt, const int index, const char *value);
static int  pg9_stmt_getCode(const DbPreparedStmt *stmt);
static char * pg9_stmt_getNativeCode(const DbPreparedStmt *stmt);
static char * pg9_stmt_getMsg(const DbPreparedStmt *stmt);
static char * pg9_stmt_escWildCard(const DbPreparedStmt *stmt, const char *value);

/*
 * These functions are used for DbResultSet structure.
 */
static void   pg9_rset_close(DbResultSet *rset);
static int    pg9_rset_next(DbResultSet *rset);
static int    pg9_rset_getColumnCount(DbResultSet *rset);
static char * pg9_rset_getColumnName(DbResultSet *rset, const int index);
static DbSqlType pg9_rset_getColumnType(DbResultSet *rset, const int index);
static int    pg9_rset_getColumnSize(DbResultSet *rset, const int index);
static apr_int32_t pg9_rset_getInt(DbResultSet *rset, const int index);
static apr_int64_t pg9_rset_getBigInt(DbResultSet *rset, const int index);
static double pg9_rset_getDouble(DbResultSet *rset, const int index);
static char * pg9_rset_getString(DbResultSet *rset, const int index);
static int    pg9_rset_getCode(const DbResultSet *rset);
static char * pg9_rset_getNativeCode(const DbResultSet *rset);
static char * pg9_rset_getMsg(const DbResultSet *rset);


/*--------------------------------------------------------------
  Declare private functions
  --------------------------------------------------------------*/
static void init_dbds_method(DbDataSource *dbds);
static void init_dbconn_method(DbConn *dbconn);
static void init_stmt_method(DbPreparedStmt *stmt);
static void init_rset_method(DbResultSet *rset);
static void set_value_query(DbPreparedStmt *stmt, const int index,
				const DbSqlType type, const char *value);
static divy_array_t * _prepare_query_string(apr_pool_t *p, const char *sql);
static const char * _build_query_string(apr_pool_t *p, divy_array_t *prepared_parts);
static void _clear_binded_value(apr_pool_t *p, divy_array_t *prepared_parts);
static DbResultSet * _pg9_stmt_executeQuery(DbPreparedStmt *stmt, apr_pool_t *pool, int use_cursor);

#if defined(DIVY_LIB_DYNAMIC_EXPORT) && defined(DIVY_USE_GNU_LD)
static void _db_pg9_init(void) __attribute__ ((constructor));
static void _db_pg9_fini(void) __attribute__ ((destructor));
#endif	/* DIVY_LIB_DYNAMIC_EXPORT && DIVY_USE_GNU_LD */
static int _db_pg9_init_func(apr_pool_t *pconf);
static int _db_pg9_create_dbdatasource(apr_pool_t *p,
					const char *providerType,
					DbDataSource **dbds);

/*--------------------------------------------------------------
  Define provider library hook structure
  --------------------------------------------------------------*/
DIVY_PROVIDER_HOOK_STRUCT(db,pg9,
		_db_pg9_init_func,		/* 初期化ハンドラ */
		divy_lib_nop_fini_func);	/* 終了ハンドラ(nop) */

/*--------------------------------------------------------------
  Define public functions
  --------------------------------------------------------------*/
DIVY_DECLARE(DbDataSource *) pg9_dbds_createDbds(apr_pool_t *pool)
{
	DbDataSource *ds;

	ds = apr_pcalloc(pool,sizeof(DbDataSource));

	init_dbds_method(ds);

	/* DBプロバイダ環境が初期化されていないことを記録する */
	ds->isFinishedInitEnv = DBENV_INIT_NOT_FINISHED;

	return ds;
}

/*--------------------------------------------------------------
  Define provider function
  --------------------------------------------------------------*/

/*
 * These functions are used for DbDataSource structure.
 */
/**
 * Databaseに接続する。
 *
 * @param dbds DbDataSource*
 * @param pool apr_poolt*
 *
 */
static DbConn *pg9_dbds_getDbConn(DbDataSource *dbds, apr_pool_t *pool)
{
	ConnStatusType status;
	DbConn *dbconn;
	int msglen;
	int ret;

	dbconn = apr_pcalloc(pool, sizeof(DbConn));
	dbconn->dbcntxt = apr_pcalloc(pool,sizeof(DbCntxt));
	dbconn->__nativecode = apr_palloc(pool,sizeof(char) * PG_EMSG_LEN);
	dbconn->__msg = apr_palloc(pool,sizeof(char) * PG_EMSG_LEN);
	
	/* 関数へのポインタに実装を提供する */
	init_dbconn_method(dbconn);

	if ( dbds == NULL ) {
		char *errmsg = "ERROR: dbds is NULL.";
		apr_size_t len = strlen(errmsg) + 1;

		dbconn->__code = DB_ERROR;
		apr_cpystrn(dbconn->__msg, errmsg, PG_GET_EMSG_LEN(len));
		return NULL;
	}

	dbconn->dbds = dbds;

	/* 新たにデータベースサーバーとの接続を確立する。*/
	dbconn->dbcntxt->pgconn = PQsetdbLogin(dbds->hostname,
					  dbds->hostport,
					  NULL,
					  NULL,
					  dbds->dbname,
					  dbds->username,
					  dbds->password);

	status = PQstatus(dbconn->dbcntxt->pgconn);
	if (PQstatus(dbconn->dbcntxt->pgconn) != CONNECTION_OK) {
		msglen  = strlen(PQerrorMessage(dbconn->dbcntxt->pgconn));
		apr_cpystrn(dbconn->__msg,
				PQerrorMessage(dbconn->dbcntxt->pgconn), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *dbconn->__msg = '\0';
		dbconn->__code = DB_ERROR;
	} else {
		dbconn->__code = DB_SUCCESS;
	}

	/* クライアントエンコーディングを設定する */
	if (dbconn->__code == DB_SUCCESS) {
		ret = PQsetClientEncoding(dbconn->dbcntxt->pgconn, PG_CLIENT_ENCODING);
		if (ret) {
			msglen  = strlen(PQerrorMessage(dbconn->dbcntxt->pgconn));
			apr_cpystrn(dbconn->__msg,
					PQerrorMessage(dbconn->dbcntxt->pgconn),
					PG_GET_EMSG_LEN(msglen));
			if (msglen == 0) *dbconn->__msg = '\0';
			dbconn->__code = DB_ERROR;
		}
		else {
			dbconn->__code = DB_SUCCESS;
		}
	}

	return dbconn;
}


/**
 * データソースを元に再接続を行う。
 * @param DbDataSource 
 *
 */

static void pg9_dbds_rebuildConn(DbDataSource *dbds)
{
	return;
}

/**
 * データソースよりエラーコードを取得する。
 * 
 * @param DbDataSource 
 * retuen int
 */
static int pg9_dbds_getCode(const DbDataSource *dbds)
{
	return dbds->__code;
}

/**
 * データソースよりネイティブなエラーコードを取得する。
 * 
 * @param DbDataSource
 * return char
 *
 */
static char *pg9_dbds_getNativeCode(const DbDataSource *dbds)
{
	return dbds->__nativecode;
}

/**
 * データソースよりメッセージを取得する。
 *
 * @param dbds DbDataSource
 * @return dbds->__msg char*
 * 
 */
static char *pg9_dbds_getMsg(const DbDataSource *dbds)
{
	return dbds->__msg;
}


/*
 * These functions are used for DbConn structure.
 */

/**
 * 接続を閉じる。
 * 発生したエラー処理は行わない。
 *
 * @param dbconn DbConn
 * 
 */
static void pg9_dbconn_close(DbConn *dbconn)
{
	if (dbconn->dbcntxt->pgconn != NULL) {
		PQfinish(dbconn->dbcntxt->pgconn);
		dbconn->dbcntxt->pgconn = NULL;
	}
}

/**
 * 接続が閉じられているか確認する。
 *
 * @param dbconn DbConn
 *
 */ 
static int pg9_dbconn_isClosed(DbConn *dbconn)
{
	if (PQstatus(dbconn->dbcntxt->pgconn) == CONNECTION_BAD)
	{
		return DB_FALSE;
	} else {
		return DB_TRUE;
	}

}

/**
 * SQLを解析する。
 * postgresqlではpreparの処理がない（実際にはPQexecが行う）為
 * ここではstmtの作成とSQLの文字列を作成するだけである。
 * そのため、ここではデータベースからのエラーは戻らない。
 *
 * @param dbconn DbConn
 * @param sqltext const char*
 * @param pool apr_pool_t*
 * @retun stmt DbPrepardStmt
 *
 */
static DbPreparedStmt *pg9_dbconn_prepareStatement(DbConn *dbconn, const char *sqltext, apr_pool_t *pool)
{
	DbPreparedStmt *stmt;
	
	stmt = apr_pcalloc(pool, sizeof(DbPreparedStmt));
	stmt->stmtcntxt = apr_pcalloc(pool,sizeof(DbStmtCntxt));
	stmt->__msg = apr_pcalloc(pool,sizeof(char) * PG_EMSG_LEN);
	stmt->__nativecode = apr_pcalloc(pool,sizeof(char) * PG_EMSG_LEN);
	stmt->dbconn = dbconn;
	stmt->__pool = pool;

	/* 関数へのポインタに実装を提供する */
	init_stmt_method(stmt);

	if (IS_EMPTY(sqltext)) {
		char *errmsg = "ERROR: sqltext is NULL.";
		apr_size_t len = strlen(errmsg) + 1;

		stmt->__code = DB_ERROR;
		apr_cpystrn(stmt->__msg, errmsg, PG_GET_EMSG_LEN(len));

		return stmt;
	}

	/* sqltextを解析する */
	stmt->stmtcntxt->prepared_parts = _prepare_query_string(pool, sqltext);

	/* オリジナルSQLへのポインタを退避(コピーしていない) */
	stmt->stmtcntxt->query = sqltext;

	stmt->__code = DB_SUCCESS;

	return stmt;
}

/**
 * トランザクションを開始する。
 *
 * @param dbconn DbConn
 * @param timeout const unsigned int
 *
 */
static void pg9_dbconn_startTrans(DbConn *dbconn, const unsigned int timeout)
{

	ExecStatusType status;
	PGresult   *pgres;
	apr_size_t codelen;
	apr_size_t msglen;

	pgres = PQexec(dbconn->dbcntxt->pgconn, "BEGIN");

	status = PQresultStatus(pgres);
	if (status == PGRES_COMMAND_OK){
		dbconn->__code = DB_SUCCESS;
	} else {
		codelen = strlen(PQresStatus(status));
		apr_cpystrn(dbconn->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
		msglen  = strlen(PQresultErrorMessage(pgres));
		apr_cpystrn(dbconn->__msg,PQresultErrorMessage(pgres), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *dbconn->__msg = '\0';
		dbconn->__code = DB_ERROR;
	}

	PQclear(pgres);
}

/**
 * トランザクションをコミットする。
 *
 * @param dbconn DbConn
 *
 */
static void pg9_dbconn_commit(DbConn *dbconn)
{

	ExecStatusType status;
	PGresult   *pgres;
	apr_size_t codelen;
	apr_size_t msglen;

	pgres = PQexec(dbconn->dbcntxt->pgconn, "COMMIT");

	status = PQresultStatus(pgres);
	if (status == PGRES_COMMAND_OK){
		dbconn->__code = DB_SUCCESS;
	} else {
		codelen = strlen(PQresStatus(status));
		apr_cpystrn(dbconn->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
		msglen  = strlen(PQresultErrorMessage(pgres));
		apr_cpystrn(dbconn->__msg,PQresultErrorMessage(pgres), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *dbconn->__msg = '\0';
		dbconn->__code	      = DB_ERROR;
	}


	PQclear(pgres);
}

/**
 * トランザクションをロールパックする。
 *
 * @param dbconn DbConn
 *
 */
static void pg9_dbconn_rollback(DbConn *dbconn)
{
	ExecStatusType status;
	PGresult   *pgres;
	apr_size_t codelen;
	apr_size_t msglen;

	pgres = PQexec(dbconn->dbcntxt->pgconn, "ROLLBACK");

	status = PQresultStatus(pgres);
	if (status == PGRES_COMMAND_OK){
		dbconn->__code = DB_SUCCESS;
	} else {
		codelen = strlen(PQresStatus(status));
		apr_cpystrn(dbconn->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
		msglen  = strlen(PQresultErrorMessage(pgres));
		apr_cpystrn(dbconn->__msg,PQresultErrorMessage(pgres), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *dbconn->__msg = '\0';
		dbconn->__code        = DB_ERROR;
	}


	PQclear(pgres);
}

/**
 * 現在のトランザクション特性の設定
 * postgresqlではIsolationLevel は READ_COMMITTED | SERIALIZABLE のみの為
 * それ以外の処理の場合はすべてデフォルトのREAD_COMMITTEDにする。
 *
 * @param dbconn DbConn*
 * @param level  const DbIsolationLv 
 *
 */
static void pg9_dbconn_setTransIsolation(DbConn *dbconn, const DbIsolationLv level)
{

	char sql[256] = "SET TRANSACTION ISOLATION LEVEL ";
	char *p;
	PGresult   *pgres;
	ExecStatusType status;
	apr_size_t codelen;
	apr_size_t msglen;

	p = sql;
	switch (level)
	{
			case TRANSACTION_SERIALIZABLE:
				strcat(p,"SERIALIZABLE");
				break;
			case TRANSACTION_NONE:
			case TRANSACTION_READ_UNCOMMITTED:
			case TRANSACTION_READ_COMMITTED:
			case TRANSACTION_REPEATABLE_READ:
			default:
				strcat(p,"READ COMMITTED");
				break;
	}

	pgres = PQexec(dbconn->dbcntxt->pgconn,p);

	status = PQresultStatus(pgres);
	if (status == PGRES_COMMAND_OK){
		dbconn->__code = DB_SUCCESS;
	} else {
		codelen = strlen(PQresStatus(status));
		apr_cpystrn(dbconn->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
		msglen  = strlen(PQresultErrorMessage(pgres));
		apr_cpystrn(dbconn->__msg,PQresultErrorMessage(pgres), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *dbconn->__msg = '\0';
		dbconn->__code = DB_ERROR;
	}

	PQclear(pgres);
}


/**
 * PostgreSQLに対して常に発行可能なSQL文(SELECT文)を取得する。
 *
 * @param dbconn DbConn *
 * @return const char * SQL文
 */
static const char * pg9_dbconn_getExecutableSQL(DbConn *dbconn)
{
	return "SELECT 1";
}

/**
 * 接続時に取得したコードを返す。
 *
 * @param dbconn const DbConn
 * @return dbconn->__code
 *
 */ 
static int pg9_dbconn_getCode(const DbConn *dbconn)
{
	return dbconn->__code;
}

/**
 * 接続時に取得したDBMSからのコードを返す。
 *
 * @param dbconn const DbConn
 * @return dbconn->__nativecode
 *
 */ 
static char *pg9_dbconn_getNativeCode(const DbConn *dbconn)
{
	return dbconn->__nativecode;
}

/**
 * 接続時に取得したメッセージを返す。
 *
 * @param dbconn const DbConn
 * @return dbconn->__msg
 *
 */ 
static char *pg9_dbconn_getMsg(const DbConn *dbconn)
{
	return dbconn->__msg;
}


/*
 * These functions are used for DbPreparedStmt structure.
 */

/**
 * stmtを閉じる（クリアする）
 *
 * @param stmt DbPreparedStmt
 *
 */
static void pg9_stmt_close(DbPreparedStmt *stmt)
{
	/* 何もしなくてよい */
}

/**
 * SQLを実行する（SELECT)
 *
 * @param stmt DbPreparedStmt*
 * @param pool apr_pool_t*
 *
 */
static DbResultSet *pg9_stmt_executeQuery(DbPreparedStmt *stmt, apr_pool_t *pool)
{
	return _pg9_stmt_executeQuery(stmt, pool, 1 /* カーソルを使う */);
}

/**
 * SQLを実行する（SELECT ... FOR UPDATE) (2005/05/16 Mon takehara 新規追加)
 *
 * @param stmt DbPreparedStmt*
 * @param pool apr_pool_t*
 *
 */
static DbResultSet *pg9_stmt_executeQueryForUpdate(DbPreparedStmt *stmt, apr_pool_t *pool)
{
//	return _pg9_stmt_executeQuery(stmt, pool, 0 /* カーソルを使わない */);
	return _pg9_stmt_executeQuery(stmt, pool, 1 /* カーソルを使う */);
}

/**
 * SQLを実行する。(UPDATE)
 *
 * @param stmt DbPreparedStmt*	
 * @param pool apr_pool_t*
 * @return retnum int		実行に影響した行数
 *
 */
static int pg9_stmt_executeUpdate(DbPreparedStmt *stmt, apr_pool_t *pool)
{

	ExecStatusType status;
	PGresult  *pgres;
	int retnum;
	apr_size_t codelen;
	apr_size_t msglen;
	const char *query;

	/* SQL文を組み立てる */
	query = _build_query_string(pool, stmt->stmtcntxt->prepared_parts);
	if (query == NULL) {
		char *errmsg = "ERROR: invalid bind value found.";
		msglen = strlen(errmsg) + 1;
		stmt->__code = DB_ERROR;
		apr_cpystrn(stmt->__msg, errmsg, PG_GET_EMSG_LEN(msglen));

		/* 次の検索に備えてバインド変数をクリアする */
		_clear_binded_value(pool, stmt->stmtcntxt->prepared_parts);

		return 0;
	}

	/* 次の検索に備えてバインド変数をクリアする(もう不要なので) */
	_clear_binded_value(pool, stmt->stmtcntxt->prepared_parts);

	/* SQLの実行 */
	pgres = PQexec(stmt->dbconn->dbcntxt->pgconn, query);

	status = PQresultStatus(pgres);
	if (status != PGRES_COMMAND_OK) {
		stmt->__code = DB_ERROR;
		codelen = strlen(PQresStatus(status));
		apr_cpystrn(stmt->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
		msglen  = strlen(PQresultErrorMessage(pgres));
		apr_cpystrn(stmt->__msg,PQresultErrorMessage(pgres), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *stmt->__msg = '\0';

		/* 次の検索に備えてバインド変数をクリアする */
		_clear_binded_value(pool, stmt->stmtcntxt->prepared_parts);

		return 0;
	}

	stmt->__code = DB_SUCCESS;
	retnum =  atoi(PQcmdTuples(pgres));
	PQclear(pgres);

	return retnum;

}

/**
 * int 型をsqltextのindex番目の?にセットする。
 *
 * @param stmt DbPreardStmt
 * @param index const int
 * @param value const apr_int32_t 32bit整数
 *
 */ 
static void pg9_stmt_setInt(DbPreparedStmt *stmt, const int index, const apr_int32_t value)
{

	char *bind;

	bind = apr_itoa(stmt->__pool, (int)value);
	set_value_query(stmt, index, DB_SQL_INT, bind);

}

/**
 * apr_int64_t型(long long)をsqltextのindex番目の?にセットする。
 *
 * @param stmt DbPreardStmt
 * @param index const int
 * @param value const apr_int64_t
 *
 */
static void pg9_stmt_setBigInt(DbPreparedStmt *stmt, const int index, const apr_int64_t value)
{
	char *bind; 

	bind = apr_psprintf(stmt->__pool,"%"APR_INT64_T_FMT,value);
	set_value_query(stmt, index, DB_SQL_BLG, bind);

}

/**
 * double型をsqltextのindex番目の?にセットする。
 *
 * @param stmt DbPreardStmt
 * @param index const int 
 * @param value const double
 *
 */
static void pg9_stmt_setDouble(DbPreparedStmt *stmt, const int index, const double value)
{
 	char *bind;

	bind = apr_psprintf(stmt->__pool,"%.16G",value);
	set_value_query(stmt, index, DB_SQL_FLT, bind);
}

/**
 * char型をsqltextのindex番目の?にセットする。
 *
 * @param stmt DbPreardStmt
 * @param index const int 
 * @param value const char*
 *
 */
static void pg9_stmt_setString(DbPreparedStmt *stmt, const int index, const char *value)
{
	set_value_query(stmt,index, DB_SQL_STR, value);
}

/* 26 E*/
static int pg9_stmt_getCode(const DbPreparedStmt *stmt)
{
	return stmt->__code;
}

/* 27 E*/
static char *pg9_stmt_getNativeCode(const DbPreparedStmt *stmt)
{
	return stmt->__nativecode;
}

/* 28 E*/
static char *pg9_stmt_getMsg(const DbPreparedStmt *stmt)
{
	return stmt->__msg;
}

static char * pg9_stmt_escWildCard(const DbPreparedStmt *stmt, const char *value)
{
	char *target = NULL, *first = NULL;

	if (value == NULL || *value == '\0') {
		return (char *) value;
	}

	/* esc_str は最大value の３倍になる */
	first = target = apr_pcalloc(stmt->__pool,
					sizeof(char) * (3*strlen(value) + 1));
	while(*value != '\0') {
		switch (*value) {
			case '%':
			case '_':
				/* 先頭に「\」を付ける */
				*target = '\\';
				target++;
				*target = *value;
				break;
			default:
				*target = *value;
				break;
		}
		target++;
		value++;
	}
	*target = '\0';

	return first;
}


/*
 * These functions are used for DbResultSet structure.
 */
/**
 * リザルトセットを閉じる 
 *
 * @param rset*
 * @return void
 *
 **/
static void pg9_rset_close(DbResultSet *rset)
{
	if (rset->rsetcntxt->transref > 0) {
		if (rset->rsetcntxt->pgres != NULL) {
			PQclear(rset->rsetcntxt->pgres);
			rset->rsetcntxt->pgres = NULL;
		}
		rset->rsetcntxt->transref--;
	}
}

/* 
 * カーソルを進める
 * postgresにはカレントカーソルの概念が存在しないために
 * contextにカレントタプル（curtup）と最大タプル(maxtup)
 * を保持し、maxtupになったときにDB_FALSEで処理を終了する
 * ようにする。
 *
 * @param 	rset 			DbResuleSet
 * @return	DB_[TRUE|FALSE]	int
 */
static int pg9_rset_next(DbResultSet *rset)
{
	/* 1件も無かった場合 */
	if (rset->rsetcntxt->maxtup == 0) {
		return DB_FALSE;
	}

	rset->rsetcntxt->curtup++;
	if (rset->rsetcntxt->curtup < rset->rsetcntxt->maxtup ) {
		return DB_TRUE;	/* 未だ続きがある */
	}

#ifdef PG_SUPPORT_PARTIAL_FETCH
	/* カーソルを使わないのならば残りはなし */
	if (!rset->rsetcntxt->use_cursor) {
		return DB_FALSE;
	}
	/* カーソルを使う場合 -> 続きを調べてみる */
	else {
		ExecStatusType status;

		/* FETCHして次のCURSORに進める */
		PQclear(rset->rsetcntxt->pgres);	/* 昔のpgres はクリア */
		rset->rsetcntxt->pgres = NULL;

		/* FETCHの実行 */
		rset->rsetcntxt->pgres = PQexec(rset->stmt->dbconn->dbcntxt->pgconn,
				apr_pstrcat(rset->__pool, "FETCH FORWARD "PG_MAX_FETCH_CNT" FROM ",
					rset->rsetcntxt->cursorname, NULL));

		/* ステータスの取得 */
		status = PQresultStatus(rset->rsetcntxt->pgres);
		if (status != PGRES_TUPLES_OK ){
			apr_size_t codelen;
			apr_size_t msglen;

			codelen = strlen(PQresStatus(status));
			apr_cpystrn(rset->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
			msglen  = strlen(PQresultErrorMessage(rset->rsetcntxt->pgres));
			apr_cpystrn(rset->__msg,PQresultErrorMessage(rset->rsetcntxt->pgres), PG_GET_EMSG_LEN(msglen));
			if (msglen == 0) *rset->__msg = '\0';
			rset->__code = DB_ERROR;
			PQclear(rset->rsetcntxt->pgres);
			rset->rsetcntxt->pgres = NULL;

			/* __msg に詰めたエラーは恐らく誰も出力しないので、ここで出しておく */
			LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Failed to execute FETCH. (Reason: %s)", rset->__msg);

			return DB_FALSE;
		}
		rset->__code = DB_SUCCESS;
		rset->rsetcntxt->maxtup = PQntuples(rset->rsetcntxt->pgres);

		if (rset->rsetcntxt->maxtup > 0) {
			rset->rsetcntxt->curtup = 0;	/* 先頭を指すようにする */
			return DB_TRUE;	/* まだ続く */
		}
		else {
			return DB_FALSE;	/* もうない */
		}
	}
#else	/* !PG_SUPPORT_PARTIAL_FETCH */
	/* ここに来るのは有効なデータが無い時だけ */
	return DB_FALSE;
#endif	/* PG_SUPPORT_PARTIAL_FETCH */
}

/**
 * 検索したカラムの数を返す。
 * @param rset DbResultSet*
 * @retuen rset->__columnCount int
 */
static int pg9_rset_getColumnCount(DbResultSet *rset)
{
	return rset->__columnCount;
}

/**
 * 検索したカラムのカラム名を返す。
 * @param rset DbResultSet*
 * @param index const int
 * 
 */
static char * pg9_rset_getColumnName(DbResultSet *rset, const int index)
{
	return rset->__columnNames[index - 1];
}

/**
 * 検索したカラムのタイプを返す。
 * (note)
 * 	このプロバイダは、常にDB_SQL_UNCLEAR を返却します。
 * 	理由の詳細は関数pg_stmt_executeQueryを参照して下さい。
 *
 * @param rset DbResultSet*
 * @param index const int
 * @return DbSqlType 常にDB_SQL_UNCLEAR
 *
 */
static DbSqlType pg9_rset_getColumnType(DbResultSet *rset, const int index)
{
	return rset->__columnTypes[index - 1];
}

/**
 * 検索したカラムのサイズを返す。
 * postgresは可変長の場合戻りを-1を返す為あまりこれは意味がない。
 * @param rset DbResultSet*
 * @param index const int
 *
 */
static int pg9_rset_getColumnSize(DbResultSet *rset, const int index)
{
	return rset->__columnSizes[index - 1];
}

/**
 * int型のデータの取得
 * indexで指定されたオフセットの内容をintで取得する。
 * @param rset  DbResultSet*
 * @param index int
 * @return value apr_int32_t 32bit整数値
 *
 */
static apr_int32_t pg9_rset_getInt(DbResultSet *rset, const int index)
{
	char *buf;

	buf    = PQgetvalue(rset->rsetcntxt->pgres,
				rset->rsetcntxt->curtup, index - 1);
	return (apr_int32_t) atoi(buf);
}

/**
 * long long(apr_int64_t)型のデータの取得
 * indexで指定されたオフセットの内容をlong long(apr_int64_t)で取得する。
 * @param rset  DbResultSet*
 * @param index int
 * @return value apr_atoi64_t
 *
 */
static apr_int64_t pg9_rset_getBigInt(DbResultSet *rset, const int index)
{
	char* buf;

	buf    = PQgetvalue(rset->rsetcntxt->pgres, 
				rset->rsetcntxt->curtup, index - 1);

	/* (note) NULL をapr_atoi64 に入力するとSIGSEGV するので先に逃がす */
	if (buf == NULL) {
		return APR_INT64_C(0);
	}

	return apr_atoi64(buf);
}

/**
 * double型のデータの取得
 * indexで指定されたオフセットの内容をdoubleで取得する。
 * @param rset  DbResultSet*
 * @param index int
 * @return value double
 *
 */
static double pg9_rset_getDouble(DbResultSet *rset, const int index)
{
	char* buf;

	buf    = PQgetvalue(rset->rsetcntxt->pgres, 
				rset->rsetcntxt->curtup, index - 1);
	return atof(buf);
}

/**
 *
 * indexで指定されたオフセットの内容をcharで取得する。
 * @param rset  DbResultSet*
 * @param index int
 * @return value string
 *
 */
static char *pg9_rset_getString(DbResultSet *rset, const int index)
{
	/*
	 * 値がNULLの場合はNULLを返すようにする。
	 */
	if (PQgetisnull(rset->rsetcntxt->pgres, rset->rsetcntxt->curtup,
							index -1) == 1) {
		return NULL;
	}

	return  apr_pstrdup(rset->__scratchpool, PQgetvalue(rset->rsetcntxt->pgres,
					   	      rset->rsetcntxt->curtup,
						      index - 1));
}

/* 38 E*/
static int pg9_rset_getCode(const DbResultSet *rset)
{
	return rset->__code;
}

/* 39 E*/
static char *pg9_rset_getNativeCode(const DbResultSet *rset)
{
	return rset->__nativecode;
}

/* 40 E*/
static char *pg9_rset_getMsg(const DbResultSet *rset)
{
	return rset->__msg;
}

/*--------------------------------------------------------------
  Define private functions
  --------------------------------------------------------------*/
/**
 * DbDataSourceの中で宣言された関数へのポインタに実体の関数を提供する.
 * この関数は、DBプロバイダを登録する際に
 *
 * @param dbds DbDataSource*
 */
static void init_dbds_method(DbDataSource *dbds)
{
	dbds->getDbConn     = pg9_dbds_getDbConn;
	dbds->rebuildConn   = pg9_dbds_rebuildConn;
	dbds->getCode       = pg9_dbds_getCode;
	dbds->getNativeCode = pg9_dbds_getNativeCode;
	dbds->getMsg        = pg9_dbds_getMsg;
}

/**
 * DbConnの中で宣言された関数へのポインタに実体の関数を提供する.
 *
 * @param dbconn DbConn*
 */
static void init_dbconn_method(DbConn *dbconn)
{
	dbconn->close             = pg9_dbconn_close;
	dbconn->isClosed          = pg9_dbconn_isClosed;
	dbconn->prepareStatement  = pg9_dbconn_prepareStatement;
	dbconn->startTrans        = pg9_dbconn_startTrans;
	dbconn->commit            = pg9_dbconn_commit;
	dbconn->rollback          = pg9_dbconn_rollback;
	dbconn->setTransIsolation = pg9_dbconn_setTransIsolation;
	dbconn->getExecutableSQL  = pg9_dbconn_getExecutableSQL;
	dbconn->getCode           = pg9_dbconn_getCode;
	dbconn->getNativeCode     = pg9_dbconn_getNativeCode;
	dbconn->getMsg            = pg9_dbconn_getMsg;
}

/**
 * DbPreparedStmtの中で宣言された関数へのポインタに実体の関数を提供する.
 *
 * @param stmt DbPreparedStmt*
 */
static void init_stmt_method(DbPreparedStmt *stmt)
{
	stmt->close         = pg9_stmt_close;
	stmt->executeQuery  = pg9_stmt_executeQuery;
	stmt->executeQueryForUpdate  = pg9_stmt_executeQueryForUpdate;
	stmt->executeUpdate = pg9_stmt_executeUpdate;
	stmt->setInt        = pg9_stmt_setInt;
	stmt->setBigInt     = pg9_stmt_setBigInt;
	stmt->setDouble     = pg9_stmt_setDouble;
	stmt->setString     = pg9_stmt_setString;
	stmt->getCode       = pg9_stmt_getCode;
	stmt->getNativeCode = pg9_stmt_getNativeCode;
	stmt->getMsg        = pg9_stmt_getMsg;
	stmt->escWildCard   = pg9_stmt_escWildCard;
}

/**
 * DbResultSetの中で宣言された関数へのポインタに実体の関数を提供する.
 *
 * @param rset DbResultSet*
 */
static void init_rset_method(DbResultSet *rset)
{
	rset->close         = pg9_rset_close;
	rset->next          = pg9_rset_next;
	rset->getColumnCount= pg9_rset_getColumnCount;
	rset->getColumnName = pg9_rset_getColumnName;
	rset->getColumnType = pg9_rset_getColumnType;
	rset->getColumnSize = pg9_rset_getColumnSize;
	rset->getInt        = pg9_rset_getInt;
	rset->getBigInt     = pg9_rset_getBigInt;
	rset->getDouble     = pg9_rset_getDouble;
	rset->getString     = pg9_rset_getString;
	rset->getCode       = pg9_rset_getCode;
	rset->getNativeCode = pg9_rset_getNativeCode;
	rset->getMsg        = pg9_rset_getMsg;
}

/**
 * setXXX系のvalueを文字列に変換し、sqltxtにあるindex番目の?をvalueで置換する。
 *
 * @param stmt DbPreparedStmt *
 * @param index const int インデックス番号
 * @param type const DbSqlType valueの型
 * @param value const char * バインドする値
 */
static void set_value_query(DbPreparedStmt *stmt, const int index,
				const DbSqlType type, const char *value)
{
	pg9_prepared_part *part;
	divy_array_t *prepared_parts = stmt->stmtcntxt->prepared_parts;
	int err = 0;

	if (prepared_parts == NULL ||
		index > divy_array_getlength(prepared_parts) || index <= 0) {
		return;	/* 何も出来ない(後でエラーになるので何もしない) */
	}

	/* 対応するSQLパートを取り出す */
	part = divy_array_get(prepared_parts, index - 1);
	if (part == NULL) {
		return;	/* 何も出来ない(後でエラーになるので何もしない) */
	}

	if (value != NULL) {
		apr_size_t len = strlen(value);
		char *esc_value = apr_pcalloc(stmt->__pool,
				sizeof(char) * (len * 2 + 1));

		/* value に含まれる値を文字エスケープする */
		(void) PQescapeStringConn(stmt->dbconn->dbcntxt->pgconn,
								  		esc_value, value, len, &err);

		/* バインド変数として保持する */
		part->bind = esc_value;
		part->type = type;
	}
	else {
		part->bind = NULL;
	}
}

/**
 * クエリー文字列内の半角の"？"位置を解析し、プリペアされたSQLを生成する。
 * (note)
 * 	文字列として指定された"?"(シングルクオートで囲まれた"?")は、バインド変数
 * 	として抽出されません。
 *
 * @param pool apr_poot_t * 作業用のプール
 * @param sql const char * 解析対象のSQL文
 * @return divy_array_t * プリペアされたSQL(pg_prepared_part)を持つ配列
 */
static divy_array_t * _prepare_query_string(apr_pool_t *p, const char *sql)
{
	const char *sp, *prev_sp;
	apr_size_t len;
	pg9_prepared_part *part;
	divy_array_t *query_parts = NULL;

	if (sql == NULL || *sql == '\0') return NULL;

	/* 配列を作成する */
	query_parts = divy_array_make(p, 10);

	prev_sp = sp = sql;
	len = strlen(sql);

	while(1) {
		/* 配列の要素を1つ作成する */
		part = apr_palloc(p, sizeof(pg9_prepared_part));
		divy_array_add(query_parts, part);

		part->bind = PG_BIND_INIT_STR;	/* 初期化 */

		/* '?'を探す */
		sp = dav_divy_find_bindposition(sp);
		if (sp) {
			/* '?'までのSQL文字列をコピー */
			part->sql  = apr_pstrmemdup(p, prev_sp, (sp - prev_sp));
			part->has_bind = 1;	/* バインド変数が存在した */
		}
		else {
			/* 最後までをコピーする */
			part->sql  = apr_pstrmemdup(p, prev_sp, (&sql[len] - prev_sp));
			part->has_bind = 0;	/* バインド変数がない */
			break;	/* 終了 */
		}
		sp++;	/* '?' を1つ飛ばす */
		if (*sp == '\0') {
			break;
		}
		prev_sp = sp;
	}

	return query_parts;
}

/**
 * prepared_parts が示すSQLの部分集合を組み合わせてSQL文を組み立てる。
 *
 * @param p apr_pool_t *
 * @param prepared_parts divy_array_t * SQLの断片からなる配列
 * @return const char * 組み立てられたSQL
 * 		NULLの場合には組み立てに失敗したということ。
 */
static const char * _build_query_string(apr_pool_t *p, divy_array_t *prepared_parts)
{
	int i;
	pg9_prepared_part *part;
	divy_sbuf *sbuf = NULL;
	apr_int32_t length;

	if (prepared_parts == NULL ||
		(length = divy_array_getlength(prepared_parts)) == 0) {
		return NULL;
	}

	divy_sbuf_create(p, &sbuf, 512);

	/* sqlとbind を単純に連結する */
	for (i = 0; i < length; i++) {
		part = divy_array_get(prepared_parts, i);

		divy_sbuf_append(sbuf, part->sql);
		if (part->has_bind) {	/* バインド変数があったとき */
			if (part->bind) {
				/* 数値系バインド変数の場合 */
				if (part->type == DB_SQL_INT ||
				    part->type == DB_SQL_BLG ||
				    part->type == DB_SQL_FLT) {
					divy_sbuf_append(sbuf, part->bind);
				}
				/* 上記以外(つまり文字列)の場合 */
				else {
					divy_sbuf_appendbyte(sbuf, 1, "\'");
					divy_sbuf_append(sbuf, part->bind);
					divy_sbuf_appendbyte(sbuf, 1, "\'");
				}
			}
			else {
				/* "NULL"をバインドする */
				divy_sbuf_appendbyte(sbuf, 4, "NULL");
			}
		}
	}

	return (const char *) divy_sbuf_tostring(sbuf);
}

/**
 * prepared_parts が持つバインド変数の値をクリアし、再バインド可能な状態にする。
 *
 * @param p apr_pool_t *
 * @param prepared_parts divy_array_t * バインド変数を保持する配列
 */
static void _clear_binded_value(apr_pool_t *p, divy_array_t *prepared_parts)
{
	int i;
	pg9_prepared_part *part;
	apr_int32_t length;

	if (prepared_parts == NULL || 
		(length = divy_array_getlength(prepared_parts)) == 0) {
		return;
	}

	for (i = 0; i < length; i++) {
		part = divy_array_get(prepared_parts, i);
		part->bind = PG_BIND_INIT_STR;
	}
}

/**
 * SQLを実行する（SELECT)
 *
 * @param stmt DbPreparedStmt *
 * @param pool apr_pool_t *
 * @param use_cursor int カーソルを利用するかどうか (1: 利用する / 0: 利用しない)
 * @return DbResultSet * 生成した結果セットへのポインタ
 */
static DbResultSet * _pg9_stmt_executeQuery(DbPreparedStmt *stmt, apr_pool_t *pool, int use_cursor)
{
	DbResultSet *rset;
	ExecStatusType status;
	PGresult   *pgres;
	int 	    i;
	char 	    *sqlstmt;
	char	    *cursorname;	/* カーソルの名前	*/
	apr_size_t codelen;
	apr_size_t msglen;
	const char *query;
	
	rset = (DbResultSet *)apr_pcalloc(pool, sizeof(DbResultSet));
	rset->rsetcntxt = (DbRsetCntxt *)apr_pcalloc(pool,sizeof(DbRsetCntxt));
	rset->rsetcntxt->pgres = NULL;
	rset->rsetcntxt->transref = 0;
	rset->__msg = apr_palloc(pool,sizeof(char) * PG_EMSG_LEN);
	rset->__nativecode = apr_palloc(pool,sizeof(char) * PG_EMSG_LEN);
	rset->stmt = stmt;
	rset->__pool = pool;

	/* __scratchpool のデフォルト値を設定する */
	rset->__scratchpool = pool;

#ifdef PG_SUPPORT_PARTIAL_FETCH
	/* カーソルを利用するかどうかを記録する */
	rset->rsetcntxt->use_cursor = use_cursor;
#endif	/* PG_SUPPORT_PARTIAL_FETCH */

	/* 関数へのポインタに実装を提供する */
	init_rset_method(rset);

	/* SQL文を組み立てる */
	query = _build_query_string(pool, stmt->stmtcntxt->prepared_parts);
	if (query == NULL) {
		char *errmsg = "ERROR: invalid bind value found.";
		msglen = strlen(errmsg) + 1;
		rset->__code = DB_ERROR;
		apr_cpystrn(rset->__msg, errmsg, PG_GET_EMSG_LEN(msglen));

		/* 次の検索に備えてバインド変数をクリアする */
		_clear_binded_value(pool, stmt->stmtcntxt->prepared_parts);

		return rset;
	}

	/* 次の検索に備えてバインド変数をクリアする(もう不要なので) */
	_clear_binded_value(pool, stmt->stmtcntxt->prepared_parts);

	/* SQL実行 */
	sqlstmt = apr_pstrcat(pool, "select * from ( " , query,") T WHERE false",NULL);

	pgres = PQexec(stmt->dbconn->dbcntxt->pgconn,sqlstmt);

	status = PQresultStatus(pgres);
	if (status != PGRES_TUPLES_OK ){
		codelen = strlen(PQresStatus(status));
		apr_cpystrn(rset->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
		msglen  = strlen(PQresultErrorMessage(pgres));
		apr_cpystrn(rset->__msg,PQresultErrorMessage(pgres), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *rset->__msg = '\0';
		rset->__code = DB_ERROR;
		PQclear(pgres);

		return rset;
	}

	/*
	 * フィールドの領域確保をpoolより行い、フィールドタイプ、フィールド名、
	 * フィールドサイズを設定する。
	 *
	 * Postgresqlのカラム番号は 0(zero) が基点となる。
	 *
	 */
	rset->__columnCount = PQnfields(pgres);
	rset->__columnTypes = apr_palloc(pool,
				sizeof(DbSqlType *) * rset->__columnCount);
	rset->__columnNames = apr_palloc(pool,
				sizeof(char *)      * rset->__columnCount);
	rset->__columnSizes = apr_palloc(pool,
				sizeof(int)         * rset->__columnCount);

	for (i=0; i<rset->__columnCount; i++) {

	/*
	 * (note)
	 * カラムデータ型の取得ロジックは以下の理由で導入を断念しました。
	 * 将来良い方法が見つかったらサポートして下さい。
	 *
	 * (1) PostgreSQLにアクセスするDBクライアントはPQftypeから戻される
	 *     Oidの値をカタログテーブルpg_typeに問い合わせないとどのデータ型
	 *     なのか分からない。
	 *     そうかといって、問い合わせを出すとパフォーマンスに影響を与える。
	 * (2) 以下のロジックのようにPostgreSQLサーバの内部で使用している
	 *     INT2OIDなどの定義値を取り込んでしまうと、異なるバージョンの
	 *     PostgreSQLサーバではOidに別々の値が与えられている場合があり得るので
	 *     偽りのSQL型を返してしまう。(クライアント、サーバ共に同じPostgreSQL
	 *     であれば問題なし)
	 * (3) PostgreSQLサーバが内部的に使用している定義値を取り込むために
	 *     ヘッダをinclude すると、本来見えなくてもよい定義値や定義も一緒に
	 *     取り込まれてしまう。このときApacheのヘッダで定義されている定義値
	 *     との2重定義問題が発生していた。
	 */
#if 0
		switch(PQftype(pgres,i))
		{
			case INT2OID:
				rset->__columnTypes[i] = DB_SQL_INT;
				break;

			case INT4OID:
				rset->__columnTypes[i] = DB_SQL_LNG;
				break;

			case INT8OID:
				rset->__columnTypes[i] = DB_SQL_BLG;
				break;

			case FLOAT8OID:
				rset->__columnTypes[i] = DB_SQL_FLT;
				break;

			case ABSTIMEOID:
				rset->__columnTypes[i] = DB_SQL_DATE;
				break;
	
			case TIMESTAMPOID:
				rset->__columnTypes[i] = DB_SQL_TIMESTAMP;
				break;

			default:
				rset->__columnTypes[i] = DB_SQL_STR;
				break;
		}
#else
		rset->__columnTypes[i] = DB_SQL_UNCLEAR;	/* 理由は上の通り */
#endif
		rset->__columnNames[i] = apr_pstrdup(pool,PQfname(pgres,i));
		rset->__columnSizes[i] = PQfsize(pgres,i);
	}

	/* 属性を取得した為、初期化を行う */
	PQclear(pgres);

	/* カーソルを使う場合 */
	if (use_cursor) {
		/* (note)
		 * (2004/07/29 Thu takehara) CURSOR はSELECT - FOR UPDATE が使用できません。
		 */
		/*
		 * カーソル作成 
		 * postgresqlではDECARE カーソルの名前 CURSOR FORでカーソル作成する
		 */
		cursorname = apr_psprintf(pool, "CURSOR_%d", (stmt->dbconn->dbcntxt->cursorcnt)++);
#ifdef PG_SUPPORT_PARTIAL_FETCH
		rset->rsetcntxt->cursorname = cursorname;	/* 記録しておく */
#endif	/* PG_SUPPORT_PARTIAL_FETCH */

		sqlstmt = apr_pstrcat(pool, "DECLARE ", cursorname, " NO SCROLL CURSOR FOR ", query, NULL);

		pgres = PQexec(stmt->dbconn->dbcntxt->pgconn,sqlstmt);
		status = PQresultStatus(pgres);
		if (status != PGRES_COMMAND_OK ){
			codelen = strlen(PQresStatus(status));
			apr_cpystrn(rset->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
			msglen  = strlen(PQresultErrorMessage(pgres));
			apr_cpystrn(rset->__msg,PQresultErrorMessage(pgres), PG_GET_EMSG_LEN(msglen));
			if (msglen == 0) *rset->__msg = '\0';
			rset->__code = DB_ERROR;
			PQclear(pgres);
			return rset;
		}

		/*
		 * カーソル用のpgresはクリアする。ここ以降はローカルでpgresは
		 * 利用しない。rsetcntxt内のpgresを利用する。
		 */
		PQclear(pgres);

#ifdef PG_SUPPORT_PARTIAL_FETCH
		rset->rsetcntxt->pgres = PQexec(stmt->dbconn->dbcntxt->pgconn,
				apr_pstrcat(pool, "FETCH FORWARD "PG_MAX_FETCH_CNT" FROM ",
					cursorname, NULL));

#else	/* !PG_SUPPORT_PARTIAL_FETCH */
		rset->rsetcntxt->pgres = PQexec(stmt->dbconn->dbcntxt->pgconn,
				apr_pstrcat(pool, "FETCH ALL IN ",
					cursorname, NULL));
#endif	/* PG_SUPPORT_PARTIAL_FETCH */
	}
	/* カーソルを使わない場合 */
	else {
		rset->rsetcntxt->pgres = PQexec(stmt->dbconn->dbcntxt->pgconn, query);
	}

	status = PQresultStatus(rset->rsetcntxt->pgres);
	if (status != PGRES_TUPLES_OK ){
		codelen = strlen(PQresStatus(status));
		apr_cpystrn(rset->__nativecode,PQresStatus(status), PG_GET_EMSG_LEN(codelen));
		msglen  = strlen(PQresultErrorMessage(rset->rsetcntxt->pgres));
		apr_cpystrn(rset->__msg,PQresultErrorMessage(rset->rsetcntxt->pgres), PG_GET_EMSG_LEN(msglen));
		if (msglen == 0) *rset->__msg = '\0';
		rset->__code = DB_ERROR;
		PQclear(rset->rsetcntxt->pgres);
		rset->rsetcntxt->pgres = NULL;
		return rset;
	}

	/*
	 * fetchを他で行えるように、最大行数（tuple count)とカレント番号
	 * を保持する。
	 * カレント番号(curtup)は初期値を-1としているのは一番初めのフェッチ
	 * で0からデータを取得する為。
	 */
	rset->rsetcntxt->maxtup = PQntuples(rset->rsetcntxt->pgres);
	rset->rsetcntxt->curtup = -1;
	rset->__code = DB_SUCCESS;

	rset->rsetcntxt->transref = 1;

	return rset;
}

/*-----------------------------------------------------------------------------
  Define provider library handler
 *---------------------------------------------------------------------------*/
#if defined(DIVY_LIB_DYNAMIC_EXPORT)
/**
 * スタートアップハンドラ (コンストラクタ関数)
 * (note)
 * 	このハンドラはdlopen 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	コールの直前に呼び出されます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlopen された時だけ実施する特別な初期化処理
 * 	    * 関数divy_register_lib_provider のコール
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _db_pg9_init(void)
#else
void _init(void)
#endif	/* DIVY_USE_GNU_LD */
{
	/* プロバイダライブラリ構造体の登録 */
	divy_register_lib_provider(&DIVY_PROVIDER_HOOK_STRUCT_NAME(db,pg9));
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"pg(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
}

/**
 * 終了ハンドラ (デストラクタ関数)
 * (note)
 * 	このハンドラはdlclose 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	が終了した後、コールされます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlclose された時だけ実施する特別な終了処理
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _db_pg9_fini(void)
#else
void _fini(void)
#endif	/* DIVY_USE_GNU_LD */
{
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"pg(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return;
}
#endif	/* DIVY_LIB_DYNAMIC_EXPORT */

/**
 * 初期化ハンドラ
 * (note)
 * 	このハンドラは、Apacheモジュールの一部として動作する場合には
 * 	post_config ステージ(まだChildも起動していないステージ) で
 * 	呼び出されます。
 *
 * 	Child 起動前に実施しなければならない処理や、pconf を使った
 * 	処理を記述できます。
 * 	Child_init でコールバックを受け取るには、このハンドラの
 * 	中で関数divy_hook_child_init を使ってコールバックハンドラを
 * 	レジスタして下さい。
 *
 * @pconf apr_pool_t * コンフィグプール
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
static int _db_pg9_init_func(apr_pool_t *pconf)
{
	/* create_dbdatasource ハンドラへの登録 */
	divy_hook_create_dbdatasource(_db_pg9_create_dbdatasource,
					NULL, NULL, APR_HOOK_MIDDLE);
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"pg(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return DIVY_LIB_ST_OK;
}

/**
 * 指定されたproviderType が自身のDBプロバイダを示していた時、
 * 自身のDbDataSource を生成してdbds に返却する。
 *
 * @param p apr_pool_t *
 * @param providerType const char * プロバイダタイプ
 * @param dbds DbDataSource ** 作成したDbDataSource構造体
 * @return
 * 	DB_OK       : providerType が示すプロバイダは自分自身、かつ処理が成功した
 * 	DB_DECLINED : providerType が示すプロバイダは自分ではなかった
 */
static int _db_pg9_create_dbdatasource(apr_pool_t *p,
					const char *providerType,
					DbDataSource **dbds)
{
	*dbds = NULL;

	/*
	 * providerType は自分を示しているか？
	 */
	if (providerType == NULL ||
	    strcmp(providerType, PG_DB_PROVIDER_TYPE) != 0) {
		return DB_DECLINED;
	}

	/*
	 * プロバイダ構造体を作成する
	 */
	*dbds = pg9_dbds_createDbds(p);

	return DB_OK;
}


