/*
 * TeamFile Server Module DownLoader
 * $Id$
 * gettfmodule.c
 *
 * copyrigth(c) TeamFile team
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>

#include <curl/curl.h>

typedef struct tfdlctx {
	CURL *curl;

	char *user;
	char *pass;
	char *host;
	char *rootpath;
	char *location;
	char *dfile;

	char baseurl[256];

	char *proxy_host;
	char *proxy_user;
	char *proxy_pass;
	char usrpwd[256];
	char proxy_usrpwd[256];
	int secure;
	int verbose;
	unsigned short port;
	unsigned short proxy_port;
	int use_filelist;
} tfdlctx;

/* 戻り値 */
enum TFDL_RESULT{
	TFDLOK          =  0,	/* 成功 */
	TFDLINVPARAM    =  1,	/* 不正なコマンドラインパラメータが指定された */
	TFDLCURLFAIL	=  2,	/* CURLの失敗 */
	TFDLAUTHERR     = 10,	/* 認証エラー */
	TFDLCONFAILED   = 11,	/* 接続失敗 */
	TFDLNOTFOUND    = 12,	/* ファイルが無かった */
	TFDLFILELISTERR = 13,	/* ファイルリストのPROPFINDに失敗した */ 
	TFDLDIRNOTFOUND = 20,	/* ダウンロードフォルダがない */
	TFDLWRITEFAILED = 21,	/* 書き込みエラー */
	TFDLDOWNLOADMOD = 22,	/* 更新モジュールのGETに失敗した */
	TFDLGENERICERR  = 99	/* 不明なエラー */
};

/* filelist */
#define FILELIST	"filelist"

/* User-Agentの名前 */
#define AGENTNAME	"TeamFile Updater"

/* デフォルトのダウンロードホスト */
#define DEFDOWNLOADHOST	"dl.teamfile.com"

/* デフォルトのダウンロードポート */
#define DEFDOWNLOADPORT 443

/* デフォルトのダウンロードパス */
#define DEFDOWNLOADPATH	"/var/lib/teamfile/download/"

/* デフォルトのダウンロードロケーション */
#define DEFDOWNLOADLOCATION "/support/Group Folder/download/"

static int initialize(tfdlctx *ctx);
static int terminate(tfdlctx *ctx);
static int setparamall(tfdlctx *ctx);
static void setbuildurl(tfdlctx *ctx);
static int downloadurl(tfdlctx* ctx, char* target);

int main(int argc, char *argv[])
{
	int ch;
	int ret;
	setlocale(LC_ALL, NULL);

	tfdlctx ctx = {0};

	curl_global_init(CURL_GLOBAL_DEFAULT);
	if (initialize(&ctx) != 0) return TFDLCONFAILED;

	while ((ch = getopt(argc, argv, "b:c:d:e:U:P:L:O:Sh:p:l:vf:")) != -1) {
		switch(ch)
		{
			case 'U':	/* ユーザ */
				ctx.user = strdup(optarg);
				break;
			case 'P':	/* パスワード */
				ctx.pass = strdup(optarg);
				break;
			case 'h':	/* ホスト */
				ctx.host = strdup(optarg);
				break;
			case 'S':	/* SSL */
				ctx.secure = 1;	
				break;
			case 'p':	/* ポート */
				ctx.port = (unsigned short)atoi(optarg);
				break;
			case 'O':	/* 出力パス */
				ctx.rootpath = strdup(optarg);
				break;
			case 'l':	/* ロケーション */
				ctx.location = strdup(optarg);
				break;
			case 'b':	/* プロキシサーバ */
				ctx.proxy_host = strdup(optarg);
				break;
			case 'c':	/* プロキシポート番号 */
				ctx.proxy_port = (unsigned short)atoi(optarg);
				break;
			case 'd':	/* プロキシ認証ユーザ */
				ctx.proxy_user = strdup(optarg);
				break;
			case 'e':	/* プロキシ認証パスワード */
				ctx.proxy_pass = strdup(optarg);
				break;
			case 'v':	/* 詳細表示 */
				ctx.verbose = 1;
				break;
			case 'f':
				ctx.dfile = strdup(optarg);
				break;
			default:
				return TFDLINVPARAM;
		}
	}

	/* パラメーターをすべて設定する */
	if (setparamall(&ctx)) return TFDLINVPARAM;

	/* ローカルパスを調べる */
	if (opendir(ctx.rootpath) == NULL) {
		return TFDLDIRNOTFOUND;
	}

	/* build baseurl */
	setbuildurl(&ctx);

	/* download filelist */
	if (ctx.use_filelist)
		ret = downloadurl(&ctx, FILELIST);
	else
		ret = downloadurl(&ctx, ctx.dfile);

	switch(ret) {
		case CURLE_OK:
			break;

		case CURLE_COULDNT_CONNECT:
			return TFDLCONFAILED;
			break;

		default:
			return TFDLFILELISTERR;
	}

	/*  ファイルリストを利用するならファイル内部のファイルを取得する */
	if (ctx.use_filelist) {
		char path[PATH_MAX];
		char line[256];
		char *p;
		char prg[256];

		sprintf(path, "%s/%s", ctx.rootpath, FILELIST);
		FILE* fp = fopen(path, "r");
		if (fp == NULL) return TFDLWRITEFAILED;

		while(fgets(line, sizeof(line), fp)) {
			if (*line == '#') continue;
			p = strchr(line, ',');
			memset(prg, 0, 256);
			if (p != NULL) {
				/* found rpm */
				memcpy(prg, line, p-&line[0]);
				ret = downloadurl(&ctx, prg);
				switch(ret) {
					case CURLE_OK:
						break;

					case CURLE_HTTP_NOT_FOUND:
						return TFDLNOTFOUND;
						break;

					default:
						return TFDLDOWNLOADMOD;
				}
			}
		}

		fclose(fp);
	}

	/* cleanup */
	terminate(&ctx);
	curl_global_cleanup();

	return TFDLOK;
}

static int initialize(tfdlctx *ctx) {
	ctx->curl = curl_easy_init();
	
	if (ctx->curl == NULL) return 1;

	return 0;
}

static int terminate(tfdlctx *ctx) {
	curl_easy_cleanup(ctx->curl);
	return 0;
}

static int setparamall(tfdlctx *ctx) {

	if (ctx->host == NULL)     ctx->host     = DEFDOWNLOADHOST;
	if (ctx->port == 0)        ctx->port     = DEFDOWNLOADPORT;
	if (ctx->rootpath == NULL) ctx->rootpath = DEFDOWNLOADPATH;
	if (ctx->location == NULL) ctx->location = DEFDOWNLOADLOCATION;
	if (ctx->dfile == NULL)    ctx->use_filelist = 1;

	/* set user credentials */
	if (ctx->user == NULL || ctx->pass == NULL) return 1;
	sprintf(ctx->usrpwd, "%s:%s", ctx->user, ctx->pass);
	curl_easy_setopt(ctx->curl, CURLOPT_USERPWD, ctx->usrpwd);

	/* set cURL options */
	/* user agenet */
	curl_easy_setopt(ctx->curl, CURLOPT_USERAGENT, AGENTNAME);

	/* use ssl */
	if (ctx->secure) 
		curl_easy_setopt(ctx->curl, CURLOPT_SSL_VERIFYPEER, 0L);

	/* use proxy */
	if (ctx->proxy_host != NULL) {
		curl_easy_setopt(ctx->curl, CURLOPT_PROXY, ctx->proxy_host);

		if (ctx->proxy_port == 0)  ctx->proxy_port = 3128;	/* default */
		curl_easy_setopt(ctx->curl, CURLOPT_PROXYPORT, ctx->proxy_port);

		if (ctx->proxy_user != NULL && ctx->proxy_pass != NULL) {
			sprintf(ctx->proxy_usrpwd, "%s:%s", ctx->proxy_user, ctx->proxy_pass);
			curl_easy_setopt(ctx->curl, CURLOPT_PROXYUSERPWD, ctx->proxy_usrpwd);
		}
	}

	/* verbose */
	curl_easy_setopt(ctx->curl, CURLOPT_VERBOSE, ctx->verbose);

	return 0;

}

static void setbuildurl(tfdlctx *ctx) {
	memset(ctx->baseurl, 0, 256);
	sprintf(ctx->baseurl, "http%s://%s:%d/", (ctx->secure)?"s":"",
													ctx->host, ctx->port); 
}

/*
 * 指定されたファイルをダウンロードする
 * @param ctx tfdlctx*
 * @param target char *	ダウンロードファイル
 * @return int 0: 成功
 *
 */
static int downloadurl(tfdlctx* ctx, char* target) {

	CURLcode res;
	char url[256] = {0};
	char localpath[256] = {0};
	char tmp[256] = {0};
	struct stat buf;

	sprintf(url, "%s%s/%s", ctx->baseurl, ctx->location, target);
	sprintf(localpath, "%s/%s", ctx->rootpath, target);

	/* 一時ファイルを作成する */
	sprintf(tmp, "%s/%s.XXXXXX", ctx->rootpath, target);
	mkstemp(tmp);

	FILE *fp = fopen(tmp, "w");
	curl_easy_setopt(ctx->curl, CURLOPT_FAILONERROR, 1L);
	curl_easy_setopt(ctx->curl, CURLOPT_URL, url);
	curl_easy_setopt(ctx->curl, CURLOPT_WRITEFUNCTION, fwrite);
	curl_easy_setopt(ctx->curl, CURLOPT_WRITEDATA, fp);
	if (stat(localpath, &buf)==0) {
		curl_easy_setopt(ctx->curl, CURLOPT_TIMECONDITION, 1L);
		curl_easy_setopt(ctx->curl, CURLOPT_TIMEVALUE, buf.st_mtime);
	}

	res = curl_easy_perform(ctx->curl);
	fclose(fp);

	if (res == CURLE_OK) {
		int code = 0;
		curl_easy_getinfo(ctx->curl, CURLINFO_RESPONSE_CODE, &code);
		if (code == 200) {
			rename(tmp, localpath);
		}
		else if (code == 304) {
			res = CURLE_HTTP_RETURNED_ERROR;
		}
	}
	
	/* 無条件に削除 */
	remove(tmp);

	return res;
}
