/**
 * $Id$
 *
 * Autoindex 及びブラウザインターフェースを実現する構造体、関数の宣言＆定義
 *
 * (note) このファイル導入の背景
 *
 * 	ここに定義されているものは、元々リポジトリプロバイダ(repos.c) に
 * 	実装されていたものです。ですが、autoindex関連のコード量が多かったため、
 * 	repos.c がこれらのコードに埋もれてしまっていました。
 * 	近い将来、ブラウザ経由でアップロードを行うような機能を考えると
 * 	ますますrepos.c に入れておく訳にはいないので、別ファイルにしました。
 * 	それ以上の意味はありません。
 * 	分離したことで多くの関数がPublic化されてはいますが、実際のところ
 * 	repos.c 以外には興味のない、もしくはまともに利用できない機能ばかり
 * 	です。これはフレームワークの性質上仕方ありません。無理矢理利用しない
 * 	ように。
 */
#ifndef INCLUDE_TF_AUTOINDEX_H
#define INCLUDE_TF_AUTOINDEX_H

#include "httpd.h"
#include "util_filter.h"
#include "apr.h"
#include "apr_pools.h"

#include "mod_dav_tf.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/*
 * mobile ページに対するアクセスの時、1ページに表示する最大行数
 */
#define DIVY_AUTOINDEX_DISPCNT	10

/*
 * 先頭ページの先頭オフセット
 */
#define DIVY_AUTOINDEX_FIRSTOFFSET 0

/*
 *  エンコード変換に失敗した時に使用する文字列(表示名)
 */
#define DIVY_AUTOINDEX_FAILEDCONV_STR	"???"
#define DIVY_AUTOINDEX_FAILEDCONV_URI	"#"

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * Autoindex 処理を行う.
 *
 * @param resource const dav_resource *
 * @param data dav_postdata *
 * @param output ap_filter_t* フィルタchainを表す構造体へのポインタ(util_filter.h)
 * @return DAV エラー
 */
DIVY_DECLARE(dav_error *) divy_do_autoindex(const dav_resource *resource,
						dav_postdata *data,
						ap_filter_t *output);

/**
 * GET 要求に対するAutoIndex 機能をアクティブにするかどうかを判定する.
 * [ アクティブになる条件 ]
 * 	・resource がコレクションであること
 * 	・非アクティブを表す環境変数が設定されていないこと
 * 	・$roo/.mobile, $root,プライベートコレクション以下,グループコレクション以下に
 * 	  対するアクセスであること
 * 	・スタイルシートRootパスが存在すること
 * 	・スタイルシートが存在すること
 *
 * @param r request_rec * リクエスト構造体
 * @param resource const dav_resource *
 * @return int アクティブかどうか (1 : アクティブ / 0: 非アクティブ)
 */
DIVY_DECLARE(int) divy_enable_autoindex(request_rec *r, const dav_resource *resource);

/**
 * スタイルシート(XSLT)のパスを組み立てて返却する.
 *
 * @param wp apr_pool_t *
 * @param prefix const char * スタイルシートプレフィックス名
 * @param r request_rec *
 * @return char * 組み立てたスタイルシートパス
 */
DIVY_DECLARE(char *) divy_get_stylesheet_path(apr_pool_t *wp, const char *prefix, request_rec *r);

/**
 * ログイン画面が有効になるか判定する
 * [アクティブになる条件]
 * ・usesessionがOnとなっている 
 * $root/.loginへのアクセスであること
 * @param r request_rec * リクエスト構造体
 * @param resource const dav_resource *
 * @return int アクティブかどうか (1 : アクティブ / 0: 非アクティブ)
 */
DIVY_DECLARE(int) divy_enable_login(request_rec *r, const dav_resource *resource);

/**
 * SAML選択画面を出力する
 * SAMLの場合はこれが一番始めの画面になります
 *
 * @param resource const dav_resource*
 * @param output ap_filter_t
 * @return dav_error*
 */
DIVY_DECLARE(dav_error *)
divy_do_saml_idp_select(const dav_resource *resource, ap_filter_t *output);

/**
 * SAML画面を出力する
 * 
 * @param resource const dav_resource*
 * @param output ap_filter_t
 * @return dav_error*
 *
 */
DIVY_DECLARE(dav_error *)
divy_do_saml_login(const dav_resource *resource, ap_filter_t *output);

/**
 * SAML アサーション・コンシューマー・サービス(ACS)へのリダイレクトPOSTを
 * 処理する
 * この関数に来るケースは主にIdPからのリダイレクトです。
 *
 * @param pdata const dav_postdata*
 * @param resource const dav_resource*
 * @param output ap_filter_t*
 * @return dav_error*
 */
DIVY_DECLARE(dav_error *)
divy_do_saml_acs_login(const dav_resource *resource, const dav_postdata *pdata, ap_filter_t *output);

/**
 * ログイン画面を出力する
 * 
 * @param resource const dav_resource*
 * @param output ap_filter_t
 * @return dav_error*
 */
DIVY_DECLARE(dav_error *)
divy_do_login(const dav_resource *resource, ap_filter_t *output);

/**
 * ログインのPOSTを処理する
 *
 * @param pdata const dav_postdata*
 * @param resource const dav_resource*
 * @param output ap_filter_t*
 * @return dav_error*
 */
DIVY_DECLARE(dav_error *)
divy_do_post_login(const dav_resource *resource, const dav_postdata* pdata, ap_filter_t *output);

/*
 * リクエストのURIの短縮URLもとにリソースのプロパティを設定します
 * 短縮URL戻りがおかしい場合はNULLを返却します
 */
DIVY_DECLARE(divy_rdbo_resource*) divy_convert_shorten(apr_pool_t* pool, request_rec* r);

/*
 * 短縮URLへのリクエスト
 */
DIVY_DECLARE(dav_error *) divy_do_shorten(const dav_resource *resource, ap_filter_t *output);

/*
 * 短縮URLへのPOSTリクエスト
 */
DIVY_DECLARE(dav_error *) divy_do_post_shorten(const dav_resource *resource, const dav_postdata* pdata, ap_filter_t *output);

/**
 * SAML画面を有効にするか判定する
 */
DIVY_DECLARE(int) divy_enable_saml(request_rec *r, const dav_resource *resource);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_AUTOINDEX_H */

