/**
 * $Id$
 *
 * TeamFile CGI プログラム呼び出し用インターフェースの定義
 *
 * [ 役割 ]
 *   * TeamFile で使用するシステムCGI、プラグインCGI用コードの共通化
 */
#ifndef INCLUDE_TF_CGI_H
#define INCLUDE_TF_CGI_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_file_io.h"
#include "httpd.h"

#include "tfr.h"
#include "tf_env.h"
#include "tf_folder.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef DAV_SUPPORT_POST_HANDLING
/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
typedef struct divy_cgi_property	divy_cgi_property;
struct divy_cgi_property {
	const char *name;	/* CGIプログラムの表示名 */
	const char *path;	/* CGIプログラムファイルのパス */
	const char *ppath;	/* CGIプログラム格納ディレクトリ(親フォルダのパス) */
	const char *relativepath;	/* CGIルートディレクトリからの相対パス */
	apr_finfo_t finfo;	/* path が示すファイルの状態 */
	int is_nph;			/* NPH CGIかどうか */

	divy_uri_spec *u_spec;	/* CGIプログラムのURI */

	/* 起動したCGIプロセスのstdin, stdout, stderr */
	apr_file_t *cgi_in;
	apr_file_t *cgi_out;
	apr_file_t *cgi_err;

	/* 作業用のプール */
	apr_pool_t *p;
};

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * u_spec が示すリソースはCGI実行が可能かどうかを判定する.
 *
 * @param r request_rec *
 * @param u_spec divy_uri_spec *
 * @return int 処理ステータス (1: 可能 / 0: 不可能)
 */
DIVY_DECLARE(int) divy_cgi_executable(request_rec *r, divy_uri_spec *u_spec);

/**
 * DIVY_SYSCGI_TOP_NAME が示すトップCGIが有効かどうか.
 *
 * @param r request_rec *
 * @return int 処理ステータス (1: 有効 / 0: 無効)
 */
DIVY_DECLARE(int) divy_cgi_support_topcgi(request_rec *r);

/**
 * u_spec が示すURIパートのCGIプログラムのプロパティを取得する.
 * u_spec が$root を示す場合、トップCGIのプロパティを取得します.
 *
 * @param r request_rec *
 * @param u_spec divy_uri_spec *
 * @param cgi_pr divy_cgi_property ** 取得したCGIプロパティへのポインタ
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_cgi_get_property(request_rec *r, divy_uri_spec *u_spec,
                                        divy_cgi_property **cgi_pr);

/**
 * cgi_pr で指定されたCGIプログラムを実行する.
 *
 * @param r request_rec *
 * @param cgi_pr divy_cgi_property * CGIプロパティ
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_cgi_run(request_rec *r, divy_cgi_property *cgi_pr);

/**
 * TeamFile 独自の"グローバル"CGI 環境変数をセットアップする.
 * (note)
 *   この関数は、post_read_request以降のステージで呼び出してください.
 *   さもなければ意図した動作をしません.
 *   また、グローバル環境変数ですので、1度だけコールすれば十分です.
 *
 * (導入の背景)
 *   多くのTeamFile 独自の環境変数は、TeamFile が管理するロケーションの中に制御が
 *   移らなければ取得できません. Apacheの機構上の制約です.
 *   ところが、幾つかの環境変数はTeamFile 管理下のロケーションにアクセスするかどうか
 *   判らないステージでも必要となります.
 *   例えば、不正アクセスに対するエラーメッセージ表示の言語などです. 
 *
 *   これらを救済するため、TeamFileサーバモジュールがインストールされていれば
 *   必ずセットされる環境変数(グローバル環境変数と呼びます)を導入します.
 *   この関数をコールすることでこれらがセットされるようになります.
 *   なお、TZなどのようにenvvarsファイルで明示的に設定された環境変数も
 *   グローバル環境変数と同様、どのステージにおいても参照可能です.
 *
 * (note) セットされる環境変数
 *    * TF_LANGPARAM
 *    * TF_SUPPORTED_LANGTAG
 *    * TF_DEFAULT_LANGTAG
 *    * TF_SYSCGI_ROOT
 *
 * @param r request_rec *
 */
DIVY_DECLARE(void) divy_cgi_setup_gl_envvars(request_rec *r);

/**
 * TeamFile 独自の"ローカル"CGI環境変数をセットアップする.
 *
 * "グローバル"に対する対比として"ローカル"という用語を使用.
 * ロケーション内に制御が移った時に判る環境変数をセットアップする関数です.
 *
 * (note)
 *   * ロケーション内に制御が移ったことが確実に判るステージにおいて
 *     コールして下さい.
 *     1度コールすれば十分です.
 *   * cgi_pr がNULLの場合、CGI専用の幾つかの環境変数が設定されなくなります.
 *
 * @param r request_rec *
 * @param cgi_pr divy_cgi_property *
 */
DIVY_DECLARE(void) divy_cgi_setup_lc_envvars(request_rec *r, divy_cgi_property *cgi_pr);

#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_CGI_H */

