/**
 * $Id$
 *
 * TeamFile アクション拡張コマンド呼び出し用インターフェースの定義
 *
 * [ 役割 ]
 *  * アクション拡張コマンドの実行に必要なコードをまとめること.
 */
#ifndef INCLUDE_TF_EXTCMD_H
#define INCLUDE_TF_EXTCMD_H

#include "apr.h"
#include "apr_pools.h"
#include "httpd.h"

#include "tfr.h"
#include "tf_env.h"
#include "tf_folder.h"
#include "tf_storage.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * 関数divy_extcmd_execute のexitcode に戻される値のビット値
 */
#define DIVY_EXTCMD_BIT_FAIL	0x01	/* アクション拡張コマンドがエラーで失敗した */
#define DIVY_EXTCMD_BIT_NOCNT	0x02	/* 以降の処理を続けない */
#define DIVY_EXTCMD_BIT_NTFAIL	0x04	/* 処理を続行しない場合、クライアントに異常通知する */
#define DIVY_EXTCMD_BIT_DELFILE	0x08	/* 処理を続ける場合、ファイルを残さない */

#define DIVY_EXTCMD_BITNONE	0x00
#define DIVY_EXTCMD_BITMASK	0x0f

/**
 * アクション拡張コマンドの状態を判定するマクロ
 */
#define DIVY_EXTCMD_IS_FAIL(cd)	(((cd) & DIVY_EXTCMD_BITMASK) & DIVY_EXTCMD_BIT_FAIL)
#define DIVY_EXTCMD_IS_NOCONTINUE(cd)	(((cd) & DIVY_EXTCMD_BITMASK) & DIVY_EXTCMD_BIT_NOCNT)
#define DIVY_EXTCMD_IS_NOTIFYFAIL(cd)	(((cd) & DIVY_EXTCMD_BITMASK) & DIVY_EXTCMD_BIT_NTFAIL)
#define DIVY_EXTCMD_IS_DELETEFILE(cd)	(((cd) & DIVY_EXTCMD_BITMASK) & DIVY_EXTCMD_BIT_DELFILE)

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
/**
 * アクション拡張コマンドに渡されるある1つのリソースを表す構造体
 */
typedef struct divy_extcmd_resource	divy_extcmd_resource;
struct divy_extcmd_resource {
	/*
	 * ファイルに関するWebDAV上の情報
	 */
	char *uri;					/* リソースのURI */
	char *displayname;			/* 表示名称 */
	char *getcontenttype;		/* Content-Type ヘッダの値 */
	time_t getlastmodified;		/* 最終更新日付 */
	char *creator_userid;		/* 作成者のユーザID (新規PUTではNULL) */
	char *lastmodifier_userid;	/* 更新者のユーザID */

	divy_uri_spec *u_spec;		/* URI パース情報 */

	/*
	 * ファイルに関する物理情報
	 */
	apr_int64_t flen;		/* ファイルサイズ */
	divy_pfile_t *pfile;	/* 物理ファイルの抽象パス表現 */
};

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * アクション拡張コマンド機構をサポートしているかどうか.
 *
 * @param r request_rec *
 * @return int サポートしているかどうか (1: サポートしている / 0: していない)
 */
DIVY_DECLARE(int) divy_extcmd_support(request_rec *r);

/**
 * アクション拡張コマンドを実行できるリクエストか調べる
 *
 * @param r request_rec *
 * @return int 実行できるかどうか（1: 実行可能 / 0: 実行しない)
 */
DIVY_DECLARE(int) divy_extcmd_match(request_rec *r);

/**
 * res が示すファイルに対するアクション拡張コマンドを実行する.
 *
 * @param r request_rec *
 * @param res const divy_extcmd_resource * アクション拡張コマンドが対象とするリソースの情報
 * @param exitcode int * アクションコマンドの実行状態を示す.
 *        状態は、定義値DIVY_EXTCMD_BIT_xxxx のビットOR として報告されます. 
 *	      マクロDIVY_EXTCMD_IS_xxx を使って判定してください.
 * @return int 本関数自体の成功可否 (1: 失敗 / 0: 成功)
 */
DIVY_DECLARE(int) divy_extcmd_execute(request_rec *r,
						const divy_extcmd_resource *res, int *exitcode);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_EXTCMD_H */

