/**
 * $Id$
 *
 * tf_folder.h
 *
 * "フォルダ"、URIを管理する構造体、配列を定義するヘッダファイル。
 * このヘッダ自身はApache本体に依存することはありません。
 * ですが、このヘッダはlibtfutil ライブラリには含めてはなりません。このヘッダが
 * 定義する内容にはTeamFileサーバのビジネスロジックが含まれてるからです。
 * これを"util"と呼ぶには無理があります。
 *
 * (note) このヘッダ導入の理由
 *  スタンドアローン型のプログラム(管理ツール)から利用するためです。
 *  このプログラムは、Apache依存ライブラリを利用できないため、mod_dav_tf.h を
 *  include出来ませんでした。それを救済する目的にこのヘッダを導入しました。
 *  それ以上の理由はありません。後で、この理屈を覆すことの出来る十分な理由が
 *  見つかった時には、適切なヘッダに移動して下さい。
 */
#ifndef INCLUDE_TF_FOLDER_H
#define INCLUDE_TF_FOLDER_H

#include "apr.h"
#include "apr_pools.h"
#include "ap_config.h"
#include "ap_regex.h"

#include "tfr.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define Fixed values and macro
  ----------------------------------------------------------------------------*/
/**
 * リレーションを表す名称に付けるプレフィックス文字列
 */
#define DIVY_PREFIX_RUSER  ".RU_"	/* ユーザのリレーション   */
#define DIVY_PREFIX_RGROUP ".RG_"	/* グループのリレーション */
#define DIVY_PREFIX_RSQL   ".RS_"	/* SQLのリレーション      */

/**
 * BOXなどで利用されるゲストユーザ
 */
#define DIVY_GUEST_ID "._guest_"	

/**
 * ごみ箱フォルダの名前(uri のfinal path segment / 表示名)
 */
#define DIVY_TRASHFOLDER_NAME	"Trash_Folder"

/**
 * ごみ箱フォルダのContent-Type
 */
#define DIVY_TRASHFOLDER_CONTENTTYPE	"httpd/trash-directory"

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * システムCGIの特殊パス名
 */
#define DIVY_SYSCGIFOLDER_NAME	".cgi-bin"

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * プラグインCGIの特殊パス名
 */
#define DIVY_PLUGINCGIFOLDER_NAME	"cgi-bin"
#endif	/* DIVY_SUPPORT_PLUGIN */

#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * プラグイン特殊フォルダ名
 */
#define DIVY_PLUGINFOLDER_NAME	".plugin"
#endif	/* DIVY_SUPPORT_PLUGIN */

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * トップ画面を表示するCGIの名称
 */
#define DIVY_SYSCGI_TOP_NAME	"top"
#endif	/* DAV_SUPPORT_POST_HANDLING */

/*
 * URIパラメータのキー名 (for autoindex)
 */
/* コマンドパラメータ */
#define DIVY_URIPARAM_CMD	"cmd"

/* 暗号化されたリソースID */
#define DIVY_URIPARAM_CRSID	"cid"

/* 表示したいページの番号 */
#define DIVY_URIPARAM_REQPAGE	"rp"

/* DIVY_URIPARAM_CMD に指定される値の種類 */
#define DIVY_CMD_SHOW_SUMMARY	"s"		/* 概要表示(autoindex) */
#define DIVY_CMD_SHOW_DETAIL	"d"		/* 詳細表示(autoindex) */
#define DIVY_CMD_SHOW_GLIST	"g"		/* グループフォルダのリスト表示(autoindex) */
#define DIVY_CMD_SHOW_ULIST	"u"			/* プライベートコレクションのリスト表示(autoindex) */
#define DIVY_CMD_DOWNLOAD	"dl"		/* ファイルダウンロード(autoindex) */
#define DIVY_CMD_BOX        "box"		/* BOXを示すパラメータ(autoindex) */

/* 公開チケット用URLパラメータ */
#define DIVY_URIPARAM_TICKET_URI "u"

/* ダウンロード専用リンクを示すパラメータ(autoindex経由) */
#define DIVY_URIPARAM_DOWNLOAD_ONLY	"ct"
#define DIVY_URIPARAM_DOWNLOAD_ONLY_VAL	"at"

/* 短縮URLを表すパラメーター(主に直接ダウンロード) */
#define DIVY_URIPARAM_SHORTEN "st"

/* 短縮URLからのダウンロードの時の親短縮URL */
#define DIVY_URIPARAM_PARENT "p"

/*
 * divy_infotype のタイプを判定するマクロ集
 */

/* リレーションリソースかどうか(引数 divy_infotype) */
#define IS_RELATION_RESOURCE(type)	\
	((type) == DIVY_INFOTYPE_m_group_ruser ||	\
	 (type) == DIVY_INFOTYPE_m_group_rsql  ||	\
	 (type) == DIVY_INFOTYPE_m_user_rgrp   ||	\
	 (type) == DIVY_INFOTYPE_m_sql_rgrp)


/*------------------------------------------------------------------------------
  Define enum and structure
  ----------------------------------------------------------------------------*/
/**
 * 通常のリソースやコレクションとは異なる専用の特殊なリソースと
 * コレクション(ユーザ、グループ、SQLなど)を表すフォルダの識別ID
 * divy_special_folder_spec 構造体の配列 _special_folders (tf_folder.c) の
 * インデックス番号と正確に対応しています。
 */
enum __divy_special_folderid_id {

	DIVY_FOLDER_ID_user = 0,	/* ユーザフォルダ       */
	DIVY_FOLDER_ID_group,		/* グループフォルダ     */
	DIVY_FOLDER_ID_dbfolder,	/* DB検索結果フォルダ   */
	DIVY_FOLDER_ID_dbshfolder,	/* 共有リンクフォルダ   */
	DIVY_FOLDER_ID_management,	/* Management フォルダ  */
	DIVY_FOLDER_ID_m_update,	/* UPDATE フォルダ      */
	DIVY_FOLDER_ID_m_sql,		/* SQL フォルダ         */
	DIVY_FOLDER_ID_m_group,		/* GROUP フォルダ       */ 
	DIVY_FOLDER_ID_m_user,		/* USER フォルダ        */
	DIVY_FOLDER_ID_m_status,	/* STATUS フォルダ      */
	DIVY_FOLDER_ID_m_msg,		/* MSG フォルダ         */
	DIVY_FOLDER_ID_m_dbms,		/* DBMS フォルダ        */
	DIVY_FOLDER_ID_m_execsql,	/* SQL実行フォルダ      */
	DIVY_FOLDER_ID_userinfo,	/* (roottree request用) */
	DIVY_FOLDER_ID_dblink,		/* (linkdb request用)   */
	DIVY_FOLDER_ID_reposdblink,	/* (repos request用)    */
	DIVY_FOLDER_ID_mail,		/* (メールrequest用)    */ 
	DIVY_FOLDER_ID_mobile,		/* (モバイルrequest用)  */ 
	DIVY_FOLDER_ID_login,       /* (ログインrequest用） */
	DIVY_FOLDER_ID_shorten,		/* 短縮URL              */
	DIVY_FOLDER_ID_saml,        /* SAML フォルダ        */
	DIVY_FOLDER_ID_utility,		/* Util フォルダ        */
	DIVY_FOLDER_ID_u_update,	/* Util UPDATE フォルダ */
	DIVY_FOLDER_ID_u_msg,		/* Util MSG フォルダ    */
	DIVY_FOLDER_ID_ticket,		/* (ticket request用)   */
	DIVY_FOLDER_ID_fileviewer,	/* AutoindexスタートアップURI */
#ifdef DAV_SUPPORT_POST_HANDLING
	DIVY_FOLDER_ID_cgibin,		/* グローバルCGI実行フォルダ         */
#endif	/* DAV_SUPPORT_POST_HANDLING */
#ifdef DIVY_SUPPORT_PLUGIN
	DIVY_FOLDER_ID_plugin,		/* プラグインベースフォルダ          */
#endif	/* DIVY_SUPPORT_PLUGIN */
	DIVY_FOLDER_ID_autodelete,	/* 自動削除起動用       */
	DIVY_FOLDER_ID_confirmreading,	/* 開封通知の自動解除起動用 */

	DIVY_FOLDER_ID_END	/* sentinel */
};
typedef enum __divy_special_folderid_id divy_special_folderid_id;

/**
 * あるURI を解析したとき、そのURI自身が持っている情報とそのURIに
 * よって示されるリソース・コレクションの情報を表す列挙型
 */
enum __divy_infotype {

	DIVY_INFOTYPE_unclassified = 0, /* (未分類)  */
	DIVY_INFOTYPE_userinfo,         /* roottreesearch 用URI                   */
	DIVY_INFOTYPE_root,             /* ルートフォルダ                         */
	DIVY_INFOTYPE_user_e,           /* プライベートコレクション               */
	DIVY_INFOTYPE_user_trash,	/* 個人ごみ箱                             */
	DIVY_INFOTYPE_user_trash_e0,	/* 個人ごみ箱の直下にあるリソース         */
	DIVY_INFOTYPE_user_trash_ex,	/* 個人ごみ箱の直下よりも下にあるリソース */
	DIVY_INFOTYPE_user_e_regular,	/* プライベートコレクション以下のリソース */
	DIVY_INFOTYPE_group,            /* グループコレクション格納フォルダ       */
	DIVY_INFOTYPE_group_e,          /* グループコレクション                   */
	DIVY_INFOTYPE_group_trash,	/* グループごみ箱                         */
	DIVY_INFOTYPE_group_trash_e0,	/* グループごみ箱の直下にあるリソース     */
	DIVY_INFOTYPE_group_trash_ex,	/* グループごみ箱のよりも下にあるリソース */
	DIVY_INFOTYPE_group_e_regular,	/* グループコレクション以下のリソース     */
	DIVY_INFOTYPE_dblink,           /* データベース                           */
	DIVY_INFOTYPE_dblink_e,         /* 使用可能なSQL                          */
	DIVY_INFOTYPE_reposdblink,	/* リポジトリDB                           */
	DIVY_INFOTYPE_reposdblink_e,	/* リポジトリDB検索用SQL                  */
	DIVY_INFOTYPE_dbfolder,         /* SQL検索結果格納フォルダ                */
	DIVY_INFOTYPE_dbfolder_e,       /* SQL検索結果フォルダ                    */
	DIVY_INFOTYPE_dbfolder_e_regular,/* SQL検索結果フォルダ以下のリソース     */
	DIVY_INFOTYPE_dbshfolder,       /* 共有コレクション格納フォルダ           */
	DIVY_INFOTYPE_dbshfolder_e,     /* 共有コレクション                       */
	DIVY_INFOTYPE_dbshfolder_e_regular,/* 共有コレクション以下のリソース      */
	DIVY_INFOTYPE_m_management,     /* 管理者フォルダ                         */
	DIVY_INFOTYPE_m_update,         /* 更新クライアントモジュールフォルダ     */
	DIVY_INFOTYPE_m_update_e,       /* 更新クライアントモジュール             */
	DIVY_INFOTYPE_m_user,           /* ユーザ (エンティティ)格納フォルダ      */
	DIVY_INFOTYPE_m_user_e,         /* ユーザ (エンティティ)                  */
	DIVY_INFOTYPE_m_user_rgrp,      /* ユーザが保持するグループのリレーション */
	DIVY_INFOTYPE_m_group,          /* グループ (エンティティ)格納フォルダ    */
	DIVY_INFOTYPE_m_group_e,        /* グループ (エンティティ)                */
	DIVY_INFOTYPE_m_group_ruser,    /* グループが保持するユーザのリレーション */
	DIVY_INFOTYPE_m_group_rsql,     /* グループが保持するSQLのリレーション    */
	DIVY_INFOTYPE_m_sql,            /* SQL (エンティティ)格納フォルダ         */
	DIVY_INFOTYPE_m_sql_e,          /* SQL (エンティティ)                     */
	DIVY_INFOTYPE_m_sql_rgrp,       /* SQLが保持するグループのリレーション    */
	DIVY_INFOTYPE_m_status,         /* ユーザステータスフォルダ               */
	DIVY_INFOTYPE_m_status_e,       /* ユーザステータス                       */
	DIVY_INFOTYPE_m_msg,            /* システムメッセージフォルダ             */
	DIVY_INFOTYPE_m_msg_e,          /* システムメッセージ                     */
	DIVY_INFOTYPE_m_dbms,           /* DBMS情報フォルダ                       */
	DIVY_INFOTYPE_m_dbms_e,         /* DBMS情報                               */
	DIVY_INFOTYPE_m_execsql,        /* SQL実行(SQLためし実行)                 */
	DIVY_INFOTYPE_m_execsql_e,      /* SQL実行(SQLためし実行)用SQL            */
	DIVY_INFOTYPE_mail,             /* 送信メール格納フォルダ                 */
	DIVY_INFOTYPE_mail_e,           /* 送信メール                             */
	DIVY_INFOTYPE_mobile,           /* モバイル用フォルダ                     */
	DIVY_INFOTYPE_login,            /* ログイン用フォルダ                     */
	DIVY_INFOTYPE_saml,             /* SAML用フォルダ                         */
	DIVY_INFOTYPE_shorten,		/* 短縮URLフォルダ                        */
	DIVY_INFOTYPE_utility,		/* ユーティリティフォルダ                 */
	DIVY_INFOTYPE_u_update,		/* 更新クライアントフォルダ(in Utlity)    */
	DIVY_INFOTYPE_u_update_e,	/* 更新クライアント (in Utility)          */
	DIVY_INFOTYPE_u_msg,		/* システムメッセージフォルダ(in Utility) */
	DIVY_INFOTYPE_u_msg_e,		/* システムメッセージ (in Utility)        */
	DIVY_INFOTYPE_ticket,		/* Ticket リクエスト用                    */
	DIVY_INFOTYPE_fileviewer,	/* AutoindexスタートアップページURI       */
#ifdef DAV_SUPPORT_POST_HANDLING
	DIVY_INFOTYPE_cgibin,		/* グローバルCGI実行フォルダ              */
	DIVY_INFOTYPE_cgibin_e,		/* グローバルCGI実行フォルダのCGI         */
#endif	/* DAV_SUPPORT_POST_HANDLING */
#ifdef DIVY_SUPPORT_PLUGIN
	DIVY_INFOTYPE_plugin,		   /* プラグインベースフォルダ                       */
	DIVY_INFOTYPE_plugin_e,		   /* 個々のプラグインを表すフォルダ                 */
	DIVY_INFOTYPE_plugin_uploads,  /* 個々のプラグインが持つ専用アップロード領域     */	
	DIVY_INFOTYPE_plugin_uploads_e,/* 個々のプラグインが持つ専用アップロードファイル */	
	DIVY_INFOTYPE_plugin_cgibin,   /* 個々のプラグインのCGI実行フォルダ              */	
	DIVY_INFOTYPE_plugin_cgibin_e, /* 個々のプラグインのCGI実行フォルダのCGI         */	
	DIVY_INFOTYPE_plugin_conf,     /* 個々のプラグインのコンフィグ格納フォルダ       */
	DIVY_INFOTYPE_plugin_conf_e,   /* 個々のプラグインのコンフィグファイル           */
#endif	/* DIVY_SUPPORT_PLUGIN */
	DIVY_INFOTYPE_autodelete,       /* 自動削除処理用フォルダ */
	DIVY_INFOTYPE_confirmreading,   /* 開封通知の自動解除処理用のフォルダ */

	DIVY_INFOTYPE_END               /* sentinel */
};
typedef enum __divy_infotype divy_infotype;


/**
 * 特殊フォルダを表す構造体
 *
 */
struct __divy_special_folder_spec {

	/* フォルダID */
	divy_special_folderid_id id;

	/* このURIの種類 */
	divy_infotype infotype;

	/* 表示名 */
	const char *displayname;

	/* 相対URI (Location 名は入りません) */
	const char *relativeuri;

	/*
	 * トップ階層に並べるときのフォルダタイプ
	 * (for roottreesearch)
	 */
#define DIVY_FOLDER_TYPE_UNDEFINE	NULL	/* 未定義であることを示す */
#define DIVY_FOLDER_TYPE_user           "user"
#define DIVY_FOLDER_TYPE_group          "group"
#define DIVY_FOLDER_TYPE_group_elm      "mgroup"
#define DIVY_FOLDER_TYPE_dblink         "dblink"
#define DIVY_FOLDER_TYPE_sqlstmt        "sqlstmt"
#define DIVY_FOLDER_TYPE_reposdblink    "reposdblink"
#define DIVY_FOLDER_TYPE_reposdbsqlstmt "reposdbsqlstmt"
#define DIVY_FOLDER_TYPE_manage         "management"
#define DIVY_FOLDER_TYPE_manage_elm     "melement"
#define DIVY_FOLDER_TYPE_execsqlstmt    "execsqlstmt"
	const char *foldertype;

	/*
	 * トップ階層に並べるときの表示順番 (DIVY_DISPLAYORDER_UNDEFINEなら未定義)
	 * (for roottreesearch)
	 */
#define DIVY_DISPLAYORDER_UNDEFINE	 -1	/* displayorder 未定義 */
	int displayorder;

	/*
	 * ツリービューで同時に表示するグループ
	 * (for roottreesearch)
	 */
#define DIVY_SHOW_HIDDEN     	0x01 	/* ツリーにフォルダとして表示しない */
#define DIVY_SHOW_ALLLIST    	0x02 	/* alllist 時に表示する 	    */
#define DIVY_SHOW_MANAGEMENT	0x04	/* management 時に表示する	    */
#define DIVY_SHOW_DB 		0x08	/* DB連携機能の使用可の時に表示する */
	int show_group;

	/*
	 * 特殊フォルダに対するアクセス制御に利用する識別名称。
	 * (usrdiscoveryタグのaccessdenyの中に指定される名前)
	 */
	const char *accessdeny_name;

	/**
	 * relativeuri の長さ(初期値は0)
	 */
	apr_size_t uri_len;
};
typedef struct __divy_special_folder_spec divy_special_folder_spec;

/**
 * 解析されたURI の情報を保持する構造体
 */
struct __divy_uri_spec {

	/* 解析前のオリジナルURI */
	const char *uri;

	/* location に指定されたURI */
	const char *root;

	/* root が "/"かどうか (1: "/" である / 0: "/"ではない) */
	int is_default_root;

	/*
	 * URIに含まれていた特殊フォルダをあらわすURI部分 $root は除く。
	 * 特殊フォルダ部分がなければ空文字
	 */
	const char *special_part;

	/*
	 * root, special_part 以外の残りのURI部分
	 */
	const char *other_part;

	/*
	 * URI をpath segment に分割した時、最後に現れるpath segment 名
	 * other_part と重複するかもしれません。(でも気にしない)
	 * (例)
	 *     uri = /aaa/bbb/ccc --> final_path_segment = "ccc"
	 */
	const char *final_path_segment;

 /* このURI のリソースが表している種類を分類するタイプ */
#define DIVY_URITYPE_UNKNOWN 0x01    /* 分類されていない(不正アクセス)URIを表す */
#define DIVY_URITYPE_REGULAR 0x02    /* 通常リソースをあらわすURI               */
#define DIVY_URITYPE_VIRTUAL 0x04    /* 仮想フォルダを表すURI                   */
#define DIVY_URITYPE_SPECIAL 0x08    /* 仮想フォルダの下の特殊フォルダを表すURI */
#define DIVY_URITYPE_ARBITER 0x10    /* 処理場所を表すURI                       */
	int uritype;

	/* このURI が表している情報(リソース)の種類 */
	divy_infotype infotype;	

	/* このURIの親リソースの種類 */
	divy_infotype p_infotype;
};
typedef struct __divy_uri_spec divy_uri_spec;

/**
 * あるURIパターンのinfotype とuritype との関係を保持する構造体
 * パターンマッチングしたURIからURIの種類を特定する目的で使用します。
 */
struct __divy_uri_pattern {
	/*
	 * URIパターン
	 */
	const char *uri_pt;

	/* uri_pt のinfotype */
	divy_infotype infotype;

	/*
	 * このURIパターンの祖先の特殊フォルダID
	 */
	divy_special_folderid_id fid;

	/* uri_pt のuritype */
	int uritype;

	/* infotype の親のinfotype */
	divy_infotype p_infotype;

	/* uri_pt をコンパイルした結果を持つ変数(初期値NULL) */
	ap_regex_t *preg;
};
typedef struct __divy_uri_pattern	divy_uri_pattern;

/*------------------------------------------------------------------------------
  Declar prototype public functions
  ----------------------------------------------------------------------------*/
/**
 * リソースのURIパターン文字列をコンパイルして正規表現ライブラリから
 * 利用できるようにする。 (パフォーマンス対応)
 * (note)
 * 	この関数は１つのプロセスにつきただ１回だけ実行されるよう呼び出し元
 * 	にて保証して下さい。複数回実行しても構いませんが、無駄です。
 * 	ただし複数回実行するのなら、同期化を必ず行って下さい。
 */
DIVY_DECLARE(void) divy_precomp_uri_pattern(apr_pool_t *p);

/**
 * 特殊フォルダID fid が示す divy_special_folder_spec 構造体へのポインタを
 * 取得して返却する。
 *
 * @param p apr_pool_t *
 * @param fid divy_special_folderid_id 特殊フォルダのID
 * @return const divy_special_folder_spec *
 * 			取得に失敗したらNULL
 */
DIVY_DECLARE(const divy_special_folder_spec *) divy_get_special_fspec(
					apr_pool_t *p,
					divy_special_folderid_id fid);

/**
 * プライベートコレクションまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_user_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * グループコレクションまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_group_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * linkdbsearch 結果格納フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_dbfolder_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * 共有コレクションまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_dbshfolder_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * 管理者フォルダのURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_management_uri(apr_pool_t *p, const char *root_uri);

/**
 * 更新クライアントフォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_update_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);
/**
 * SQL フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_sql_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * グループフォルダ(in 管理者フォルダ) またはそのフォルダの下位階層の
 * URIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_group_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * ユーザフォルダ(in 管理者フォルダ) またはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_user_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * 統計情報(in 管理者フォルダ) またはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_status_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 *  システムメッセージフォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_msg_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * DBMS フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_dbms_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * SQL ためし実行フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_m_execsql_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * roottreesearch 受け付けURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_userinfo_uri(apr_pool_t *p, const char *root_uri);

/**
 * linkdbsearch 受け付けURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_dblink_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * リポジトリ検索受け付けURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param segment const char * path segment (名前 / 相対URI)
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_reposdblink_uri(apr_pool_t *p, const char *root_uri,
							const char *segment);

/**
 * メール送信リクエスト受け付けURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_mail_uri(apr_pool_t *p, const char *root_uri);

/**
 * モバイルアクセス用URIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param cmd const char *  コマンド文字列
 * @param rsid const char * リソースID
 * @param rp int リクエストページ
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_mobile_uri(apr_pool_t *p, const char *root_uri,
					const char *cmd, const char *rsid, int rp);

/**
 * ユーティリティフォルダのURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_utility_uri(apr_pool_t *p, const char *root_uri);

/**
 * 更新クライアント(in ユーティリティ) フォルダのURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_u_update_uri(apr_pool_t *p, const char *root_uri);

/**
 * システムメッセージ(in ユーティリティ) フォルダのURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_u_msg_uri(apr_pool_t *p, const char *root_uri);

/**
 * グループの下のごみ箱フォルダまたはそのフォルダの下位階層のURIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param grp_segment const char * グループ名 / グループ相対URI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_group_trash_uri(apr_pool_t *p, const char *root_uri,
							const char *grp_segment);

/**
 * プライベートコレクションの下のごみ箱フォルダまたはそのフォルダの下位階層の
 * URIを組み立てる。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param userid const char * ユーザID
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_user_trash_uri(apr_pool_t *p, const char *root_uri,
							const char *userid);

/**
 * 公開チケット用URLを組み立てる。(相対パス)
 *
 * [ 組み立てられたURL ]
 * 		$root/.ticket?u=xxxxxxx
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param ticket const char * チケット文字列
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_ticket_url(apr_pool_t *p, const char * root_uri,
							const char *ticket);

/**
 * 公開用短縮URLを生成する [ 相対パス ]
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param ticket const char * 短縮文字列
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_shorten_url(apr_pool_t *p, const char *root_uri,
							const char *shorten);

/**
 * Autoindex スタートアップページ用URIを組み立てる.
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_autoindex_startup_uri(apr_pool_t *p,
                                                      const char * root_uri);
/**
 * login用URIを組み立てる.
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_login_uri(apr_pool_t *p,
                                                      const char * root_uri);

/**
 * SAML用URIを組み立てる.
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_saml_uri(apr_pool_t *p,
                                                      const char * root_uri);
/**
 * 短縮用URIを組み立てる.
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_shorten_uri(apr_pool_t *p,
                                                      const char * root_uri);
#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * トップCGI用URIを組み立てる.
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_topcgi_uri(apr_pool_t *p, const char * root_uri);
#endif	/* DAV_SUPPORT_POST_HANDLING */

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * プラグイン名 plugin_name、相対URI uri のURIパスを組み立てて返却する.
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param plugin_name const char * プラグイン名称
 * @param uri const char * 相対URI
 * @return char * 組み立てられたURI
 */
DIVY_DECLARE(char *) divy_build_plugin_uri(apr_pool_t *p, const char * root_uri,
                                           const char *plugin_name, const char *uri);
#endif	/* DIVY_SUPPORT_PLUGIN */

/**
 * 指定されたfid の特殊フォルダが持つ表示名(displayname) を取得して返却する。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param fid divy_special_folderid_id 特殊フォルダを識別する数値(ID)
 * @return char * 取得した特殊フォルダの表示名
 * 		fid が存在する特殊フォルダをさしていなければNULLを返却します。
 */
DIVY_DECLARE(char *) divy_get_sfolder_name(apr_pool_t *p,
						divy_special_folderid_id fid);

/**
 * 指定されたfid の特殊フォルダが持つ相対URIのfinal path segment を
 * 取得して返却する。
 * final path segment が表示名(displayname)と一致するかどうか保証されていない
 * ことに注意して下さい。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param fid divy_special_folderid_id 特殊フォルダを識別する数値(ID)
 * @return char * 取得した特殊フォルダのfinale path segment
 * 		fid が存在する特殊フォルダをさしていなければNULLを返却します。
 */
DIVY_DECLARE(char *) divy_get_sfolder_finalpathsegment(apr_pool_t *p,
						divy_special_folderid_id fid);

/**
 * 指定されたfid の特殊フォルダが持つ"アクセス拒否メソッド名" を取得して返却する。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param fid divy_special_folderid_id 特殊フォルダを識別する数値(ID)
 * @return char * 取得した特殊フォルダの"アクセス拒否メソッド名称"
 * 		fid が存在する特殊フォルダをさしていなければNULLを返却します。
 */
DIVY_DECLARE(char *) divy_get_sfolder_accessdeny_name(apr_pool_t *p,
						divy_special_folderid_id fid);

/**
 * エンティティの名称として正しいかどうかを検証する。
 * (検証内容)
 * 	リレーションプレフィックスを持たないこと
 *
 * @param p apr_pool_t *
 * @param name const char * エンティティの名称
 * @return int 検証結果 (0: 正常 / 1: 問題あり)
 */
DIVY_DECLARE(int) divy_validate_having_entity_prefix(apr_pool_t *p,
							const char *name);

/**
 * pcol_name が示すプライベートコレクションの名称が特殊フォルダの名称と
 * バッティングしていないかどうか判定する。
 *
 * @param p apr_pool_t *
 * @param name const char * プライベートコレクションの名前
 * @return int 検証結果 (0: 問題なし / 1: 名前がバッティングしていた)
 */
DIVY_DECLARE(int) divy_validate_different_sfolder_name(apr_pool_t *p,
							const char *pcol_name);

/**
 * 指定されたuri を解析して、divy_uri_spec 構造体を生成する。
 * (note)
 * 	一部、現在のコレクション階層の構造に依存したコードがあります。
 * 	コレクション構造を変えるときには、このコードを変更してください。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI (Apache であれば Location の名前)
 * @param uri const char * 解析対象のURIを表す文字列
 * @param u_spec divy_uri_spec ** URI を解析して生成した構造体
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_uri(apr_pool_t *p, const char *root_uri,
					const char *uri, divy_uri_spec **u_spec);

/**
 * 指定されたuri からメール監視状態を設定可能なリソースのuri を切り出す。
 *
 * (note)
 * 	なお、切り出されたuriは、"メール監視状態を設定可能なコレクションのuri"
 * 	であり、メール監視状態がついているとは限りません。
 * 	それは、呼び出し元が得られたuriを元にdivy_mailwatch テーブルを検索して
 * 	確かめてください。
 *
 * [ メール監視状態を設定可能な親コレクションの種類 ]
 *
 * 	* 共有コレクション
 * 	* linkdbsearch 検索結果フォルダ
 * 	* グループコレクション
 *
 * 	管理者フォルダのグループフォルダには、メール監視状態は付きません。
 * 	そのグループに所属したユーザに自動的に割り当てるデフォルトの
 * 	メール監視状態を持っているだけです。(groupmailwatch)
 *
 * @param p apr_pool_t * 作業用のプール
 * @param root_uri const char * ルートURI
 * @param uri const char * 抽出対象のuri
 * @param mwatch_uri char ** メール監視状態を設定可能な親リソースのuri
 * 			不正なuriまたは、メール監視状態がそもそも設定し得ない
 * 			リソースuriの時にはNULL値を返却する
 * @param char ** メール監視フラグ設定可能なフォルダのURI
 * 			メール監視状態が立たないリソースが指定されていたらNULLです。
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 * 			メール監視フラグが立たないリソースの場合は正常となります。
 */
DIVY_DECLARE(int) divy_extract_mailwatch_uri(apr_pool_t *p, const char *root_uri,
						const char *uri,char **mwatch_uri);

/**
 * 指定されたuri のリソースが捨てられる時、そのリソースのごみ箱となる
 * フォルダのURIを算出して返却する。
 * uri がごみ箱を持たない種類のリソースを表していた場合、返却値はNULLになります。
 *
 * [ ごみ箱のURIパターン ]
 *   * $root/Group Folder/$groupid/Trash_Folder/
 *   * $root/Group Folder/%groupname/Trash_Folder/
 *   * $root/$userid/Trash_Folder/
 *
 * @param p apr_pool_t * 返却される文字列のアロケーター
 * @param root_uri const char * ルートURI
 * @param uri const char * URI
 * @param char ** ごみ箱フォルダのURI
 * 	uri がごみ箱を持たない種類のリソースを表していた場合、返却値はNULL
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 * 			ごみ箱を持たない種類のリソースを表していた場合は正常。
 */
DIVY_DECLARE(int) divy_extract_trashfolder_uri(apr_pool_t *p, const char *root_uri,
						const char *uri, char **trash_uri);

/**
 * 指定されたグループコレクション以下のリソースuri から
 * グループコレクションURIを算出して返却する。
 * uri がグループコレクション以下のURIではない場合にはNULLとなります。
 *
 * @param p apr_pool_t * 返却される文字列のアロケーター
 * @param root_uri const char * ルートURI
 * @param uri const char * URI
 * @param char ** グループコレクションのURI
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 * 			グループコレクション以下のリソースではなくても正常。
 */
DIVY_DECLARE(int) divy_extract_groupcol_uri(apr_pool_t *p, const char *root_uri,
						const char *uri, char **grpcol_uri);

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * 指定されたURI情報u_spec からプラグイン名称を抽出する.
 *
 * @param p apr_pool_t *
 * @param u_spec divy_uri_spec *
 * @param name char ** 抽出したプラグイン名称
 */
DIVY_DECLARE(int) divy_extract_plugin_name(apr_pool_t *p, divy_uri_spec *u_spec,
						char **name);
#endif	/* DIVY_SUPPORT_PLUGIN */

/**
 * str から公開チケット用文字列を生成して返却する
 *
 * @param p apr_pool_t *
 * @param str const char * オリジナル文字列
 * @return char * チケット文字列
 */
DIVY_DECLARE(char *) divy_create_ticket_str(apr_pool_t *p, const char *str);

/**
 * 公開チケット用文字列ticket からオリジナル文字列に復号化して返却する。
 *
 * @param p apr_pool_t *
 * @param ticket const char *
 * @return char * オリジナル文字列
 */
DIVY_DECLARE(char *) divy_parse_ticket_str(apr_pool_t *p, const char *ticket);

/**
 * リソースIDを元に短縮文字列を生成します。
 *
 * rid  = 000003228573 (12桁)
 * salt = 82           ランダム文字 (2桁)
 * base = 82000003228573 => Ujenskgx (62進数)
 *
 * @param p     apr_pool_t * プール
 * @param rid   const char * リソースID
 * @return 短縮文字列 char * / 失敗: NULL
 */
DIVY_DECLARE(char *) divy_get_rid2shorten(apr_pool_t *pool, const char *rid, const char *old);

/**
 * 短縮文字列をリソースIDへ戻します
 *
 * @param p      apr_pool_t * プール
 * @param shorten const char * リソースID
 * @return リソースID char * / 失敗: NULL
 */
DIVY_DECLARE(char*) divy_get_shorten2rid(apr_pool_t *pool, const char *shorten);

#ifdef DAV_SUPPORT_POST_HANDLING
/**
 * CGIの"SCRIPT_NAME"パスを生成する(CGI専用)
 *
 * @param p apr_pool_t *
 * @param infotype divy_infotype CGIスクリプトの種類
 * @param scriptpath const char * CGIスクリプトの相対パス
 * @return char * 組み立てられた文字列
 */
DIVY_DECLARE(char *) divy_build_cgi_scriptname(apr_pool_t *p, divy_infotype infotype,
                                               const char *scriptpath);
#endif	/* DAV_SUPPORT_POST_HANDLING */

/**
 * ユーザリレーションURI($root/.management/GROUP/$gname/.RU_xxxx) を組み立てて
 * 返却する.
 *
 * @param p apr_pool_t *
 * @param guri const char * グループURI ($root/.mangement/GROUP/$gname)
 * @param userid const char * ユーザID (xxx)
 */
DIVY_DECLARE(char *) divy_build_rusr_uri(apr_pool_t *p, const char *guri, const char *userid);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_FOLDER_H */

