/**
 * $Id$
 *
 * エラーロガー関数の定義
 * (note)
 *   mod_dav_tf.h で定義されていたロガーマクロをここに移動しました.
 */
#ifndef INCLUDE_TF_LOGGER_H
#define INCLUDE_TF_LOGGER_H

#include "apr.h"
#include "apr_pools.h"
#include "httpd.h"
#include "http_log.h"
#include "util_common.h"
 
#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define structure for MACRO
  ----------------------------------------------------------------------------*/
typedef struct __divy_errlog_ctx divy_errlog_ctx;
struct __divy_errlog_ctx {
	server_rec *s;
	const char *usrid;		/* アクセスユーザID */
	const char *location;	/* ロケーション名   */
};

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/**
 * エラーログマクロの定義
 * (note)
 *   以下のログマクロでは、post_read_request ステージを終了した後でなければ
 *   ログレベル(LogLevel) に応じたログ出力制限が機能しません.
 */
#define _GET_LOG_ENV(pool) \
	server_rec *__s = NULL; \
	divy_errlog_ctx *__lctx = NULL; \
	apr_pool_t *__p = pool; \
	while (__p != NULL) { \
		__lctx = divy_pcache_get_data(__p, DIVY_PCACHE_DAT_REQ_LOGENV); \
		if (__lctx != NULL) { \
			__s = __lctx->s; \
			break; \
		} \
		__p = apr_pool_parent_get(__p); \
	} \

#define _GET_LOG_LEVEL(lv) \
	int _lm = lv & APLOG_LEVELMASK;

#define TRACE_S(__s) \
{ \
	if (__s == NULL || (((APLOG_INFO & APLOG_LEVELMASK) <= __s->log.level))) { \
		ap_log_error(APLOG_MARK, APLOG_INFO, 0, __s, \
			"- - TF-TRACE(%"APR_PID_T_FMT"): %s", getpid(),__func__ ); \
	} \
}

/**
 * トレースログマクロ
 *
 * @param pool apr_pool_t * request_recのプール
 */
#define TRACE(pool) \
{ \
	_GET_LOG_ENV(pool) \
	TRACE_S(__s) \
}

/**
 * エラーログマクロ(引数0)
 *
 * @param pool apr_pool_t * request_recのプール
 * @param lv int ログエラーレベル
 * @param stcd int ステータスコード
 * @param f const char * 出力文字列
 */
#define ERRLOG0(pool,lv,stcd,f) \
{ \
	_GET_LOG_LEVEL(lv) \
	_GET_LOG_ENV(pool) \
	if (__lctx == NULL || (_lm == APLOG_NOTICE) || (_lm <= (__s)->log.level)) { \
		const char *__location = (__lctx != NULL) ? __lctx->location : "-"; \
		const char *__usrid    = (__lctx != NULL) ? __lctx->usrid : "-"; \
		ap_log_error(APLOG_MARK, lv, 0, __s, "%s %s %s(%d): (%d) "f, \
				__location, __usrid, __func__, __LINE__, stcd); \
	} \
}

/**
 * エラーログマクロ(引数1)
 *
 * @param pool apr_pool_t * request_recのプール
 * @param lv int ログエラーレベル
 * @param stcd int ステータスコード
 * @param f const char * 書式指定文字列
 * @param a1 const char * 引数1
 */
#define ERRLOG1(pool,lv,stcd,f,a1) \
{ \
	_GET_LOG_LEVEL(lv) \
	_GET_LOG_ENV(pool) \
	if (__lctx == NULL || (_lm == APLOG_NOTICE) || (_lm <= __s->log.level)) { \
		const char *__location = (__lctx != NULL) ? __lctx->location : "-"; \
		const char *__usrid    = (__lctx != NULL) ? __lctx->usrid : "-"; \
		ap_log_error(APLOG_MARK, lv, 0, __s, "%s %s %s(%d): (%d) "f, \
				__location, __usrid, __func__, __LINE__, stcd, a1); \
	} \
}

/**
 * エラーログマクロ(引数2)
 *
 * @param pool apr_pool_t * request_recのプール
 * @param lv int ログエラーレベル
 * @param stcd int ステータスコード
 * @param f const char * 書式指定文字列
 * @param a1 const char * 引数1
 * @param a2 const char * 引数2
 */
#define ERRLOG2(pool,lv,stcd,f,a1,a2) \
{ \
	_GET_LOG_LEVEL(lv) \
	_GET_LOG_ENV(pool) \
	if (__lctx == NULL || (_lm == APLOG_NOTICE) || (_lm <= __s->log.level)) { \
		const char *__location = (__lctx != NULL) ? __lctx->location : "-"; \
		const char *__usrid    = (__lctx != NULL) ? __lctx->usrid : "-"; \
		ap_log_error(APLOG_MARK, lv, 0, __s, "%s %s %s(%d): (%d) "f, \
				__location, __usrid, __func__, __LINE__, stcd, a1, a2); \
	} \
}

/**
 * エラーログマクロ(引数3)
 *
 * @param pool apr_pool_t * request_recのプール
 * @param lv int ログエラーレベル
 * @param stcd int ステータスコード
 * @param f const char * 書式指定文字列
 * @param a1 const char * 引数1
 * @param a2 const char * 引数2
 * @param a3 const char * 引数3
 */
#define ERRLOG3(pool,lv,stcd,f,a1,a2,a3) \
{ \
	_GET_LOG_LEVEL(lv) \
	_GET_LOG_ENV(pool) \
	if (__lctx == NULL || (_lm == APLOG_NOTICE) || (_lm <= __s->log.level)) { \
		const char *__location = (__lctx != NULL) ? __lctx->location : "-"; \
		const char *__usrid    = (__lctx != NULL) ? __lctx->usrid : "-"; \
		ap_log_error(APLOG_MARK, lv, 0, __s, "%s %s %s(%d): (%d) "f, \
				__location, __usrid, __func__, __LINE__, stcd, a1, a2, a3); \
	} \
}

/**
 * 指定されたerr(divy_error) が示すエラーログをApacheログに出力するマクロ
 * エラースタックから古い順に出力します。
 *
 * @param pool apr_pool_t * request_rec のプール
 * @param derr divy_error * エラー情報
 */
#define DUMP_ERRLOGS(pool,derr)	\
{ \
	if (pool != NULL && derr != NULL) { \
		divy_error *__err, *__last; \
		int _lm; \
		_GET_LOG_ENV(pool) \
		for (__last = derr; __last->prev; __last = __last->prev); \
		for (__err = __last; __err; __err = __err->next) { \
			_lm = __err->level & APLOG_LEVELMASK; \
			if (__lctx == NULL || (_lm == APLOG_NOTICE) || (_lm <= __s->log.level)) { \
				const char *__location = (__lctx != NULL) ? __lctx->location : "-"; \
				const char *__usrid    = (__lctx != NULL) ? __lctx->usrid : "-"; \
				ap_log_error(APLOG_MARK, __err->level, 0, __s, "%s %s %s(%d): (%d) %s", \
						__location, __usrid, __err->funcname, __err->lineno, \
						__err->status, __err->desc); \
			} \
		} \
	} \
}

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * エラーロガーコンテキストを生成してキャッシュする.
 *
 * @param r request_rec *
 * @return divy_errlog_ctx * 生成されたコンテキスト
 */
DIVY_DECLARE(divy_errlog_ctx *) divy_logger_create(request_rec *r);

/**
 * エラーロガーコンテキストにアクセスユーザIDを書き込む.
 *
 * @param r request_rec *
 * @param usrid const char * アクセスユーザID
 */
DIVY_DECLARE(void) divy_logger_set_userid(request_rec *r, const char *usrid);

/**
 * エラーロガーコンテキストにロケーション名を書き込む.
 *
 * @param r request_rec *
 * @param location const char * ロケーション名
 */
DIVY_DECLARE(void) divy_logger_set_location(request_rec *r, const char *location);

/**
 * level のエラーメッセージmsg をロガーで出力する.
 * (ERRLOG0 - ERRLOG3 までのマクロが利用できない場合に限定して使用すること)
 *
 * @param r request_rec *
 * @param level int ログレベル
 * @param msg const char * メッセージ
 */
DIVY_DECLARE(void) divy_logger_outmsg(request_rec *r, int level, const char *msg);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_LOGGER_H */

