/**
 * $Id:$
 * 
 * tf_memcache.h
 *
 * memcacheを利用する為の関数
 *
 * [ 導入理由 ]
 * 共有メモリでは、一台マシン共有となってしまい複数台になるときのキャッシュ
 * としては足りなかった。
 * あと、そもそもTeamFileの共有メモリはLDAP専用となってしまっている
 */
#ifndef INCLUDE_TF_MEMCACHE_H
#define INCLUDE_TF_MEMCACHE_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_memcache.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/
/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_memcache_t divy_memcache_t;

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * 新規でmemcacheの構造体を作成します
 * @param pool	apr_pool_t * cpool
 * @@aram myhostname const char* 
 * @param memd  divy_memcache_t **
 */
DIVY_DECLARE(void) divy_memcache_create(apr_pool_t* cpool, const char* myhostname, divy_memcache_t** memd);

/**
 * memcachedサーバを登録します
 *
 * @param cpool		apr_pool_t* 		config pool
 * @param memd       divy_memcache_t *
 * @param hostport   const char* 		ホストとポートが連結された文字列
 * @return int 0: 成功 / 1:失敗
 */
DIVY_DECLARE(int) divy_memcache_add_server(apr_pool_t *cpool, divy_memcache_t *memd, const char* hostport);

/**
 * 設定されたdivy_memcache_tの内容をもとにmemcachedを構築します
 * @param cpool		apr_pool_t*
 * @param  memd divy_memcache_t *
 * @param memcache apr_memcache_t**
 * @return int 0: 成功 / 1:失敗
 */
DIVY_DECLARE(int) divy_memcache_open(apr_pool_t* pool, divy_memcache_t *memd, apr_memcache_t **memcache);

/**
 * 設定されたdivy_memcache_tの内容をもとにmemcachedとの接続を行います。
 *
 * @param cpool		apr_pool_t*
 * @param  memd divy_memcache_t *
 * @param memcache apr_memcache_t**
 * @return apr_status_t
 */
DIVY_DECLARE(apr_status_t) divy_memcache_connect(apr_pool_t* pool, divy_memcache_t* memd, apr_memcache_t** memcache);

/**
 * 設定されたmemcacheサーバに対してデータを設定します
 * @param pool			apr_pool_t*			pool
 * @param memd			divy_memcache_t*	memcache構造体
 * @param prefix		const char*			ユニークにするための文字列
 * @param key			const char*			キー
 * @param value			char *				値
 * @param value_size	apr_size_t			valueの長さ
 * @param timeout		apr_uint32_t		キャッシュ保持期間（秒）
 * @param filags		apr_uint16_t		フラグ
 */
DIVY_DECLARE(int) divy_memcache_set(apr_pool_t *pool, divy_memcache_t *memd,
									const char* prefix, const char* key,
									char* value, const apr_size_t value_size,
									apr_uint32_t timeout, apr_uint16_t flags);

/**
 * memcacheよりデータを取得します
 * @param pool			apr_pool_t		pool
 * @param memd			divy_memcache_t	memcache構造体
 * @param prefix		const char*		ユニークにするための文字列
 * @param key			const char*		キー
 * @param value			const char**	値
 * @param length		apr_size_t		値の長さ
 * @param flags			apr_uint16_t	フラグ
 */
DIVY_DECLARE(apr_status_t) divy_memcache_get(apr_pool_t *pool,
									divy_memcache_t *memd, const char* prefix,
									const char* key, char **value,
									apr_size_t *length, apr_uint16_t *flags);

/**
 * memcacheから指定したキーを削除する
 * @param pool    apr_pool_t *		 pool
 * @param memd    divy_memcache_t *	 
 * @param prefix  const char*        削除キーをユニークにしたいときの文字列
 * @param key     const char*        削除したいキー
 * @param timeout apr_uint32_t       キーを再利用できるまでの時間（秒）0:即利用可能
 * @return apr_status_t               aprステータス
 */
DIVY_DECLARE(apr_status_t) divy_memcache_delete(apr_pool_t *pool,
									 divy_memcache_t *memd, const char* prefix,
									 const char *key, apr_uint32_t timeout);
#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_MEMCACHE_H */
