/**
 * $Id$
 *
 * LiveプロパティなどのXMLをパースする関数を集めたヘッダファイル
 * パース結果(xxx_iscreen) からrdbo オブジェクトへの変換する関数群も
 * 保持しています.
 * (note)
 *     liveprop.c からLiveプロパティのパースロジックを分離しました.
 *     複雑度が増したためです.
 */
#ifndef INCLUDE_TF_PARSER_H
#define INCLUDE_TF_PARSER_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_xml.h"

#include "tfr.h"
#include "tf_errno.h"
#include "tf_rdbo.h"
#include "tf_box.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
/*------------------------------------------------------------------------------
  Define Fixed values and macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define enum and structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_user_iscreen	divy_user_iscreen;
typedef struct __divy_group_iscreen	divy_group_iscreen;
typedef struct __divy_sql_iscreen	divy_sql_iscreen;
typedef struct __divy_sysmsg_iscreen	divy_sysmsg_iscreen;
typedef struct __divy_dbms_iscreen	divy_dbms_iscreen;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
typedef struct __divy_sysquota_iscreen	divy_sysquota_iscreen;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
typedef struct __divy_mailwatch_iscreen	divy_mailwatch_iscreen;
typedef struct __divy_sharedcollection_iscreen	divy_sharedcollection_iscreen;
typedef struct __divy_resourcestate_iscreen	divy_resourcestate_iscreen;
typedef struct __divy_clmodule_iscreen	divy_clmodule_iscreen;
typedef struct __divy_thumbnail_iscreen	divy_thumbnail_iscreen;
#ifdef DIVY_SUPPORT_PASSPOLICY
typedef struct __divy_passwordpolicy_iscreen	divy_passwordpolicy_iscreen;
typedef struct __divy_changepassword_iscreen	divy_changepassword_iscreen;
#endif	/* DIVY_SUPPORT_PASSPOLICY */
typedef struct __divy_changeowner_iscreen	divy_changeowner_iscreen;
typedef struct __divy_changeleader_iscreen	divy_changeleader_iscreen;
typedef struct __divy_confirmreading_iscreen	divy_confirmreading_iscreen;

/**
 * ユーザからの入力値を保持するScreen 構造体の定義
 * 主としてvalidate に使用します。
 */

/* ユーザ */
struct __divy_user_iscreen {

	char *usrid;                    /* ユーザID                   */
	char *password;                 /* パスワード                 */
	char *fullname;                 /* フルネーム                 */
	char *mailaddr;                 /* メールアドレス             */
	apr_int32_t adminmode;          /* 管理者モード               */
	char *susedst;                  /* ディスク使用容量(byte)     */
	char *smaxst;                   /* 最大使用可能容量(byte)     */
	char *susedres;                 /* 保持しているファイル数     */
	char *smaxres;                  /* 最大保持可能なファイル数   */
	apr_int32_t *accessdeny;        /* アクセス拒否リスト         */
	char *registdt;                 /* 登録日付を文字列で表した値 */
	char *updatedt;                 /* 更新日付を文字列で表した値 */
	char *comment;                  /* ユーザに関する短い説明     */
	char *sexpiration;              /* ユーザの有効期限           */
	divy_rdbo_extstatus *extstatus; /* ユーザ拡張ステータス       */

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	/* 旧バージョンのクライアントは幾つかのプロパティを提出しないので
	 * それを考慮するためのフラグ。1 であれば古いクライアントということ
	 * (note) ユーザ拡張ステータス以前のクライアントという意味になります */
	int is_old_client;

	char *smaxusercreation;	/* 管理できるユーザ数 (for グループリーダ) */
	char *allowhosts;				/* 接続可能なIPアドレス		  */

	/* 現在DBに格納されているユーザプロパティ */
	const divy_rdbo_usr *src_usr_pr;
};

/* グループ */
struct __divy_group_iscreen {

	char *grpid;		/* グループID                 */
	char *name;		/* グループ名称               */
	char *comment;		/* グループに関する短い説明   */
	char *registdt;		/* 登録日付を文字列で表した値 */
	char *updatedt;		/* 更新日付を文字列で表した値 */
	divy_rdbo_mailwatch *mailwatch_pr;	/* groupmailwatch */

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;
	divy_rdbo_extstatus *grp_extstatus; /* グループ拡張ステータス */

	divy_rdbo_upload_policy *upload_policy;	/* アップロードポリシー */
#ifdef DIVY_SUPPORT_GROUPQUOTA
	divy_rdbo_grpquota *grp_q;	/* グループQuota */
#endif /* DIVY_SUPPORT_GROUPQUOTA */

	/* 現在DBに格納されているグループプロパティ */
	const divy_rdbo_grp *src_grp_pr;
};

/* SQL */
struct __divy_sql_iscreen {

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	/* 入力SQLプロパティ */
	const divy_rdbo_sql *sql_pr;

	/* 現在DBに格納されているSQLプロパティ*/
	const divy_rdbo_sql *db_sql_pr;
};

/* システムメッセージ */
struct __divy_sysmsg_iscreen {

	char *msgid;		/* メッセージID (DB はinteger) */
	char *msg;		/* メッセージテキスト */
	char *subject;      /* 題名 */
	char *lang;		/* 言語  */
	char *scope;		/* 適用範囲 */
	char *opendt;		/* 公開日（日付を文字列で示した値） */
	char *closedt;		/* 停止日（日付を文字列で示した値） */
	apr_int32_t active;	/* アクティブ状態     */
	char *registdt;		/* 登録日付を文字列で表した値 */
	char *updatedt;		/* 更新日付を文字列で表した値 */

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;
};
/* DBMS情報 */ struct __divy_dbms_iscreen {

	char *dbmsid;		/* DBMS 識別番号             */
	char *type;		/* DBMS 種別                 */
	char *name;		/* DBMS 接続別識別名         */
	char *hostname;		/* DBMS 動作ホスト名         */
	char *sport;		/* DBMS 動作ポート番号       */
	char *dbname;		/* DBMS データベース名       */
	char *username;		/* DBMS 接続ユーザ名         */
	char *password;		/* DBMS 接続ユーザパスワード */
	char *comment;		/* コメント                  */
	divy_rdbo_dbmsactive active;	/* DBMS アクティブフラグ */

	char *registdt;		/* 登録日付を文字列で表した値 */
	char *updatedt;		/* 更新日付を文字列で表した値 */

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;
};

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/* システムQuota情報 */
struct __divy_sysquota_iscreen {

	divy_rdbo_qttype type;	/* Quota種別 */
	divy_statfs_t *sfs;	/* ファイルシステムの情報     */

	/* 数値系プロパティ(parse からvalidateに渡す際に利用) */
	char *s_usedst;
	char *s_maxst;
	char *s_total_bytes;
	char *s_avail_bytes;
	char *s_usedres;
	char *s_maxres;
	char *s_total_files;
	char *s_avail_files;

	/* assigned-quotaを無視するかどうか */
	int ignore_assigned_quota;

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	divy_rdbo_diskquota *src_sysquota_pr;
};
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * mailwatch プロパティ
 */
struct __divy_mailwatch_iscreen {

	/* 入力mailwatch */
	divy_rdbo_mailwatch *mwatch_pr;

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;
};

/**
 * sharedcollection プロパティ
 */
struct __divy_sharedcollection_iscreen {
	divy_rdbo_shlink *shlink_pr;

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;
};

/**
 * resourcestate プロパティ
 */
struct __divy_resourcestate_iscreen {

	char *uri;	/* 対象リソースのuri */

	int type;	/* 状態の種類 */

	/* type に応じて決まる値 */
	union {
		/* 公開されているかどうか(1: 公開されている / 0: 非公開) */
		int published;

		/* BOXの状態 (1：公開されている / 0:非公開 / 2: 期限切れ */
		int box;
	} state;

	/* type に応じて決まるオプション */
	union {
		/* BOXに関する情報 */
		divy_boxinfo *boxinfo;
	} optional;

	/* プロパティの更新動作(DBへの書き込み)をすべきかどうか
	 * (背景)
	 *   外部仕様では本来エラーとなるケースでもオペミスを想定し
	 *   正常終了させるよう努力している。このようなケースでは、
	 *   結果的にPROPPATCH動作を全くする必要がないこともあるので
	 *   それを識別する目的で以下を導入しました。*/
	int need_to_update;	/* 1: 書き込みが必要 / 0: 不要 */

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	/* 対象リソースのリソースタイプ */
	divy_rdbo_resourcetype resourcetype;

	/* もともと持っていたリソースの状態 */
	divy_rdbo_resourcestate *src_rstate_pr;

	divy_resourcestate_iscreen *next;	/* 次の属性へ */
};

/**
 * 更新クライアントプロパティ
 */
struct __divy_clmodule_iscreen {
	divy_rdbo_clmodule *clmodule_pr;

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;
};

/**
 * thumbnail プロパティ
 */
struct __divy_thumbnail_iscreen {
	/* サムネイルデータ */
	char *data;

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;
};

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティ
 */
struct __divy_passwordpolicy_iscreen {

	/* ポリシーを有効にするかどうか(1: 有効にする / 0: 無効にする) */
	int on;

	/* 最小パスワード長さ(NULL: 制限なし / NOT NULL (n) : n文字制限) */
	char *sminlen;	

	/* パスワード更新周期(有効期限)(日). (NULL: 無期限 / NOT NULL (n): n 件) */
	char *schange_pw_cycle;

	/* パスワード期限切れ後の猶予期間(日) (NULL: 猶予なし / n: n 件) */
	char *sprobation;

	/* 禁則文字一覧 (NULL: 禁則なし / NOT NULL: その値(をchar 毎に分割したもの) */
	char *ngword;

	/* パスワード履歴保持件数. (NULL: 0件 / NOT NULL (n): n 件) */
	char *shistory_pw_num;

	/* ユーザID文字列を含めてはならない (1: 含めてはならない / 0: 任意) */
	int denyUseridString;

	/* 記号が必須かどうか (0: 任意 / 1: 必須) */
	int isSymbolChrNeeded;

	/* アルファベット大文字が必須かどうか (0: 任意 / 1: 必須) */
	int isUpAlphabet;

	/* アルファベット小文字が必須かどうか (0: 任意 / 1: 必須) */
	int isLoAlphabet;

	/* 数値が必須かどうか (0: 任意 / 1: 必須) */
	int isNumericNeeded;

	/* 連続文字を許可するかどうか (0: 任意 / 1: 許可しない) */
	int denyCycleChr;

	/* 初回ログインパスワード変更が必要かどうかと、
	 * 必要な場合には何日以内の変更が必要かどうか表す数値
	 *
	 * (NULL: 変更しなくていい /
	 *  "0" : 変更しない限り利用できない(パスワード変更以外は)
	 *  "n" : n日以内に変更しなければならない)  */
	char *sfirstlogin_limitday;

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	divy_rdbo_passwordpolicy *passpolicy_pr;
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * changepassword プロパティ
 */
struct __divy_changepassword_iscreen {
	char *password;	/* パスワード */

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	/* 現在DBに格納されているユーザプロパティ */
	const divy_rdbo_usr *src_usr_pr;
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/*
 * changeowner プロパティ
 */
struct __divy_changeowner_iscreen {

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	/* 現在DBに格納されているユーザプロパティ */
	const divy_rdbo_usr *src_usr_pr;
};

/*
 * changeleader プロパティ
 */
struct __divy_changeleader_iscreen {

	/* 新しくグループリーダとなるユーザのユーザID */
	char *ownerid;

	/* アクション */
	enum {
		DIVY_CHLEADER_ACTION_UNKNOWN = 0,	/* 未定義 */
		DIVY_CHLEADER_ACTION_APPOINT,		/* 任命   */
		DIVY_CHLEADER_ACTION_DISMISS,		/* 解任   */
	} action;

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	/* 現在DBに格納されているグループプロパティ */
	const divy_rdbo_grp *src_grp_pr;

	/* validator でセットされるオーナーのユーザ情報 (内部用) */
	divy_rdbo_usr *usr_pr;
};

/*
 * confirmreading プロパティ
 */
struct __divy_confirmreading_iscreen {

	char *uri;				/* リソースのURI */
	char *userid;			/* ユーザID */
	time_t creationdate;	/* 開封通知を付けた日付 */

	/* 対象リソースのURI解析結果 */
	divy_uri_spec *u_spec;

	/* このプロパティが付いているリソースの情報 */
	const divy_rdbo_resource *rdb_r;
};


/*------------------------------------------------------------------------------
  Declar public functions
  ----------------------------------------------------------------------------*/
/**
 * mailwatch プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param mwatch_pr divy_rdbo_mailwatch ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_mailwatch_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_mailwatch **mwatch_pr); 

/**
 * sharedcollection プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * shlink_pr エレメントの中身
 * @param shlink_pr divy_rdbo_shlink ** 解析したsharedcollection タグの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_sharedcollection_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_shlink **shlink_pr);

/**
 * updatediscovery プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * updatediscovery エレメントの中身
 * @param clmodule_pr divy_rdbo_clmodule ** 解析したupdatediscovery タグの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_updatediscovery_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_clmodule **clmodule_pr);

/**
 * keylist プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * keylist エレメントの中身
 * @param keylist_pr divy_rdbo_keylist ** 解析したkeylist タグの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_keylist_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_keylist **keylist_pr);

/**
 * userdiscovery プロパティをパースする。
 * (note)
 * 	accessdeny メンバは、accessdenyタグが指定されていない or
 * 	EMPTYタグで指定された場合には、NULLになります。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param user_iscreen divy_user_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_user_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_user_iscreen **user_iscreen);

/**
 * groupdiscovery プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param group_iscreen divy_group_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_group_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_group_iscreen **group_iscreen);

/**
 * sqldiscovery プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param sql_pr divy_rdbo_sql ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_sql_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_rdbo_sql **sql_pr);

/**
 * sysmsgdiscovery プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param sysmsg_iscreen divy_sysmsg_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_sysmsg_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_sysmsg_iscreen **sysmsg_iscreen);

/**
 * dbmsdiscovery プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param dbms_iscreen divy_dbms_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_dbms_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_dbms_iscreen **dbms_iscreen);

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * systemquota プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param sysquota_iscreen divy_sysquota_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_systemquota_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_sysquota_iscreen **sysquota_iscreen);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * resourcestate プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param rstate_iscreen divy_resourcestate_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_resourcestate_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_resourcestate_iscreen **rstate_iscreen);

/**
 * thumbnail プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param thumbnail_iscreen divy_thumbnail_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_thumbnail_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_thumbnail_iscreen **thumbnail_iscreen);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param passpolicy_iscreen divy_passwordpolicy_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_passwordpolicy_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_passwordpolicy_iscreen **passpolicy_iscreen);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * changepassword プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param changepassword_iscreen divy_changepassword_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_changepassword_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_changepassword_iscreen **changepassword_iscreen);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * changeowner プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param changeleader_iscreen divy_changeowner_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_changeowner_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_changeowner_iscreen **changeowner_iscreen);

/**
 * changeleader プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param changeleader_iscreen divy_changeleader_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_changeleader_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_changeleader_iscreen **changeleader_iscreen);

/**
 * confirmreading プロパティをパースする。
 *
 * @param p apr_poot_t *
 * @param elem const apr_xml_elem * Liveプロパティの中身を持つエレメント
 * @param confirmreading_iscreen divy_confirmreading_iscreen ** パースしたLiveプロパティの内容
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_parse_confirmreading_property(apr_pool_t *p,
					const apr_xml_elem *elem,
					divy_confirmreading_iscreen **confirmreading_iscreen);


/*------------------------------------------------------------------------------
  map iscreen to rdbo object
  ----------------------------------------------------------------------------*/
/**
 * user_iscreen を grp_pr にマップする。
 *
 * @param p apr_pool_t *
 * @param user_iscreen const divy_user_iscreen *
 * @param usr_pr divy_rdbo_usr **
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_map_user_property(apr_pool_t *p,
					const divy_user_iscreen *user_iscreen,
					divy_rdbo_usr **usr_pr);

/**
 * group_iscreen を grp_pr にマップする。
 *
 * @param p apr_pool_t *
 * @param group_iscreen const divy_group_iscreen *
 * @param grp_pr divy_rdbo_grp **
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_map_group_property(apr_pool_t *p,
					const divy_group_iscreen *group_iscreen,
					divy_rdbo_grp **grp_pr);

/**
 * sysmsg_iscreen を sysmsg_pr にマップする。
 *
 * @param p apr_pool_t *
 * @param sysmsg_iscreen const divy_sysmsg_iscreen *
 * @param sysmsg_pr divy_rdbo_sysmsg **
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_map_sysmsg_property(apr_pool_t *p,
					const divy_sysmsg_iscreen *sysmsg_iscreen,
					divy_rdbo_sysmsg **sysmsg_pr);

/**
 * dbms_iscreen を dbms_pr にマップする。
 *
 * @param p apr_pool_t *
 * @param dbms_iscreen const divy_dbms_iscreen *
 * @param dbms_pr divy_rdbo_dbms **
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_map_dbms_property(apr_pool_t *p,
					const divy_dbms_iscreen *dbms_iscreen,
					divy_rdbo_dbms **dbms_pr);

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * sysquota_iscreen を sysquota_pr にマップする。
 *
 * @param p apr_pool_t *
 * @param sysquota_iscreen const divy_sysquota_iscreen *
 * @param sysquota_pr divy_rdbo_diskquota **
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_map_systemquota_property(apr_pool_t *p,
					const divy_sysquota_iscreen *sysquota_iscreen,
					divy_rdbo_diskquota **sysquota_pr);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * rstate_iscreen を rstate_pr にマップする。
 *
 * @param p apr_pool_t *
 * @param rstate_iscreen const divy_resourcestate_iscreen *
 * @param rstate_pr divy_rdbo_resourcestate **
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_map_resourcestate_property(apr_pool_t *p,
					const divy_resourcestate_iscreen *rstate_iscreen,
					divy_rdbo_resourcestate **rstate_pr);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passpolicy_iscreen をpasspolicy_pr にマップする.
 *
 * @param p apr_pool_t *
 * @param rpasspolicy_iscreen const divy_passwordpolicy_iscreen *
 * @param passpolicy_pr divy_rdbo_passwordpolicy **
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_map_passwordpolicy_property(apr_pool_t *p,
					const divy_passwordpolicy_iscreen *passpolicy_iscreen,
					divy_rdbo_passwordpolicy **passpolicy_pr);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_PARSER_H */

