/**
 * $Id$
 *
 * TeamFile プラグインインターフェース (共通)
 *
 * [ 役割 ]
 *   * TeamFile 本体からプラグイン呼び出しコードの共通化
 */
#ifndef INCLUDE_TF_PLUGIN_H
#define INCLUDE_TF_PLUGIN_H

#include "httpd.h"
#include "apr.h"
#include "apr_pools.h"
#include "mod_dav_tf.h"
#include "tf_folder.h"
#include "tf_rdbo.h"

#ifdef DIVY_SUPPORT_PLUGIN
/* Server Plugin Headers */
#include "tfsp.h"
#include "tfsp_spec.h"
#include "tfsp_ihdr.h"
#include "tfsp_ihdr_ccipher.h"
#include "tfsp_loader.h"
#endif	/* DIVY_SUPPORT_PLUGIN */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef DIVY_SUPPORT_PLUGIN
/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
/**
 * プラグイン処理のステータスコード
 */
enum {
	DIVY_PL_ST_OK = 0,		/* 正常終了 */
	DIVY_PL_ST_ERR,			/* 予期しないエラー */
	DIVY_PL_ST_NOPLUGIN,	/* 対象のプラグインが存在しなかった */
	DIVY_PL_ST_NOPROCESS,	/* 処理は行なわれなかった */
	DIVY_PL_ST_NOLICENSE,	/* プラグインライセンスが無かった */
};

/**
 * ロケーションメニューに表示する情報を表す構造体
 */
typedef struct divy_lmenu_item	divy_lmenu_item;
struct divy_lmenu_item {
	char *uri;			/* 設定画面のURI(ロケーション名を含む相対URI) */
	char *iconpath;		/* 表示アイコンのパス(相対URI) */
	char *displayname;	/* 画面表示用名称 */
	char *displayname_en;	/* 画面表示用名称(英語) */
	char *screentype;	/* FILE, PERSONAL, UTILITY */
	char *usertype;		/* 利用できるユーザの種類 admin, normal, limited 複数の時にはハイフンで区切る */
};

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * TeamFile プラグインローダーの初期化.
 *
 * @param pluginroot const char * プラグインルートディレクトリパス
 * @param pconf apr_pool_t * プラグインローダーをアロケートするプール
 *                       (プロセスプール(server->process->pconf) でなければならない)
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_pi_init(const char *pluginroot, apr_pool_t *pconf);

/**
 * プラグインのChildプロセス生成時の初期化処理をコールバックする.
 * [ライフサイクル管理用]
 *
 * @param pchild apr_pool_t * Childスコープのプール
 * @param s server_rec *
 */
DIVY_DECLARE(void) divy_pi_cinit(apr_pool_t *pchild, server_rec *s);

/**
 * プラグインのリクエストレベルの初期化処理.
 * [ライフサイクル管理用]
 *
 * @param r request_rec *
 */
DIVY_DECLARE(void) divy_pi_rinit(request_rec *r);

/**
 * プラグイン種類ptype, 名称pname のプラグインインターフェースを取得する.
 * pname がNULLであれば、同一のプラグイン種類の内、プラグインチェインの先頭にある
 * プラグインインターフェースを取得する.
 *
 * @param r request_rec *
 * @param ptype tfsp_type_plugin プラグイン種別
 * @param pname const char * プラグインの名前
 * @return void * プラグインインターフェースへのポインタ
 */
DIVY_DECLARE(void *) divy_pi_lookupIf(request_rec *r, tfsp_type_plugin ptype,
                                      const char *pname);

/**
 * プラグイン種類ptype, 名称pname のプラグイン構造体へのポインタを取得する.
 * pname がNULLであれば、同一のプラグイン種類の内、プラグインチェインの先頭にある
 * プラグインイン構造体を取得する.
 *
 * @param r request_rec *
 * @param ptype tfsp_type_plugin プラグイン種別
 * @param pname const char * プラグインの名前
 * @return tfsp_plugin_module * プラグインイン構造体へのポインタ
 */
DIVY_DECLARE(const tfsp_plugin_module *) divy_pi_lookupPm(request_rec *r,
                                      tfsp_type_plugin ptype, const char *pname);

/**
 * この関数呼び出し時点で利用されているプラグインローダーのプラグインルート
 * ディレクトリパスを取得する.
 *
 * @param r request_rec *
 * @return const char * 取得したプラグインルートパス
 */
DIVY_DECLARE(const char *) divy_pi_get_pluginroot(request_rec *r);

/**
 * 各プラグイン毎のインストールパス名を組み立てて返却する.
 * ($plugin_root/$plugin_name)
 *
 * @param r request_rec *
 * @param plugin_name char * プラグイン名称
 */
DIVY_DECLARE(char *) divy_pi_build_plugin_path(request_rec *r, char *plugin_name);

/**
 * 各プラグイン毎のコンフィグパス名を組み立てて返却する.
 * ($plugin_root/$plugin_name/conf)
 *
 * @param r request_rec *
 * @param plugin_name char * プラグイン名称
 */
DIVY_DECLARE(char *) divy_pi_build_plugin_confpath(request_rec *r, char *plugin_name);

/**
 * プラグイン以下へのアクセスURI から対象ファイルまたはディレクトリの物理パスを算出する.
 *
 * @param r request_rec *
 * @param u_spec divy_uri_spec *
 */
DIVY_DECLARE(char *) divy_pi_uri2path(request_rec *r, divy_uri_spec *u_spec);

/**
 * infotype をtfsp_type_folder に変換する.
 *
 * @param infotype divy_infotype
 * @param is_dst int Destinationリソースかどうか(1/0)
 * @return tfsp_type_folder フォルダ種別 (see tfsp_spec.h)
 */
DIVY_DECLARE(tfsp_type_folder) divy_pi_infotype2foldertype(divy_infotype infotype, int is_dst);

/**
 * method をtfsp_type_operation に変換する.
 *
 * @param r request_rec *
 * @param src_rdb_r divy_rdbo_resource *
 * @param dst_rdb_r divy_rdbo_resource *
 * @return tfsp_type_operation オペレーション種別 (see tfsp_spec.h)
 */
DIVY_DECLARE(tfsp_type_operation) divy_pi_method2operation(request_rec *r,
                                        divy_rdbo_resource *src_rdb_r,
                                        divy_rdbo_resource *dst_rdb_r, int is_update);

/**
 * rdb_r が示すリソースの実体暗号化を行なう.(実体暗号化プラグインラッパー)
 *
 * @param r request_rec *
 * @param rdb_r divy_rdbo_resource * 対象リソース
 * @param filepath char * 物理ファイルのパス
 * @param pcfile tfsp_ccipher_file ** 暗号化されたファイルに関する情報
 * @return int 処理ステータス
 *		DIVY_PL_ST_OK       : 正常
 *		DIVY_PL_ST_ERR      : 予期しないエラー
 *		DIVY_PL_ST_NOPLUGIN : 対象とするプラグインが存在しなかった
 */
DIVY_DECLARE(int) divy_pi_ccipher(request_rec *r, divy_rdbo_resource *rdb_r,
                                  const char *filepath, tfsp_ccipher_file **pcfile);

/**
 * ロケーションメニューに表示する表示項目を取得する
 *
 * @param r request_rec *
 * @param items tfs_varray_t ** divy_lmenu_item * を要素に持つ可変長配列
 * @return int 処理ステータス
 *      DIVY_PL_ST_OK       : 正常
 *      DIVY_PL_ST_NOPLUGIN : プラグインが存在しなかった
 *      DIVY_PL_ST_ERR      : 予期しないエラー
 */
DIVY_DECLARE(int) divy_pi_get_locationmenu_items(request_rec *r,
                                                 tfs_varray_t **items);

/**
 * 現在のロケーションのグループのうち、実体暗号化対象になっている
 * グループのグループID一覧を取得する. (実体暗号化プラグインラッパー)
 *
 * @param r request_rec *
 * @param groupid_set tfs_hset_t ** グループIDからなるセット
 * @return int 処理ステータス
 *		DIVY_PL_ST_OK       : 正常
 *		DIVY_PL_ST_ERR      : 予期しないエラー
 *		DIVY_PL_ST_NOPLUGIN : 対象とするプラグインが存在しなかった
 */
DIVY_DECLARE(int) divy_pi_cciphered_grpset(request_rec *r, tfs_hset_t **groupid_set);

/**
 * 指定されたグループIDが示すグループが実体暗号化対象のグループであれば
 * プラグインに対象グループが削除されることを通知する.
 *
 * @param r request_rec *
 * @param groupid const char * グループID
 * @return int 処理ステータス
 *		DIVY_PL_ST_OK       : 正常
 *		DIVY_PL_ST_ERR      : 予期しないエラー
 *		DIVY_PL_ST_NOPLUGIN : 対象とするプラグインが存在しなかった
 */
DIVY_DECLARE(int) divy_pi_excludeTarget(request_rec *r, const char * groupid);

/**
 * r にキャッシュされているリクエストレベルプラグインプールを取得する.
 *
 * @param r request_rec *
 * @return tfs_pool_t *
 */
DIVY_DECLARE(tfs_pool_t *) divy_get_pi_req_pool(request_rec *r);

#endif	/* DIVY_SUPPORT_PLUGIN */

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_PLUGIN_H */

