/**
  $Id$
 *
 * tf_rdbo.h
 *
 * RDBのデータ構造をDAVプロバイダ(repos.c, livepro.cなど)から
 * 隠蔽する役割を持つ。(RDB <-> DAVプロバイダ 間のインターフェース)
 * DAVプロバイダでは"ありません"。
 *
 * [役割]
 *
 * ・DAVプロバイダから、DBのテーブル構成やDBオブジェクトの構造を
 *   隠蔽する。
 * ・RDB操作用の共通関数を提供すること。
 * ・DBMSの種類を変更したときに、特定のDBMSに依存したSQLの定義場所を
 *   このヘッダを取り込むtf_rdbo.c に集中させること。
 * ・DBとDAVプロバイダ間の結合度を下げること。
 *
 * このヘッダファイルに定義された関数、構造体には、全体として、以下の
 *  基本思想に基づいて設計・実装されています。
 *
 * ・RDBの構造(テーブル、DBオブジェクト)やDBオブジェクトの構成は
 *   知っている。(つまり、知っていてもよい)
 *
 * ・ユースケースに応じたDBの引き方も知っている。
 *   (例えば、ユーザ情報を取得するというユースケースでは、どの
 *    テーブルを結合すべきなのか理解している)
 *
 * ・tf_rdbo.c : WebDAVのプロトコルを殆ど理解していない
 *               自分が実装した関数がどのプロバイダで使われるのか
 *               全く理解していない。
 *   tf_rdbo.h : RDBにマッピングする上で必要なWebDAV上の概念は
 *               多少理解している。(Deadプロパティはどのテーブルに
 *               保持しているとか。)
 *
 * 変更履歴
 *
 * 2003/03/06 Thu takehara NEW
 *
 */
#ifndef INCLUDE_TF_RDBO_H
#define INCLUDE_TF_RDBO_H

#include "httpd.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "mod_dav.h"

#if APR_HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#include "tfr.h"
#include "mod_dav_tf.h"
#include "tf_db.h"
#include "tf_storage.h"
#include "tf_folder.h"
#include "tf_rdbo_util.h"
#include "lock.h"
#include "search.h"
#include "util_db.h"
#include "tf_cgi.h"
#include "tf_linkedlist.h"
#include "tf_autodelete.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*--------------------------------------------------------------
  Define fixed values and macros
  --------------------------------------------------------------*/
/**
 * 成功(0)/失敗(1)以外のステータスを返す際に使用する値
 */
#define DIVY_STCODE_QUOTAOVER 	  2	/* リソースのDiskOver */
#define DIVY_STCODE_RESOURCEEXIST 3	/* プライベート,グループコレクションが存在する */
#define DIVY_STCODE_REQUIRED_NF   4	/* 指定されたRequiredsql, 名前付きバインド変数が存在しない */
#if 0	/* 廃止 */
#define DIVY_STCODE_LKDBNOSQL      11	  /* 指定されたSQL文が存在しない */
#define DIVY_STCODE_LKDBINVALIDSQL 12	  /* 指定されたSQL文が正しくない */
#define DIVY_STCODE_LKDBINVALIDBRVALUE 13 /* バインド変数/requiredが正しくない */
#define DIVY_STCODE_INVALID_DBNAME 14	  /* 指定された使用dbms名称が正しくない */
#endif
#define DIVY_STCODE_SHCOL_NOTFOUND	15	/* 指定された共有コレクションが存在しなかった */ 
#if 0	/* 廃止 */
#define DIVY_STCODE_LKDBREPPSERR	16	/* repositorysql パースエラー */
#endif

/* 以下はsqlinformationsearch用 */
#define DIVY_STCODE_SQLISDEPENDNOTOPURI   21	/* dependlist でトップSQL がない */
#define DIVY_STCODE_SQLISFAILEDDEPENDENCY 22	/* Failed Dependency */

/* 以下はdivy_rdbo_exec_sql 用 */
#define DIVY_STCODE_TOOMANY_COLUMN	23	/* SELECT句のカラムが多い   */      
#define DIVY_STCODE_TOOFEW_COLUMN	24	/* SELECT句のカラムが少ない */

/* for divy_rdbo_get_userquota */
#define DIVY_STCODE_USER_NOT_EXIST	25	/* ユーザが存在しなかった */

/* for divy_rdbo_get_autoindex_data, divy_rdbo_get_group_resource_by_userid */
#define DIVY_STCODE_REMAIN_DATA		26	/* まだ取得可能なデータが残っている */

/* for divy_rdbo_get_systemquota */
#define DIVY_STCODE_SYSQUOTA_NOT_EXIST	27	/* システムQuotaエントリが無かった */

/* リソース存在しないことを表す定義値 */
#define DIVY_STCODE_NOT_EXIST		28

/* 親リソースが存在しなかった */
#define DIVY_STCODE_PARENT_NOT_EXIST	29

#define DIVY_STCODE_EFFECTIVE_RSTATE_EXIST	30	/* 有効なresourcestate が存在した   */
#define DIVY_STCODE_INEFFECTIVE_RSTATE_EXIST	31	/* 無効なresourcestate が存在した */

/* for divy_rdbo_get_groupquota */
#define DIVY_STCODE_GRPQUOTA_NOT_EXIST 32 /* グループQuotaエントリが無かった */

/**
 * 状態/属性プロパティが"有効"であることを判定するマクロ (1: 有効)
 */
#define IS_EFFECTIVE_STATE(rt) (rt != NULL && (rt)->effective == 1)

/**
 * 状態/属性プロパティが"無効"であることを判定するマクロ (1: 無効)
 * (note)
 * 	明示的に"無効化"されているかどうかまでは判りません。
 * 	自前で判定して下さい。
 */
#define IS_INEFFECTIVE_STATE(rt) (rt == NULL || (rt)->effective == 0)

/**
 * 関数 divy_rdbo_get_hierarchy_property で使用される
 * 取得プロパティのビット値
 */
#define DIVY_GET_PROP_fullname	    0x01	/* resourceのフルネームプロパティを取得する */
#define DIVY_GET_PROP_dead	    0x02	/* Dead プロパティを取得する */
#define DIVY_GET_PROP_mailwatch	    0x04	/* mailwatch プロパティを取得する */
#define DIVY_GET_PROP_shared	    0x08	/* 共有コレクションプロパティを取得する */
#define DIVY_GET_PROP_trash	    0x10	/* ごみ箱のプロパティを取得する */
#define DIVY_GET_PROP_notrashfolder 0x20	/* ごみ箱だけを除外する */
#define DIVY_GET_PROP_resourcestate 0x40	/* 状態, 属性プロパティを取得する */
#define DIVY_DEL_PROP_noviewattr    0x80	/* View属性が"付いていない"リソースを削り落とす(基点は除外) */
#define DIVY_GET_PROP_groupstate	0x100	/* groupstate を取得する */
#define DIVY_GET_PROP_confirmreading	0x200	/* confirmreading を取得する */

/**
 * リポジトリDBを表す特別なDBMS ID
 * (note)
 *   divy_dbms テーブルに挿入されるデータのうち、以下で定義されたDBMS ID のエントリは
 *   リポジトリDBのものであるとして、特別扱いします.
 */
#define DIVY_REPOS_DBMSID	"0000"


/*
 * 関数 divy_rdbo_copy_groupmailwatch_to_user で使用する定義値
 */
#define DIVY_CPMWATCH_NORMAL      0x00	/* grpid と usrid を利用する */
#define DIVY_CPMWATCH_IGNORE_USER 0x01	/* usrid を無視する          */

/*--------------------------------------------------------------
  Declare structure
  --------------------------------------------------------------*/
typedef struct __divy_rdbo_resource    divy_rdbo_resource;
typedef struct __divy_rdbo_dproperty   divy_rdbo_dproperty;
typedef struct __divy_rdbo_usr         divy_rdbo_usr;
typedef struct __divy_rdbo_grp         divy_rdbo_grp;
typedef struct __divy_rdbo_sql         divy_rdbo_sql;
typedef struct __divy_rdbo_sqlcontent  divy_rdbo_sqlcontent;
typedef struct __divy_rdbo_reqsql      divy_rdbo_reqsql;
typedef struct __divy_rdbo_sqldepend   divy_rdbo_sqldepend;
typedef struct __divy_rdbo_shlink      divy_rdbo_shlink;
typedef struct __divy_rdbo_shsrccol    divy_rdbo_shsrccol;
typedef struct __divy_rdbo_dbms        divy_rdbo_dbms;
typedef struct __divy_rdbo_sysmsg      divy_rdbo_sysmsg;
typedef struct __divy_rdbo_trash       divy_rdbo_trash;
typedef struct __divy_rdbo_clmodule    divy_rdbo_clmodule;
typedef struct __divy_rdbo_mailwatch   divy_rdbo_mailwatch;
typedef struct __divy_rdbo_keylist     divy_rdbo_keylist;
#ifdef DIVY_SUPPORT_EXTENDQUOTA
typedef struct __divy_rdbo_diskquota   divy_rdbo_diskquota;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
typedef struct __divy_rdbo_resourcestate divy_rdbo_resourcestate;
typedef struct __divy_rdbo_rusr        divy_rdbo_rusr;
typedef struct __divy_rdbo_rsql        divy_rdbo_rsql;
typedef struct __divy_rdbo_rgrp        divy_rdbo_rgrp;
typedef struct __divy_rdbo_upload_policy divy_rdbo_upload_policy;
#ifdef DIVY_SUPPORT_GROUPQUOTA
typedef struct __divy_rdbo_grpquota    divy_rdbo_grpquota;
#endif /* DIVY_SUPPORT_GROUPQUOTA */

typedef struct __divy_rdbo_lock_resource	divy_rdbo_lock_resource;
typedef struct __divy_rdbo_lock			divy_rdbo_lock;
typedef struct __divy_rdbo_search		divy_rdbo_search;
typedef struct __divy_rdbo_search_output	divy_rdbo_search_output;
typedef struct __divy_rdbo_search_params	divy_rdbo_search_params;
#ifndef DIVY_DECLARE_STRUCT_RSVALUE
#define DIVY_DECLARE_STRUCT_RSVALUE
typedef struct __divy_rdbo_rsvalue		divy_rdbo_rsvalue;
#endif	/* DIVY_DECLARE_STRUCT_RSVALUE */
typedef struct __divy_rdbo_rsvalueset		divy_rdbo_rsvalueset;

typedef struct __divy_rdbo_clist		divy_rdbo_clist;
typedef struct __divy_rdbo_sortorder	divy_rdbo_sortorder;
#ifdef DIVY_SUPPORT_PASSPOLICY
typedef struct __divy_rdbo_passwordpolicy   divy_rdbo_passwordpolicy;
typedef struct __divy_rdbo_passhistory      divy_rdbo_passhistory;
typedef struct __divy_rdbo_passngword       divy_rdbo_passngword;
typedef struct __divy_rdbo_passpolicystatus divy_rdbo_passpolicystatus;
#endif	/* DIVY_SUPPORT_PASSPOLICY */
typedef struct __divy_rdbo_confirmreading	divy_rdbo_confirmreading;
typedef struct __divy_rdbo_session			divy_rdbo_session;
typedef struct __divy_rdbo_box				divy_rdbo_box;

/*--------------------------------------------------------------
  Define incomplete type structure
  --------------------------------------------------------------*/
/**
 * "リソース"や"コレクション"として扱われる概念に関連する、または
 * 処理するために必要なデータ構造のうち、D-IVYで独自に定義した
 * データを保持しています。
 *
 * Resource descriptor (dav_resource)がこの構造体へのポインタを持っています。
 * この構造体は、dav_divy_get_resourceによって生成されます。
 * repository-specific information for a resource.
 */
struct dav_resource_private {
	
	/* 対象resourceと関連する情報を表すデータ構造へのポインタ */
	divy_rdbo_resource *rdb_r;

	/* "ネームスペースID" -> "ネームスペースURI"の関係を持つハッシュ */
	apr_hash_t *ns_id_hash;	

	/* "ネームスペースURI" -> "ネームスペースID"の関係を持つハッシュ */
	apr_hash_t *ns_uri_hash;

	union {
		/* システムメッセージへのリンクを表すポインタ */
		divy_rdbo_sysmsg *sysmsg_pr;

		/* クライアントモジュール情報へのリンクを表すポインタ */
		divy_rdbo_clmodule *clmodule_pr;

		/* ユーザリレーションへのリンクを表すポインタ */
		divy_rdbo_rusr *rusr_pr;

		/* SQLリレーションへのリンクを表すポインタ */
		divy_rdbo_rsql *rsql_pr;

		/* グループリレーションへのリンクを表すポインタ */
		divy_rdbo_rgrp *rgrp_pr;
#ifdef DAV_SUPPORT_POST_HANDLING
		/* CGIプロパティを表す構造体へのポインタ */
		divy_cgi_property *cgi_pr;
#endif	/* DAV_SUPPORT_POST_HANDLING */
	} list;

	/* 対象リソースの親リソース情報を表す構造体へのポインタ
	 * (note)
	 *   この構造体は性能向上を目的に導入されました。本来
	 *   dav_resource_private を表すメンバとしては必要のないもの
	 *   です。気にしないで下さい。
	 */
	dav_resource *parent_resource;

	/* (Live プロパティのPROPPATCH 専用)
	 * この構造体が表すリソースのLiveプロパティを変更する間だけ有効な
	 * DBトランザクションを保持するトランザクションコンテキスト
	 */
	divy_db_transaction_ctx *liveprop_ts_ctx;

	/* このリクエスト時にApache から渡された request_recへのポインタ */
	request_rec *r;	/* HTTPヘッダの取得処理などに使用する */
};

/*--------------------------------------------------------------
  Define structures and enums
  --------------------------------------------------------------*/
/**
 * resourcetypeの値
 */
enum __divy_rdbo_resourcetype {
	DIVY_TYPE_RESOURCE,	/* resourceである   */
	DIVY_TYPE_COLLECTION,	/* collectionである */
	DIVY_TYPE_UNKNOWN       /* resourcetype が分からなかった */
};

typedef enum __divy_rdbo_resourcetype  divy_rdbo_resourcetype;

/**
 * "リソース"や"コレクション"として扱われる概念が持っているデータを表す構造体。
 *
 * 上記の概念が保持すべきデータは、RDBのテーブルが管理＆格納して
 * いますが、これらテーブルの情報を構造体で表したもの、ともいえます。
 *
 * この構造体は、dav_resourceのdav_resource_privateが持っています。
 * SEARCHプロバイダ以外は、dav_resource_privateから取得すべきです。
 *
 * (note)
 * 	必ずしも、resource = ファイルとは限らないことに注意。
 * 	resource = ユーザ、グループもあります。
 *
 * リポジトリDB上のテーブル dav_resourceをコード(データ構造)に
 * マッピングする役割を持つ。(O-Rマッピング用構造体とも言える)
 */
struct __divy_rdbo_resource {

	char *rsid;                    /* dav_resourceのシリアルID  */
	char *uri;                     /* URI                       */
	time_t creationdate;           /* 作成日付(GMTベース)       */
	char *displayname;             /* 表示名称(displayname)     */
	char *getcontentlanguage;      /* Content-Language          */
	apr_int64_t getcontentlength;  /* Content-Length            */
	char *getcontenttype;          /* Content-Type              */
	char *getetag;                 /* Etag                      */
	time_t getlastmodified;        /* Last-Modified(GMTベース)  */
	divy_rdbo_resourcetype resourcetype;   /* resourcetype */
	int depth;                     /* このリソースが存在する階層数(topが1) */
	apr_int32_t isversioned;       /* バージョンを持っているか  */
	char *creator;                 /* 作成者 		    */
	char *lastmodifier;            /* 最終更新者		    */
	char *creator_userid;          /* creator のユーザID        */
	char *lastmodifier_userid;     /* lastmodifier のユーザID   */
	apr_int32_t checkin;           /* checkinバージョン         */
	apr_int32_t checkout;          /* checkoutバージョン        */
	char *physicalpath;            /* リソースの格納物理パス    */

	/* 顧客DB検索結果のキー情報を保持する構造体へのポインタ */
	divy_rdbo_keylist *keylist_pr;

	/* このリソースのURI の種類を表す構造体へのポインタ */
	divy_uri_spec *u_spec;	

	/* Deadプロパティを示す構造体へのポインタ */
	divy_rdbo_dproperty *d_pr;
	
	/* 共有コレクションへのリンクを表すポインタ */
	divy_rdbo_shlink *shlink_pr;

	/* メールによるフォルダ監視状態を保持するプロパティへのポインタ */
	divy_rdbo_mailwatch *mwatch_pr;

	/* ごみ箱用に拡張されたプロパティ */
	divy_rdbo_trash *trash_pr;

#ifdef DIVY_SUPPORT_EXTENDQUOTA
	/* システムQuota情報へのリンクを表すポインタ */
	divy_rdbo_diskquota *sysquota_pr;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

	/* 属性、状態を表すプロパティ */
	divy_rdbo_resourcestate *rstate_pr;

	/* ユーザプロパティを表す構造体へのポインタ */
	divy_rdbo_usr *usr_pr;

	/* グループプロパティを表す構造体へのポインタ */
	divy_rdbo_grp *grp_pr;

	/* SQL プロパティを表す構造体へのポインタ */
	divy_rdbo_sql *sql_pr;

	/* DBMS 情報へのリンクを表すポインタ */
	divy_rdbo_dbms *dbms_pr;

	/* 開封通知プロパティ */
	divy_rdbo_confirmreading *confirmreading_pr;

	/*
	 * search 検索結果で responce には含まれないが
	 * 返却する必要のある情報を持つ構造体へのポインタ
	 */
	divy_rdbo_search *search_pr;

	/* リクエストプール (divy_rdbo_resource の操作に使用します) */
	apr_pool_t *p;

	/* 
	 * 次の divy_rdbo_resource 構造体へのポインタ。階層的にリソースや
	 * コレクションを持つ場合、この next でリンクしています。
	 */
	divy_rdbo_resource *next;
};

/**
 * Deadプロパティを表す構造体 (dav_dead_propertyテーブルマッピング)
 * 一つのresourceに対し、複数のDeadプロパティが存在する場合を、
 * 配列ではなく、次のDeadプロパティ構造体へのリンクを持つことで表現している。
 *
 * [役割]
 *  リポジトリDB上のテーブル dav_dead_propertyをオブジェクトにマップすること。
 */
struct __divy_rdbo_dproperty {
	
	char *rsid;                    /* dav_namespaceのシリアルID       */
	apr_int32_t ns_id;             /* dav_namespaceのネームスペースID */
	const char *name;              /* プロパティ名                    */
	const char *value;             /* プロパティ値                    */
	const char *lang_tag;          /* 言語タグ属性値                  */

	/* (Deadプロパティプロバイダの内部処理専用)
	 * アクションを保持するビット値
	 */
#define DIVY_DP_NOTHING		0x00	/* 何もしていない */
#define DIVY_DP_APPLAYPATCH	0x01	/* パッチアクションを実施していた */
#define DIVY_DP_SET			0x02	/* setを実施していた */
#define DIVY_DP_REMOVE		0x04	/* remove を実施していた(set とは同居不可) */
	int patch_action;

	/*
	 * 同じrsidに対して複数のDeadプロパティが存在する場合には、
	 * 次のDeadプロパティ構造体へのリンク(ポインタ)になっている。
	 * なお、next == NULLならば、終端であるとする
	 */
	divy_rdbo_dproperty *next;
};

/**
 * adminmode の値
 */
enum __divy_rdbo_adminmode {
	DIVY_ADMINMODE_NORMAL = 0,	/* 通常ユーザ, 制限ユーザ, グループリーダ */
	DIVY_ADMINMODE_ADMIN  = 1	/* 管理者ユーザ */
};
typedef enum __divy_rdbo_adminmode divy_rdbo_adminmode;

/**
 * ユーザを表す構造体 (divy_usrテーブルマッピング)
 */
struct __divy_rdbo_usr {

	char *usrid;        /* ユーザID                   */
	char *password;     /* パスワード                 */
	char *fullname;     /* フルネーム                 */
	char *mailaddr;     /* メールアドレス             */
	divy_rdbo_adminmode adminmode;	/* 管理者モード   */
	char *rsid;         /* ユーザコレクションID       */
	apr_int32_t  usrseq;   /* ユーザID SEQ            */
	apr_int64_t usedst; /* ディスク使用容量(byte)     */
	apr_int64_t maxst;  /* 最大使用可能容量(byte)     */
	apr_int64_t usedres;/* 保持しているファイル数     */
	apr_int64_t maxres; /* 最大保持可能なファイル数   */
	time_t lastaccess;  /* 最終アクセス時刻           */
	char *lastaccesscl; /* 最終アクセスクライアント   */
	char *registdt;     /* 登録日付を文字列で表した値 */
	char *updatedt;     /* 更新日付を文字列で表した値 */
	char *comment;      /* ユーザに関する短い説明     */

	char *prvcol_uri;   /* プライベートコレクションURI*/

	/* 
	 * アクセスできるフォルダかどうかを示すフラグを保持する
	 * (divy_usraccessdenyのエントリ)
	 * (note)
	 * 	このint型のポインタは、サイズDIVY_FOLDER_ID_END のint型の配列へのポインタを
	 * 	保持します。この配列の要素は列挙型divy_special_folderid_idの値であり、
	 * 	配列の値には、0または1が格納されています。
	 * 	    0: アクセスを許可する / 1: アクセスを拒否する
	 * (note)
	 * 	このポインタがNULLの時には、divy_usraccessdenyには該当するエントリが存在
	 * 	しなかったか、全てのフラグが0であったことを示します。
	 * 	何れにしても、有効なエントリがなかったと判断して下さい。
	 * 	(０値であることを気にする必要はありません)
	 */
	apr_int32_t *accessdeny;

	time_t expiration;	/* ユーザの有効期限 */

	/* ユーザ拡張ステータス */
	divy_rdbo_extstatus *extstatus;

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* ユーザのパスワード状態 */
	divy_rdbo_passpolicystatus *passpolicy_status;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	divy_rdbo_grp *grp_pr;

	/* オーナのユーザID */
	char *ownerid;

	/* オーナの名前 (divy_usr には記録されていません) */
	char *ownername;

	/* Otherユーザかどうか (1: Otherユーザである / 0: Otherユーザではない) */
	apr_int32_t is_otheruser;

	/* グループリーダに任命されているかどうか (groupinformationsearch のavailableuser 専用 ) */
	apr_int32_t is_appointed_groupleader;

	/* 管理できるユーザ数 (for グループリーダ) */
	apr_int32_t maxusercreation;

	/* 利用できるホスト */
	char* allowhosts;

	/* ログイン失敗回数 */
	apr_int32_t loginfailedcount;

	/* このユーザのメール監視（groupinformationsearchのavailableuser 専用) */
	divy_rdbo_mailwatch *mwatch_pr;

	divy_rdbo_usr *next;	/* 次のdivy_rdbo_usrへのポインタ */
};

/**
 * グループを表す構造体 (divy_grpテーブルマッピング)
 */
struct __divy_rdbo_grp {

	char *grpid;		/* グループID                 */
	char *name;		/* グループ名称               */
	char *pgrpid;		/* 親グループのグループID     */
	char *relativeuri;	/* グループ相対URI            */
	int depth;		/* グループDepth              */
	char *rsid;		/* グループコレクションID     */
	char *comment;		/* グループに関する短い説明   */
	char *registdt;		/* 登録日付を文字列で表した値 */
	char *updatedt;		/* 更新日付を文字列で表した値 */
	char *grpcol_uri;	/* グループコレクションのURI  */

	divy_rdbo_mailwatch *mailwatch_pr;	/* groupmailwatch */

	divy_rdbo_usr *usr_pr;
	divy_rdbo_sql *sql_pr;

	/* グループ拡張ステータス */
	divy_rdbo_extstatus *grp_extstatus;

	/* グループオーナのユーザID */
	char *ownerid;

	/* グループオーナの名前 (divy_grp には記録されていません) */
	char *ownername;

	/* groupinformationsearch のtreelist でのみ使用するChildグループのカウント数 */
	int childcount;

	/* アップロードポリシー */
	divy_rdbo_upload_policy *policy;

#ifdef DIVY_SUPPORT_GROUPQUOTA
	/* グループクォータ */
	divy_rdbo_grpquota *grp_q;
#endif /* DIVY_SUPPORT_GROUPQUOTA */

	divy_rdbo_grp *next;	/* 次のdivy_rdbo_grpへのポインタ */
};

/**
 * SQL がアクティブかどうかを表す値
 */
enum __divy_rdbo_sqlactive {
	
	DIVY_SQL_INACTIVE = 0,	/* このSQL はアクティブではない */
	DIVY_SQL_ACTIVE		/* このSQL はアクティブである   */
};

typedef enum __divy_rdbo_sqlactive divy_rdbo_sqlactive;

/**
 * SQL の種類を表す値
 */
enum __divy_rdbo_sqltype {
	
	DIVY_SQL_TYPE_NORMAL = 0,	/* 通常 */
	DIVY_SQL_TYPE_REPOSITORY,	/* リポジトリ検索 */
	DIVY_SQL_TYPE_REQUIRED,		/* RequiredSQL */
	DIVY_SQL_TYPE_BIND,		/* 名前付きバインド変数 */
	DIVY_SQL_TYPE_UNKNOWN		/* 上記のどれでもなかった(エラー判定用) */
};

typedef enum __divy_rdbo_sqltype divy_rdbo_sqltype;

/**
 * SQL動作モードを表す値
 */
enum __divy_rdbo_sqlmode {

	DIVY_SQL_MODE_SHOW = 0,		/* 可視   */
	DIVY_SQL_MODE_HIDDEN,		/* 不可視 */
	DIVY_SQL_MODE_SELECTED,		/* １件選択可能 */
	DIVY_SQL_MODE_MULTISELECTED,	/* N 件選択可能 */
	DIVY_SQL_MODE_UNKNOWN		/* 上記のどれでもなかった(エラー判定用) */
};

typedef enum __divy_rdbo_sqlmode divy_rdbo_sqlmode;

/**
 * SQLが継承されていたかどうかを表す値
 */
enum __divy_rdbo_sqlinherit {
	DIVY_SQL_INHERIT_FALSE = 0,	/* 継承されていない */
	DIVY_SQL_INHERIT_TRUE		/* 継承されていた */
};

typedef enum __divy_rdbo_sqlinherit divy_rdbo_sqlinherit;

/**
 * SQLが持っている値をクライアント側がキャッシュすべきかどうかを表す値
 * (note)
 * 	これはデフォルト値を持っていない名前付きバインド変数($$Bxxx) に
 * 	対してのみ意味を持ちます。
 * 	その他では常に"キャッシュしない"(DIVY_SQL_NOCACHE) となります。
 */
enum __divy_rdbo_sqlcachemode {
	DIVY_SQL_NOCACHE = 0,	/* キャッシュしてはならない */
	DIVY_SQL_CACHE		/* キャッシュすべき         */
};

typedef enum __divy_rdbo_sqlcachemode divy_rdbo_sqlcachemode;

/**
 * SQLを表す構造体 (divy_sqlテーブルマッピング)
 */
struct __divy_rdbo_sql {
	
	char *sqlid;
	char *labelname;		/* SQL を表す短い文字列 */
	char *sql;			/* SQL 文 */
	divy_rdbo_sqlactive active;	/* SQL アクティブフラグ */
	char *relativeuri;		/* SQL 相対URI */	
	char *dbid;			/* このSQLが実行されるDBMSのID */
	char *comment;			/* SQL 文の短い説明文 */
	
	char *registdt;			/* 登録日付を文字列で表した値 */
	char *updatedt;			/* 更新日付を文字列で表した値 */

	char *usedbmsname;		/* このSQLが実行されるDBMSの名前 */

	char *subname;			/* SQL サブ名称 */
	divy_rdbo_sqltype type;		/* SQL 種類 */
	divy_rdbo_sqlcachemode cachemode;	/* SQL キャッシュモード */

	/* このSQLのWhere句に関する情報をもつ構造体へのポインタ */
	divy_rdbo_sqlcontent *sqlcnt_pr;

	/* このSQL のrequiredsql の依存関係をもつ構造体へのポインタ */
	divy_rdbo_sqldepend *sqld_pr;

	/* 利用できる DBMS のリスト構造体へのポインタ */
	divy_rdbo_dbms *dbms_pr;

	/* この SQL を参照している SQL のラベル名称のハッシュ */
	apr_hash_t *usingsql_hash;

	/* このSQLが親から継承されてきたかどうか表す値 */
	divy_rdbo_sqlinherit inheritsql;

	divy_rdbo_grp *grp_pr;

	divy_rdbo_sql *next;	/* 次のdivy_rdbo_sqlへのポインタ */
};

/**
 * sqlcontent のタイプを表す値
 */
enum __divy_rdbo_contype {

	DIVY_SQL_CONTYPE_BIND = 0,	/* 通常バインド変数 */
	DIVY_SQL_CONTYPE_REQUIRED	/* required、名前付きバインド変数 */
};

typedef enum __divy_rdbo_contype divy_rdbo_contype;

/**
 * あるSQL のWHERE 句に関する情報をもつ構造体
 */
struct __divy_rdbo_sqlcontent {

	apr_int32_t id;	/* SQL 条件句ラベルID     */
	char *name;	/* SQL 条件句ラベル名称   */
	char *fmt;	/* SQL 条件句フォーマット */
	apr_int32_t sqlposition; /* SQL文中の出現位置 */
	
	divy_rdbo_sqlcontent *next;	/* 次の divy_rdbo_sqlcontent へのポインタ */

	char *colname;			/* SQL 条件句カラム名 */
	divy_rdbo_contype contype;	/* content のタイプ */
	divy_rdbo_reqsql *reqsql;	/* required sql に関する情報 */
};

/**
 * あるrequiredsql に関する情報をもつ構造体
 */
struct __divy_rdbo_reqsql {
	
	divy_rdbo_sqlmode sqlmode;		/* SQL動作モード       */
	divy_rdbo_rsvalue *rsvalue;		/* rsvalueエレメントの値を持つ構造体 */

	divy_rdbo_sqlmode *sqlmodelist;		/* SQL動作モードのリスト */
};

/**
 * SQL間の依存関係(divy_sqldepend)を表す構造体(divy_sqldependテーブルマッピング)
 */
struct __divy_rdbo_sqldepend {

	/* トップ階層SQLシーケンス番号
	 * ptsubname とclsubname の関係を利用するSQLのシーケンス番号
	 */
	char *topid;

	/* この関係の親にあたるSQL のsubname
	 * 通常SQLとリポジトリDB検索SQLの場合には、SQLシーケンス番号が入る
	 */
	char *ptsubname;

	/* この関係の子にあたるSQL のsubname */
	char *clsubname;

	divy_rdbo_sqldepend *next;	/* 次の divy_rdbo_depend へのポインタ */
};

/**
 * 共有コレクションの集合を表す構造体。
 */
struct __divy_rdbo_shlink {
	char *uri;		/* 共有コレクションのURI        */
	char *displayname;	/* 共有コレクションの表示名称   */
	char *rsid;		/* 共有コレクションのリソースID */

	divy_rdbo_shlink *next; /* 次の divy_rdbo_shlink へのポインタ */
};

/**
 * 共有コレクションのソースコレクションを表す構造体。
 */
struct __divy_rdbo_shsrccol {
	char *name;	/* ソースコレクションの表示名称   */
	char *uri;	/* ソースコレクションのURI        */
	char *rsid;	/* ソースコレクションのリソースID */

	/* 顧客DB検索結果のキー情報を保持する構造体へのポインタ */
	divy_rdbo_keylist *keylist_pr;

	divy_rdbo_shsrccol *next;
};

/**
 * DBMS がアクティブかどうかをあらわす値
 */
enum __divy_rdbo_dbmsactive {
	DIVY_DBMS_INACTIVE = 0,	/* このDBMSエントリはアクティブではない */
	DIVY_DBMS_ACTIVE	/* このDBMSエントリはアクティブである   */
};
typedef enum __divy_rdbo_dbmsactive divy_rdbo_dbmsactive;

/*
 * DBMS 接続情報を持つ構造体 (divy_dbmsテーブルマッピング)
 */
struct __divy_rdbo_dbms {
	char *dbmsid;		/* DBMS 識別番号             */
	char *type;		/* DBMS 種別                 */
	char *name;		/* DBMS 接続別識別名         */
	char *hostname;		/* DBMS 動作ホスト名         */
	apr_int32_t port;	/* DBMS 動作ポート番号       */
	char *dbname;		/* DBMS データベース名       */
	char *username;		/* DBMS 接続ユーザ名         */
	char *password;		/* DBMS 接続ユーザパスワード */
	char *comment;		/* コメント                  */
	apr_int32_t dbpool;	/* DBMS 接続キャッシュ利用フラグ */
	apr_int32_t dbminspareconn;	/* DBMS 最小保持キャッシュ数 */
	apr_int32_t dbmaxspareconn;	/* DBMS 最大保持キャッシュ数 */
	divy_rdbo_dbmsactive active;	/* DBMS アクティブフラグ */

	char *registdt;		/* 登録日付を文字列で表した値 */
	char *updatedt;		/* 更新日付を文字列で表した値 */

	divy_rdbo_dbms *next;	/* 次のdivy_rdbo_dbms へのポインタ */
};

/*
 * システムメッセージの情報をもつ構造体 (divy_sysmsgテーブルマッピング)
 */
struct __divy_rdbo_sysmsg {
	char *msgid;		/* メッセージID (DB はinteger) */
	char *msg;			/* メッセージテキスト */
	char *subject;      /* 題名 */
	char *lang;			/* 言語 jaとかenとか*/
	char *scope;		/* メッセージ範囲 */
	apr_int64_t opendt;	/* 開始日 */
	apr_int64_t closedt;/* 停止日 */
	apr_int32_t active;	/* アクティブ状態     */

	char *registdt;		/* 登録日付を文字列で表した値 */
	char *updatedt;		/* 更新日付を文字列で表した値 */

	divy_rdbo_sysmsg *next;	/* 次のdivy_rdbo_sysmsg へのポインタ */
};

/**
 * ごみ箱の情報を持つ構造体
 */
struct __divy_rdbo_trash {
	char *rsid;
	char *deleter_id;	/* ユーザID */
	char *daletion_date;	/* ISO8601形式 */

	char *deleter_name;	/* ユーザ名 */
};

/*
 * クライアントモジュールの情報をもつ構造体 (divy_clmoduleテーブルマッピング)
 */
struct __divy_rdbo_clmodule {
	char *rsid;		/* モジュールのリソースID     */
	char *name;		/* クライアントモジュール名   */
	char *version;		/* クライアントバージョン     */
	char *lineup;		/* クライアント種別(lineup)   */
	char *digest;		/* クライアントダイジェスト値 */
	char *uri;		/* リソースURI */

	char *registdt;		/* 登録日付を文字列で表した値 */
	char *updatedt;		/* 更新日付を文字列で表した値 */

	/* 
	 * updateinformationsearch の返却値を作成する時に使用します。
	 */
	apr_int64_t getcontentlength;  /* Content-Length */

	divy_rdbo_clmodule *next;
};

/*
 * 通知の種類を表す列挙型
 */
enum __divy_rdbo_notification {
	DIVY_NOTICE_MAIL = 0,	/* メール通知 */
	DIVY_NOTICE_MSG,	/* メッセージ通知 */
	DIVY_NOTICE_BOTH,	/* メールとメッセージ通知 */
	DIVY_NOTICE_UNKNOWN,	/* 不明な種類 */
};
typedef enum __divy_rdbo_notification	divy_rdbo_notification;

/**
 * メール監視状態を保持する構造体 (divy_mailwatchテーブルマッピング)
 */
struct __divy_rdbo_mailwatch {
	char *rsid;		/* 監視対象リソースID  */
	char *uri;		/* 監視対象リソースURI */
	char *usrid;		/* 監視ユーザID        */
	/*
	 * トリガメソッドからなる配列
	 * index 番号に httpd.h で定義されたM_GETやM_PUTを指定すると
	 * 該当するメソッドのトリガ状態(0/1)が取得できるような配列
	 * (note)
	 * 使う前にはmemsetなどを使って初期化して下さい。
	 */
	apr_int32_t trigger_methods[METHODS];
	apr_int32_t delflag;	/* 論理削除フラグ */

	/*
	 * メール監視オーナーが持つトリガメソッド
	 * (note)
	 * 	オーナーが持っていなければNULLとなります。
	 * 	このメンバはmailwatch プロパティ出力の時だけ
	 * 	使用されます。書き込み時には、trigger_methods を
	 * 	引き続き利用して下さい。
	 */
	apr_int32_t *owner_trigger_methos;

	/* 通知の種類 */
	divy_rdbo_notification notification;

	divy_rdbo_mailwatch *next;
};

/**
 * keylist プロパティを表す構造体
 */
struct __divy_rdbo_keylist {
	int index;		/* ID attribute */
	char *key;		/* キーの値     */
	divy_rdbo_keylist *next;
};

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * Quota種別
 * (note)
 * 	ここで定義された値は、テーブルdivy_diskquota のQuota種別が持つ値と
 * 	一致していなければなりません。
 */
enum __divy_rdbo_qttype {
	DIVY_QTTYPE_SYSTEM = 0,	/* システムQuota */
};
typedef enum __divy_rdbo_qttype	divy_rdbo_qttype;
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * Quota情報を持つ構造体 (システムQuota)
 */
struct __divy_rdbo_diskquota {
	char *uri;		/* URI       */
	divy_rdbo_qttype type;	/* Quota種別 */
	apr_int64_t usedst;	/* ディスク使用容量(byte)     */
	apr_int64_t maxst;	/* 最大使用可能容量(byte)     */
	apr_int64_t usedres;	/* 保持しているファイル数     */
	apr_int64_t maxres;	/* 最大保持可能なファイル数   */    

	divy_statfs_t *sfs;	/* ファイルシステムの情報     */
};
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

enum __divy_upload_policy_chartype {
	TF_UPLOAD_POLICY_CHARTYPE_NONE = 0x00,		/* 設定なし		*/
	TF_UPLOAD_POLICY_CHARTYPE_NUM  = 0x01,		/* 数字			*/
	TF_UPLOAD_POLICY_CHARTYPE_ALP  = 0x02		/* 英字			*/
};

typedef enum __divy_upload_policy_chartype divy_upload_policy_chartype;

enum __divy_upload_policy_matchtype {
	TF_UPLOAD_POLICY_MATCHTYPE_ALL = 0x00,		/* すべて 		*/
	TF_UPLOAD_POLICY_MATCHTYPE_ANY = 0x01 		/* どれか		*/
};

typedef enum __divy_upload_policy_matchtype divy_upload_policy_matchtype;

struct __divy_rdbo_upload_policy {
	int active;				/* フラグ 1: アクティブ / 0:非アクティブ	*/
	time_t updatedate;		/* 更新日 									*/
	char *rules_fnames;		/* ファイル名ルール(スラッシュ区切り)		*/
	char *rules_suffix;		/* 拡張子ルール 							*/
	divy_upload_policy_chartype rules_chartype;	/* ファイル文字種ルール */
	char *rules_length;		/* 長さルール（カンマ区切り）				*/
	divy_upload_policy_matchtype rules_matchtype;	/* 一致区分				*/
	int hidden;				/* 0:公開 / 1:非公開						*/
	int isChanged;			/* 1:アクティブ<=>非アクティブ / 0:変化なし */
};

#ifdef DIVY_SUPPORT_GROUPQUOTA
/*
 * Quota情報をもつ構造体（グループQuota)
 */
struct __divy_rdbo_grpquota {
	char *grpid;             /* グループID                 */
	apr_int64_t usedst;      /* ディスク使用容量(byte)     */
	apr_int64_t maxst;       /* 最大使用可能容量(byte)     */
	apr_int64_t usedres;     /* 保持しているファイル数     */
	apr_int64_t maxres;      /* 最大保持可能なファイル数   */
};
#endif /* DIVY_SUPPORT_GROUPQUOTA */

/**
 * 状態、属性を表す構造体
 */
struct __divy_rdbo_resourcestate {
	char *uri;	/* 状態ルートURI */
	int type;	/* 状態の種類 */
	int depth;	/* 状態ルートURIの階層数 */
	int hierarchy;	/* 状態が下位階層に伝播するかどうか (1: 伝播する) */
	int effective;	/* 状態が有効かどうか (1: 有効) */

	char *rsid;	/* リソースID (内部処理専用) */
	int inherit;	/* 親から継承している状態かどうか (内部処理専用, 付かない場合あり) */

	/* 状態の種類ごとのオプション項目 */
	union {
		divy_rdbo_box *box_pr;
	} optional;

	divy_rdbo_resourcestate *next;
};

/*
 * divy_rdbo_resourcestate.type の種類
 */
enum {
	DIVY_RSTATE_TYPE_UNKNOWN = 0,	/* 未定義の状態、属性 */
	DIVY_RSTATE_TYPE_VIEW    = 1,	/* View属性(状態) */
	DIVY_RSTATE_TYPE_BOX     = 2	/* BOX属性            */
};

/**
 * １つのユーザリレーションを表す構造体
 */
struct __divy_rdbo_rusr {
	char *usrid;	/* リレーションのユーザID        */
	char *grp_uri;	/* usrid を持つグループの相対URI */ 
	char *grpid;	/* grp_uri のグループID          */
};

/**
 * １つのSQLリレーションを表す構造体
 */
struct __divy_rdbo_rsql {
	char *labelname;	/* リレーションのSQL名称             */
	char *grp_uri;		/* labelname を持つグループの相対URI */ 
	char *sqlid;		/* labelname のSQLシーケンス番号     */
	char *grpid;		/* grp_uri のグループID              */

	/* このSQLが親から継承されてきたかどうか表す値 */
	divy_rdbo_sqlinherit inheritsql;
	/* sqlid のSQLを持っているグループのグループID */
	char *owner_grpid;
};

/**
 * １つのグループリレーションを表す構造体
 */
struct __divy_rdbo_rgrp {
	char *grpid;	/* リレーションのグループID */

#define DIVY_RGRP_TO_USER 0	/* ユーザに対するリレーション */
#define DIVY_RGRP_TO_SQL  1	/* SQL に対するリレーション   */
	int kind_of_relation;

	/* grpid を持つユーザのユーザID (DIVY_RGRP_TO_USER 指定時) */
	char *usrid;

	/* grpid を持つSQLのSQL名称 (DIVY_RGRP_TO_SQL指定時) */
	char *labelname;
	char *sqlid;	/* lablename のSQLシーケンス番号 */

	/* このSQLが親から継承されてきたかどうか表す値 */
	divy_rdbo_sqlinherit inheritsql;
	/* sqlid のSQLを持っているグループのグループID */
	char *owner_grpid;
};

/**
 * ロックテーブルと同じ内容を保持する構造体 (dav_lockテーブルマッピング)
 */
struct __divy_rdbo_lock {
	char *uri;		/* uri					   */
	dav_locktoken *locktoken;/* LockToken 				   */
	dav_lock_type  kbn;	/* ロックタイプ 			   */
	dav_lock_scope  scope;	/* ロックスコープ [ exclusiv or shared ..] */
	int depth;		/* rootからの自分のファイルまでの階層 	   */
	time_t timeout;		/* ロックタイムアウト			   */
	char *owner;		/* ロックオーナー			   */
	char *key;		/* ロックを行った基準キー(uri)		   */
	char *userid;		/* ロックを行ったユーザID(r->userと同じ)   */
	apr_int32_t is_null;	/* 対象のリソースがNULLかどうか？ 1:NULL   */
	apr_int32_t lockdepth;	/* ロックの階層				   */
	divy_rdbo_lock *next;	/* 次の構造体へのポインタ		   */
};

/**
 * ロックリソースの処理状態を表す値
 */
enum __dav_lock_trans_status {
	TF_LOCK_TRANS_NOREADY = 0x01,	/* ロックはなにも処理をしていない */
	TF_LOCK_TRANS_ABORT   = 0x02	/* 問題が発生した(処理継続不可能) */
};

typedef enum __dav_lock_trans_status dav_lock_trans_status;

/**
 * lock全般で利用するロック情報が詰まれている構造体
 */
struct __divy_rdbo_lock_resource {
	/* ロックトークンを保持しているハッシュテーブル */
	apr_hash_t	*token_list_hash;

	/* ロックリソースの状態を表す */
	dav_lock_trans_status	status;
};

/**
 * search でresponce 以外の情報を保持する構造体
 */
struct __divy_rdbo_search {
	
	char **linkdb_key_values;	/* linkdb のキーを持つ */

	int reposdb_selected[3];	/* linkdb のリポジトリ検索で必須以外のselect されたプロパティ */

	divy_rdbo_sqltype sqltype;	/* SQL 文のタイプ */

	divy_rdbo_clist *namelist;	/* errdiscoveryのnamelistを格納 */

	char *dbmserrcode;		/* errdiscoveryのDBMS エラーコード */
	char *dbmserrmsg;		/* errdiscoveryのDBMS エラーメッセージ */

	/*
	 * この SQL に関連する requiredsql のサブ名称のハッシュへのポインタ
	 * キーのみ使用
	 */
	apr_hash_t *reqsqlname_hash;
};

/**
 * DB検索結果を保持するリスト構造体のコンテナ
 */
struct  __divy_rdbo_search_output {
	int optflg;	/* SEARCHの種類 */

	/* DB検索結果を持つリスト */
	union {
		divy_rdbo_resource *rdb_r;
		divy_rdbo_usr *usr_pr;
		divy_rdbo_grp *grp_pr;
		divy_rdbo_dbms *dbms_pr;
		divy_rdbo_clmodule *clmodule_pr;
		divy_rdbo_sysmsg *sysmsg_pr;
	} list;

	/* 出力したレスポンス数を保持する変数(作業用) */
	int response_cnt;
};

/**
 * DB 検索用関数に渡されるパラメータのコンテナ構造体。
 */
struct __divy_rdbo_search_params {

	request_rec *r;
	apr_pool_t *wp;		/* リクエスト一時プール */
	apr_pool_t *scratchpool;/* 1レスポンス出力ごとにクリアされるプール */
	int optflg;		/* SEARCH の種類を表すフラグ */
	const char *sqlstr;	/* SQL文 */
	int is_streaming_mode;	/* ストリーミングモードかどうか(1: Yes / 0: No) */
	
	/* walker パラメータ (for mod_dav) */
	const dav_walk_params *dwparam;

	/* このパラメ-タが渡されるSEARCHを処理するハンドラ構造体へのポインタ */
	const divy_search_process *process;

	void *ctx;	/* 何か特別に渡したいものがあればお好きにどうぞ */

	/**
	 * 1レスポンスをストリーム出力する関数へのポインタ。
	 * NULLチェックは必須
	 *
	 * @param param divy_rdbo_search_params * 入力パラメータ
	 * @param output divy_rdbo_search_output * 出力結果
	 * @return int 処理ステータス(0: 成功 / 1: 失敗)
	 */
	int (*output_response)(divy_rdbo_search_params *params, divy_rdbo_search_output *output);
};

/**
 * reposdbsearch でオプション項目が選択されたかどうかの構造体へのアドレス参照
 */
enum __divy_rdbo_reposdb_optsel {
	
	DIVY_REPOSDBOPT_CREATIONDATE = 0,	/* creationdate */
	DIVY_REPOSDBOPT_CREATOR,	 	/* creator */
	DIVY_REPOSDBOPT_LASTMODIFIER	 	/* lastmodifier */
};

/**
 * WHERE句に値を提供するRequiredSQLまたは名前付きバインド変数の変数名
 * ($$SSxxxx, $$SMxxxx, $$Bxxxx) と、それらが生成する値を表す構造体。
 * (note)
 * 	この構造体は、エレメントrsvalue のオブジェクト表現です。
 * 	rsvalueエレメントやreqsqlnameが使用されるケースにおいて
 * 	使用して下さい。
 */
struct __divy_rdbo_rsvalue {
	int id;			/* 条件句ラベルID */
	char *name;		/* 名前(reqsqlnameと対応) */

	/* 表示値(dispvalue)と値(reqsqlvalue)を表す(reqsqlvaluesetと対応) */
	divy_rdbo_rsvalueset *valueset;

	divy_rdbo_rsvalue *next;
};

/**
 * WHERE句に値を提供するRequiredSQLから得られた値を表す構造体。
 * (note)
 * 	この構造体は、エレメントreqsqlvaluesetまたはreqsqlvalue の
 * 	オブジェクト表現です。
 * 	これらのエレメントが使用されるケースにおいて使用して下さい。
 */
struct __divy_rdbo_rsvalueset {
	char *value;		/* reqsqlvalue に対応する値 */
	char *dispvalue;	/* dispvalue に対応する値 */
	
	divy_rdbo_rsvalueset *next;
};

/**
 * 文字列のリストを表す構造体 (ユーティリティ構造体)
 * (note) 特定のテーブルマッピングには使用しません。
 * 	単なる文字列リストを実現するために用意しました。
 */
struct __divy_rdbo_clist {
	char *val;	/* このリストの値 */

	divy_rdbo_clist *next;	/* 次のリストへのポインタ */
};

/**
 * 並び替えに使う構造体
 */
struct __divy_rdbo_sortorder {
	enum {
		DIVY_SORTKEY_URI = 0,
		DIVY_SORTKEY_NAME,
		DIVY_SORTKEY_SIZE,
		DIVY_SORTKEY_TYPE,
		DIVY_SORTKEY_DATE,
		DIVY_SORTKEY_MODIFIER,
		DIVY_SORTKEY_RESTYPE
	} key;

	enum {
		DIVY_SORTORDER_ASC = 0,
		DIVY_SORTORDER_DESC
	} order;
};


#ifdef DIVY_SUPPORT_PASSPOLICY
struct __divy_rdbo_passwordpolicy {
	/* デフォルトのポリシーID */
	#define DIVY_DEFAULT_POLICY_ID	1

	int policyid;	/* ポリシーID */

	/* ステータス(1: 有効 / 0: 無効)  */
	int status;

	/* ポリシーを有効にした日付のエポックタイム
	 * (0: 有効にしていない / n: エポックタイム) */
	time_t startdt;

	/* 最小パスワード長さ(0: 制限なし / n: n文字制限) */
	int minlen;

	/* パスワード更新周期(有効期限)(日). (0: 無期限 / n: n日) */
	int change_pw_cycle;

	/* パスワード期限切れ後の猶予期間(日) (0: 即日 / n: n日)
	 * (note)
	 *   change_pw_cycle が 0 以外で初めて有効となります */
	int probation;

	/* 禁則文字一覧 (NULL: 禁則なし / NOT NULL: その値(をchar 毎に分割したもの) */
	divy_rdbo_passngword *ngword;

	/* パスワード履歴保持件数. (0: 0件 / n: n件) */
	int history_pw_num;

	/* ユーザID文字列を含めてはならない (1: 含めてはならない / 0: 任意) */
	int denyUseridString;

	/* 記号が必須かどうか (0: 任意 / 1: 必須) */
	int isSymbolChrNeeded;

	/* アルファベット大文字が必須かどうか (0: 任意 / 1: 必須) */
	int isUpAlphabet;

	/* アルファベット小文字が必須かどうか (0: 任意 / 1: 必須) */
	int isLoAlphabet;

	/* 数値が必須かどうか (0: 任意 / 1: 必須) */
	int isNumericNeeded;

	/* 連続文字を許可するかどうか (0: 任意 / 1: 必須) */
	int denyCycleChr;

	/* 初回ログインパスワード変更日付のリミット日
	 * (-1: リミットなし / 0: 変えない限りログオン不可 / n: n日)  */
	int firstlogin_limitday;
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
struct __divy_rdbo_passngword {
	int policyid;	/* ポリシーID */
	divy_cset_t *ngword_set;	/* 禁則文字一覧 */
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
struct __divy_rdbo_passhistory {
	char *usrid;		/* 対象ユーザID */

	/* 最大履歴数 */
#define DIVY_MAX_PASSHISTORY_NUM	12
	/* パスワード(char*) を持つ配列 (要素番号が履歴番号と同じ) */
	char *passhistory[DIVY_MAX_PASSHISTORY_NUM + 1];	/* 1 は予備 */
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
struct __divy_rdbo_passpolicystatus {
	/* ポリシーID */
	int policyid;

	/* 対象ユーザID */
	char *usrid;

	/*
	 * 最後にパスワードを更新した日付
	 * (0: 1度も更新していない / n: 日付)
	 */
	time_t lastchange;

	/*
	 * 最終期限切れメール送信日付
	 * (0: 未送信 or パスワード変更直後 / n: 送信した日付)
	 */
	time_t sendexpiredmail;	

	/*
	 * 初回ログインした日付
	 * (0: 初回ログインしていない / n: 初回ログインした日付)
	 */
	time_t firstlogin;

	/*
	 * 完全期限切れを迎えてからログインしたユーザの
	 * パスワード期限切れ判断の基点となる日付.
	 * 対象ユーザの猶予日時や完全期限切れ日時はこの値をベースに算出します.
	 * 長期出張者対応
	 */
	time_t specialstart;
};
#endif	/* DIVY_SUPPORT_PASSPOLICY */

struct __divy_rdbo_confirmreading {
	char *uri;				/* リソースのURI */
	char *userid;			/* ユーザID */
	time_t creationdate;	/* 開封通知を付けた日付 */

	/* 開封通知が1つのリソースに複数付いていた場合にはnext メンバに繋がっている
	 * (uri は同じで、userid と creationdate が異なるだろう) */
	divy_rdbo_confirmreading *next;
};

struct __divy_rdbo_session {
	char *sid;				/* セッションID */
	char *userid;			/* ユーザID     */
	time_t logindate;		/* ログイン時間 */
	char *password;			/* パスワード   */
	char *mailaddr;			/* 宛先メール	*/
	char *authcode;			/* 2FA文字列	*/
	char *uniqueid;			/* ユニークID	*/

	divy_rdbo_session *next; /* 次 */
};

struct __divy_rdbo_box {
	char *uri;					/* URI */
	char *allowed_origin;		/* 許可されたオリジン */
	char *shorten;				/* URIの短縮 */

	/* BOXフラグ */
#define BOX_FLAG_NONE	0x00	/* 未公開 */
#define BOX_FLAG_OPEN	0x01	/* 公開中 */
#define BOX_FLAG_PRIV	0x02	/* プライバシーモード */
#define BOX_FLAG_LOCK	0x04	/* リンクロック */
#define BOX_FLAG_DELE	0x08	/* 自動削除 */
#define BOX_FLAG_NEW    0x10	/* 新規   */
	int   flag;					/* フラグ */

	char *password;				/* パスワード */
	time_t creationdate;		/* BOX作成日 */
	time_t expirationdate;		/* 有効期限 */
	const char *creator_usr_id;	/* BOX作成ユーザID */
	const char *creator;		/* BOX作成ユーザ名 */
	char *greeting;				/* 公開あいさつ文 */
	char *message;				/* 説明 */
	apr_int64_t viewcount;		/* 閲覧回数 */
	char *tomailaddr;			/* 送り先メールアドレス */
};

/*------------------------------------------------------------------------------
  Declar prototype public functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたリソースまたはコレクション(rdb_r)のリソースタイプを
 * DBから取得して返却する。
 *
 * select 対象テーブル: dav_resource
 * (note) rdb_r 必須項目: uri
 *
 * @param r request_rec * request_rec 構造体へのポインタ
 * @param rdb_r divy_rdbo_resource * 検索対象のリソースまたはコレクション
 * @return divy_rdbo_resourcetype リソースタイプ
 * 	DIVY_TYPE_RESOURCE / DIVY_TYPE_COLLECTION / DIVY_TYPE_UNKNOWN)
 */
DIVY_DECLARE(divy_rdbo_resourcetype) divy_rdbo_get_resourcetype(request_rec *r,
                                                  divy_rdbo_resource *rdb_r);

/**
 * 指定されたdepth (Depth ヘッダの値)で、divy_rdbo_resource 以下に存在する全ての
 * リソースまたはコレクションのプロパティを取得する。
 *
 * select 対象テーブル:
 * 	dav_resource, dav_dead_property,
 * 	dav_link_collection, divy_mailwatch
 *
 * (note) rdb_r 必須項目
 * 	rdb_r->uri, rdb_r->depth (depth = 1 の時), rdb_r->resourcetype
 *
 * @param r request_rec * request_rec構造体へのポインタ
 * @param rdb_r divy_rdbo_resource * 検索対象&検索結果
 * @param depth int Depth ヘッダの値 (0 / 1 / DAV_INFINITY)
 * @param propflag apr_uint32_t 特別に取得するプロパティの種類 (ORで指定する)
 * 	DIVY_GET_PROP_fullname      : 作成者,更新者名称プロパティを取得する
 * 	DIVY_GET_PROP_dead          : Dead プロパティを取得する
 * 	DIVY_GET_PROP_mailwatch     : mailwatch プロパティを取得する
 * 	DIVY_GET_PROP_shared        : 共有コレクションプロパティを取得する
 * 	DIVY_GET_PROP_trash         : ごみ箱のプロパティを取得する
 * 	DIVY_GET_PROP_notrashfolder : ごみ箱だけを除外する
 * 	DIVY_GET_PROP_resourcestate : 状態, 属性プロパティを取得する
 * 	DIVY_DEL_PROP_noviewattr    :  View属性が"付いていない"リソースを削り落とす(基点は除外)
 * @param sort divy_rdbo_sortorder * ソートキー(NULLでデフォルト)
 * @return 処理ステータス 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_rdbo_get_hierarchy_property(request_rec *r,
						divy_rdbo_resource *rdb_r,
						int depth,
						apr_uint32_t propflag, divy_rdbo_sortorder *sort);

/**
 * 指定されたSQL文、エスケープ、バインド情報に従ってSQLを組み立て
 * リソース情報をリポジトリDBから検索し検索結果を返却する(basicsearch)
 *
 * 
 * @param request_rec * apache request_rec構造体
 * @param divy_rdbo_resource * 検索結果格納
 * @param const char * SQL文
 * @param int エスケープするかどうかのフラグ
 * @param apr_hash_t バインド情報のハッシュ
 * @return int 0:成功、1:失敗
 */
DIVY_DECLARE(int) divy_rdbo_basicsearch(request_rec *r,
						divy_rdbo_resource *rdb_r,
                                                const char *sqlstr,
						int escflg,
						apr_hash_t *bindinfo_h);

/**
 * 更新クライアントの表示情報を検索・取得する。
 * (note)
 *	updateinformationsearch で使用します。
 *	このSEARCH専用になっていますので、他の用途には使用してはなりません。
 * 
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param screen divy_search_updateis_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功 / 1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_updateinformationsearch(divy_rdbo_search_params *params,
						divy_search_updateis_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * 指定された新規リソースrdb_r のプロパティを DB に書き込む。(新規)
 * rdb_r がリソースの場合、"更新者"のQuotaを消費します。
 *
 * insert 対象テーブル: dav_resource, divy_usrdiskquota, divy_diskquota
 * (note) rdb_r 必須項目
 * 	uri, displayname, depth, getcontentlanguage, getcontentlength,
 * 	getcontenttype, rsid, resourcetype, physicalpath, creationdate,
 * 	getlastmodified, getetag, creator_userid, lastmodifier_userid,
 *	isversioned, checkin, checkout 
 *
 *	以下は、コレクションの場合、NULLでもかまいません。
 *	getcontentlanguage, getcontenttype, physicalpath 
 * (note)
 * 	トランザクションがあれば継続します
 *
 * @param r request_rec * request_rec構造体へのポインタ
 * @param rdb_r const divy_rdbo_resource * DB にinsertするデータを持つ構造体
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス
 * 		0                            : 成功
 * 		1                            : 予期しないエラー
 * 		DIVY_STCODE_QUOTAOVER        : Quota オーバー
 * 		DIVY_STCODE_PARENT_NOT_EXIST : 親リソースが存在しなかった
 * 		DIVY_STCODE_USER_NOT_EXIST   : 操作者がいなかった
 */
DIVY_DECLARE(int) divy_rdbo_insert_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 既に存在していたrdb_r が示すリソースまたはコレクションのプロパティを
 * DB に書き込む。(更新)
 * rdb_r がリソースの場合、旧"更新者"または作成者に以前消費していた
 * リソースのQuotaを返し、"更新者"のQuotaを消費します。
 *
 * update 対象テーブル: dav_resource, divy_usrdiskquota, divy_diskquota
 * (note) rdb_r 必須項目
 * 	uri, rsid, creator_userid, lastmodifier_userid,
 * 	getcontentlanguage, getcontentlength, getcontenttype, getlastmodified,
 * 	getetag, isversioned, checkin, checkout
 * (note) 以下の項目は変更できません
 *	uri, rsid, displayname, depth, resourcetype, physicalpath,
 *	creationdate, creator_userid
 * (note)
 * 	トランザクションがあれば継続します
 *
 * @param r request_rec * request_rec構造体へのポインタ
 * @param rdb_r const divy_rdbo_resource * DB をupdateするデータを持つ構造体
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス
 * 		0                         : 成功
 * 		1                         : 予期しないエラー
 * 		DIVY_STCODE_QUOTAOVER     : Quota オーバー
 * 		DIVY_STCODE_USER_NOT_EXIST: 操作者がいなかった
 */
DIVY_DECLARE(int) divy_rdbo_update_property(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたロックNULLレコードを削除する。
 * 対象のURIのロックNULLレコードが削除できなくても目的達成（削除処理）
 * の為エラーを出力しません。
 *
 * @param r request_rec *
 * @param lockrec divy_rdbo_lock *
 * @return status int (0: 正常 / 1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_locknull_member(request_rec * r,
						divy_rdbo_lock *lockrec);

/**
 * 指定されたユーザID userid のユーザが持っていた全てのロック及び
 * ロックNULLリソースを削除する。
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param r request_rec *
 * @param userid const char * 対象ユーザ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 /1: 失敗)
 * 	削除すべきロックやロックNULLリソースが存在しなくても成功扱いとします。
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_lock(request_rec *r, const char *userid,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたdivy_rdbo_resource のリソースまたはコレクションをDB から削除し、
 * そのリソースまたはコレクション以下のリソースが消費していたQuotaを
 * "更新者"に返します。
 * なおこの関数は物理ファイルは削除しません。
 *
 * delete 対象テーブル：
 *   	dav_resource        : 通常リソースエントリ
 *   	dav_link_collection : 共有コレクションエントリ
 *   	dav_dead_property   : Dead プロパティ
 *   	divy_mailwatch      : メール管理フラグ(mailwatch)
 *   	divy_clmodule       : 更新クライアントプロパティ
 *   	divy_trash_info     : ごみ箱プロパティ(trashinformation)
 *
 * update 対象テーブル:
 *   	divy_usrdiskquota, divy_diskquota
 *
 * (note) rdb_r 必須項目 : uri, rsid, resourcetype
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 * 
 * @param r request_rec * request_rec構造体へのポインタ
 * @param rdb_r const divy_rdbo_resource * 削除対象のリソースまたはコレクション
 * @param del_filelist divy_linkedlist_t ** 削除対象の物理パスリスト(相対パス)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	0: 成功 / 1: 失敗 /
 *	DIVY_STCODE_NOT_EXIST: 削除元がなくなっていた
 */
DIVY_DECLARE(int) divy_rdbo_remove_resource(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					divy_linkedlist_t **del_filelist,
					divy_db_transaction_ctx *ts_ctx);

/**
 * depth = infinity として指定されたsrc_rdb_r をdist_rdb_r としてコピーする。
 * ディスティネーションの作成者は、COPY操作者になります。
 *
 * (note)
 * 	depth = DAV_INFINITY & コレクション:
 * 	-------------------------------------
 * 	    自身とそれ以下に存在する全てのリソースとコレクションも
 * 	    同時にコピーします。
 *
 * insert 対象テーブル: dav_resource, dav_dead_property, divy_usrdiskquota
 * update 対象テーブル: divy_diskquota
 * (note)
 * 	sharedcollection, mailwatch プロパティはコピーされないため、
 * 	dav_link_collection,divy_mailwatch は操作しません。
 *
 * (note) src_rdb_r 必須項目
 * 	uri, rsid, depth
 * (note) dist_rdb_r 必須項目
 * 	uri, depth, getcontentlength, creationdate, getlastmodified,
 * 	creator_userid, lastmodifier_userid, isversioned, checkin, checkout 
 * (note) 以下の項目はこの関数の中で自動的に補完します
 * 	physicalpath, getetag, rsid
 * (note) 以下の項目はcopy後もcopy 前と同じ値になります
 * 	displayname, getcontentlanguage, getcontentlength, getcontenttype,
 * 	resourcetype 
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 * 
 * @param r request_rec * request_rec構造体へのポインタ
 * @param src_rdb_r  const divy_rdbo_resource * 複製元(src)
 * @param dist_rdb_r       divy_rdbo_resource * 複製先(dist)
 * @param copyinfo divy_copyfile_info ** コピーされた物理ファイルの情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	0: 成功 / 1: 失敗 /
 *	DIVY_STCODE_NOT_EXIST     : コピー元がなくなっていた
 *	DIVY_STCODE_USER_NOT_EXIST: コピー操作者が存在しなかった
 */
DIVY_DECLARE(int) divy_rdbo_copy_resource(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dist_rdb_r,
					divy_copyfile_info **copyinfo,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたsrc_rdb_r 以下に存在する全てのリソースは無視して、自分自身
 * だけを dist_rdb_r としてコピーする。
 * (** 内部に含まれるリソースやコレクションはコピーされません。**)
 *
 * insert 対象テーブル: dav_resource, dav_dead_property, divy_usrdiskquota
 * update 対象テーブル: divy_diskquota
 * (note)
 * 	sharedcollection, mailwatch プロパティはコピーされないため、
 * 	dav_link_collection,divy_mailwatch は操作しません。
 *
 * (note) src_rdb_r 必須項目
 * 	uri, rsid, depth, resourcetype
 * (note) dist_rdb_r 必須項目
 * 	uri, depth, getcontentlength, creationdate, getlastmodified,
 * 	creator_userid, lastmodifier_userid, isversioned, checkin, checkout 
 * (note) 以下の項目はこの関数の中で自動的に補完します
 * 	physicalpath, getetag, rsid
 * (note) 以下の項目はcopy後もcopy 前と同じ値になります
 * 	displayname, getcontentlanguage, getcontentlength, getcontenttype,
 * 	resourcetype 
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 * 
 * @param r request_rec * request_rec構造体へのポインタ
 * @param src_rdb_r  const divy_rdbo_resource * 複製元(src)
 * @param dist_rdb_r       divy_rdbo_resource * 複製先(dist)
 * @param copyinfo divy_copyfile_info ** コピーされた物理ファイルの情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	0: 成功 / 1: 失敗 /
 *	DIVY_STCODE_NOT_EXIST     : コピー元がなくなっていた
 *	DIVY_STCODE_USER_NOT_EXIST: コピー操作者が存在しなかった
 */
DIVY_DECLARE(int) divy_rdbo_copy_resource_without_member(request_rec *r,
					divy_rdbo_resource *src_rdb_r,
					divy_rdbo_resource *dist_rdb_r,
					divy_copyfile_info **copyinfo,
					divy_db_transaction_ctx *ts_ctx);

/**
 * ソースとして指定されたdivy_rdbo_resource のリソースまたはコレクションを
 * ディスティネーションのdivy_rdbo_resource が示すURIの下に移動する。
 * 自身とそれ以下に存在するメンバがあれば、それらも全て移動します。
 *
 * 移動先に同じURIのレコードが存在しないものとして処理します。これは呼び出し
 * 側において保証してください。(mod_dav.c がやってくれているようです。)
 *
 * update 対象テーブル: dav_resource, dav_link_collection, divy_mailwatch
 * (note) src_rdb_r 必須項目
 * 	uri, rsid, displayname, depth, resourcetype
 * (note) dist_rdb_r 必須項目(変更項目)
 * 	uri, displayname, depth
 *
 * (note) 以下のデータは変更できません
 * 	getcontentlanguage, getcontentlength, getcontenttype, rsid,
 * 	resourcetype, physicalpath, creationdate, getetag, creator_userid,
 * 	lastmodifier_userid, isversioned, checkin, checkout
 *
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 * 
 * @param r request_rec * request_rec構造体へのポインタ
 * @param src_rdb_r  const divy_rdbo_resource * 移動元(src)
 * @param dist_rdb_r const divy_rdbo_resource * 移動先(dist)
 * @param iscommit int 確定してしまうかどうか。(1: 確定する / 0: 確定しない)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	0: 成功 / 1: 失敗 / DIVY_STCODE_NOT_EXIST: src が存在しなかった
 */
DIVY_DECLARE(int) divy_rdbo_move_resource(request_rec *r, 
					const divy_rdbo_resource *src_rdb_r,
					const divy_rdbo_resource *dist_rdb_r,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたロックリストを元にデータベースに追加する。
 *
 * @param r request_rec *
 * @param lockrec divy_rdbo_lock *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_lock_insert_record(request_rec *r,
						divy_rdbo_lock *lockrec);

/**
 * ロックテーブルを検索してnullresource(null lock)を返却する。
 *
 * @param r             request_rec *           リクエスト
 * @param uri           const char *            調べるURI
 * @param depth         int                     調べる階層(0 | 1 | INFINITY)
 * @param is_collection int                     collectionか否か
 * @param resource      divy_rdbo_resource **   nullresourceのリスト
 *
 */
DIVY_DECLARE(int) divy_rdbo_get_null_resource_member(request_rec *r,
						const char *uri,
	       			       		int depth, int is_collection,
				       		divy_rdbo_resource **resource);

/**
 * 指定されたメール監視URIに存在するユーザのメールアドレス取得する。
 * (note)
 * 	制限ユーザのメールアドレスは取得出来ません。
 *
 * @param r                     request_rec *
 * @param uri                   const char *
 * @param method		int メソッド番号
 * @return divy_rdbo_usr構造体  divy_rdbo_usr *
 */
DIVY_DECLARE(divy_rdbo_usr *) divy_rdbo_ml_get_addr(request_rec *r,
							const char *uri,
							int method);

/**
 * 指定されたロックトークンをデータベースより削除する。
 *
 * @param r request_rec *
 * @param locktoken dav_locktoken *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_locktoken(request_rec *r,
						dav_locktoken *locktoken);

/**
 * リクエスト内のURIを元にすべてのロック情報をDEPTHにより取得する。
 *
 * @param r request_rec *
 * @param lp divy_rdbo_lock_resource **
 * @param key const char *
 * @param depth int
 * @param expire int			タイムアウトチェックの有無
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_lock_record(request_rec *r,
						divy_rdbo_lock_resource **lp,
			      			const char *key, int depth,
						int expire);

/**
 * 指定されたロックトークンの時間を更新する。
 *
 * @param r request_rec *
 * @param locktoken dav_lcoktoken
 * @param new_time time_t
 * @return 結果ステータス(0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_locktoken(request_rec *r,
						dav_locktoken *locktoken,
			       			time_t new_time);

/**
 * NULLロックされているリソースを通常のロックに変更する。
 *
 * @param r request_rec *
 * @param lockrec divy_rdbo_lock *
 * @return 処理ステータス (0: 正常 / 1:失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_locknull_state(request_rec *r,
						divy_rdbo_lock *lockrec);

/**
 * ネームスペースURI(ネームスペース名) とネームスペースID の関係を以下の
 * ハッシュに格納して返却する。
 *
 * ns_id_hash : key = ネームスペースID(char*),  value = ネームスペースURI(char*)
 * ns_uri_hash: key = ネームスペースURI(char*), value = ネームスペースID (char*)
 *
 * @param r request_rec *
 * @param ns_id_hash apr_hash_t **  ネームスペースID のハッシュ
 * @param ns_uri_hash apr_hash_t ** ネームスペースURI のハッシュ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */ 
DIVY_DECLARE(int) divy_rdbo_build_nsmap_hash(request_rec *r,
						apr_hash_t **ns_id_hash,
						apr_hash_t **ns_uri_hash);

/**
 * 指定されたns_uri のネームスペースをdav_namespace テーブルに格納する。
 * ネームスペースID は、自動的に採番し、引数ns_id_str として返却します。
 *
 * @param r request_rec *
 * @param ns_uri const char * ネームスペースURI
 * @param ns_id_str char ** 採番したネームスペースIDを表す文字列
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */ 
DIVY_DECLARE(int) divy_rdbo_insert_dav_namespace(request_rec *r,
							const char *ns_uri,
							char **ns_id_str);
/**
 * 指定されたuri の Dead プロパティを取得して、返却する。
 *
 * @param r request_rec *
 * @param uri const char * URI を表す文字列。2重スラッシュ、末尾スラッシュは禁止.
 * @param d_pr divy_rdbo_dproperty ** 取得したDead プロパティ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_dead_property_by_uri(request_rec *r,
						const char *uri, divy_rdbo_dproperty **d_pr);

/**
 * 指定されたrsid, ns_id, name の Dead プロパティを取得して返却する。
 *
 * @param r request_rec *
 * @param rsid const char * リソースID
 * @param ns_id apr_int32_t ネームスペースID
 * @param name const char * Deadプロパティ名称
 * @param d_pr divy_rdbo_dproperty ** 取得したDead プロパティ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_dead_property(request_rec *r,
						const char *rsid, apr_int32_t ns_id, const char *name,
						divy_rdbo_dproperty **d_pr);

/**
 * 指定されたDead プロパティを新規登録する。
 *
 * @param r request_rec *
 * @param d_pr const divy_rdbo_dproperty * 登録するDead プロパティ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr);

/**
 * 指定されたDead プロパティの内容を更新する。
 *
 * update対象テーブル: dav_dead_property
 * d_pr 必須項目     : rsid, ns_id, name
 * (note)
 * 	lang_tag は、明示的に指定されない限り、昔の値をそのまま使います。
 * 	(WebDAV仕様からのの要請)
 *
 * @param r request_rec *
 * @param d_pr const divy_rdbo_dproperty * 更新するDead プロパティ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr);

/**
 * 指定されたDead プロパティを削除する。
 *
 * delete対象テーブル: dav_dead_property
 * d_pr 必須項目     : rsid, ns_id, name
 *
 * @param r request_rec *
 * @param d_pr const divy_rdbo_dproperty * 削除対象のDead プロパティ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_dead_property(request_rec *r,
					const divy_rdbo_dproperty *d_pr);

/**
 * ユーザの最終更新情報(アクセス時間, User-Agent) を記録する。
 * (認証が成功した場合のみ)
 * 
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int (0: 正常 / 1: エラー)
 */
DIVY_DECLARE(int) divy_rdbo_set_LastAccess(request_rec *r, divy_db_transaction_ctx *ts_ctx);

/**
 * ユーザがパスワードミスマッチでログインを失敗したことを記録
 * する。指定回数を上回った場合は、ログインを出来ないようにします
 *
 * @param r request_rec*
 * @return int (0: 正常 / 1: エラー)
 * 
 */
#define TF_LOCKOUT_NORMAL	0	/* 通常正常 */
#define TF_LOCKOUT_LOCKOUT	1	/* ロックアウトされた(正常) */
#define TF_LOCKOUT_FAILED   2	/* 致命的エラー */
DIVY_DECLARE(int) divy_rdbo_set_failed_login(request_rec *r);

/**
 * ロックアウトされているユーザをアクティブユーザへ変更する。
 * 
 * @param r request_rec *
 * @return int (0: 正常 / 1: エラー)
 *
 */
DIVY_DECLARE(int)divy_rdbo_refresh_lockout(request_rec *r);

/**
 * ユーザのセッションを調べて新規・更新を行う
 * これは認証が成功する前にも記録します。
 *
 * @param r request_rec *
 * @param uid const char ** ユーザID
 * @param sessionid char ** セッションID
 * @param password char ** パスワード
 * @return int (0: 正常 / 1: エラー)
 */
DIVY_DECLARE(int) divy_rdbo_session_manager(request_rec *r, char **uid, char **sessionid, char **password);

/**
 * セッション情報を取得する
 *
 * @param r request_rec *
 * @param sessionid const char*
 *
 * @return divy_rdbo_session (NULL: データなし / not NULL セッション情報あり)
 *
 */
DIVY_DECLARE(divy_rdbo_session*)
divy_rdbo_get_session(request_rec *r, const char *sid);

/**
 * セッション情報のパスワードを更新する
 *
 * @param r request_rec *
 * @param ts_ctx divy_db_transaction_ctx *
 * @param sessionid const char * 対象のセッションID(NULLの場合すべてのセッション)
 * @param user const char *		対象のユーザID（NULLの場合すべてのユーザ)
 * @param password const char * 変更するパスワード（NULLの場合エラー)
 *
 */
DIVY_DECLARE(int) divy_rdbo_update_session_password(request_rec *r, divy_db_transaction_ctx *ts_ctx, const char* sessionid, const char* user, const char* password);

/**
 * ユーザのセッションを削除する
 *
 * @param r request_rec *	リクエスト構造体
 * @param ts_ctx divy_db_transaction_ctx * トランザクション
 * @param user const char*	ユーザID
 * @param sessionid const char* セッションID 
 * (note)
 * userとsessionidの&で削除が行えます。
 * userだけでsessionidがNULLの場合はそのユーザIDをすべてクリア
 * sessionidだけでuserがNULLの場合は、セッションIDだけクリア
 * userとセッションIDどちらもNULLの場合はすべてのセッションをクリア
 */
DIVY_DECLARE(int) divy_rdbo_delete_session(request_rec *r,
						divy_db_transaction_ctx *ts_ctx,
						const char *user,
						const char *sessionid);

/**
 * 指定されたソースコレクション( = linkdbsearch の結果フォルダ) のuri が
 * 持っている共有コレクションの情報を取得して返却する。
 *
 * select 対象のテーブル: dav_link_collection, dav_resource
 *
 * (note) ソースコレクションのuri
 * 	uri = $root/.dbfolder/$keys
 *
 * @param r request_rec *
 * @param uri const char * ソースコレクションのuriを表す文字列
 * @param  shlink_pr divy_rdbo_shlink ** 取得した共有コレクションの情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_sharedcollection_property(request_rec *r,
						const char *uri,
						divy_rdbo_shlink **shlink_pr);

/**
 * 指定されたソースコレクションのuri にshlink_pr が示す共有コレクション
 * へのリンクを追加する。
 * 該当する古いdav_link_collection の値を削除してからshsrclink_pr で指定された
 * 値をinsert して下さい。
 *
 * insert,delete 対象のテーブル: dav_link_collection
 * (note) shlink_pr の必須項目
 * 	uri
 * (note)
 * 	uri が示すソースコレクションは、必ず存在しています。
 * (note) ts_ctxがない場合は失敗します。
 *
 * @param r request_rec *
 * @param uri const char * ソースコレクションのuri
 * @param shlink_pr const divy_rdbo_shlink * ソースコレクションにつける
 * 						共有コレクションの情報を表す
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_sharedcollection_property(request_rec *r,
					const char *uri,
					const divy_rdbo_shlink *shlink_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたrdb_r の共有コレクションを作成し、shsrccol_pr が示すソースコレクション
 * との関係を記録する。
 *
 * (note) ソースコレクションの存在について (2003/05/22, 30)
 *	ソースコレクションが存在しない場合でも、そのソースコレクションのuri を
 *	使って、共有コレクションとの関係を記録するようにしました。
 *	可能ならば、ソースコレクションが存在しなければ、自動的に作成できれば
 *	良いのですが、件数に比例して性能が著しく劣化するため、このような操作は
 *	行わないことにしました。
 *	よって、ソースコレクションのdav_resource エントリが作成されるのは
 *	ソースコレクションに対するMKCOLが発行された時です。
 * 
 * insert対象のテーブル:
 * 	dav_resource (共有コレクションの情報)
 * 	dav_link_collection (共有コレクション-ソースコレクションURI)
 *
 * (note) rdb_r の必須項目: insert したいデータ全て。
 * (note) shsrccol_pr の必須項目: uri
 *
 * @param r request_rec *
 * @param rdb_r const divy_rdbo_resource *  共有コレクションのリソース構造体
 * @param shsrccol_pr const divy_rdbo_shsrccol * 挿入するソースコレクションの情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_sharedsrccollection_info(request_rec *r,
					const divy_rdbo_resource *rdb_r,
					const divy_rdbo_shsrccol *shsrccol_pr);

/**
 * divy_userテーブルのエントリ数(現在のユーザ総数)を返却する。
 *
 * @param request_rec * request_rec 構造体へのポインタ
 * @param int * usr_cnt ユーザ数
 * @return 処理ステータス 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_rdbo_count_usr(request_rec *r, int *usr_cnt);

/**
 * 指定されたuri, usrid の メール監視情報を取得して返却する。
 *
 * select 対象のテーブル:
 * 	divy_mailwatch
 * (note) usrid がNULLの場合は uri が一致するデータ、
 * 	　それ以外は uri,usrid が一致するデータを返却します。
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param  r request_rec *
 * @param  uri const char * uriを表す文字列
 * @param  usrid const char * ユーザID
 * @param  mwatch_pr divy_rdbo_mailwatch ** 取得したメール監視の情報
 * @param  ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_mailwatch_property(request_rec *r,
					const char *uri, const char *usrid,
					divy_rdbo_mailwatch **mwatch_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたmwatch_pr->uri が示すコレクションにmailwatch_pr->trigger_methods 
 * が示すメール監視状態を追加する。
 *
 * insert 対象テーブル: divy_mailwatch
 * mwatch_pr 必須項目 : uri, usrid, trigger_methods
 * (note) delflag は設定されていても無視します。常に0を設定します。
 *        owner_trigger_methos は無視します。この変数は歴史的な経緯から
 *        書き込み時には監視ユーザの監視フラグを表さないためです。
 *        mwatch_pr->next は無視します。
 * (note)
 *   uri, usrid の組が既にdivy_mailwatch に登録されていないことを呼び出し元で
 *   保証して下さい。
 *
 * @param r request_rec *
 * @param mwatch_pr divy_rdbo_mailwatch * 追加するメール監視状況
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
                              		divy_db_transaction_ctx *ts_ctx);
/**
 * 指定されたmwatch_pr->uri が示すコレクションをmailwatch_pr->trigger_methods 
 * が示すメール監視状態で更新する。
 * 既に、mwatch_pr->uri のエントリがdivy_mailwatch に存在するものとします。
 * 純粋にmailwatch_pr->trigger_methods を更新するだけとします。
 *
 * update 対象テーブル: divy_mailwatch
 * mwatch_pr 必須項目 : uri, usrid, trigger_methods
 * (note) delflag は設定されていても無視します。常に0を設定します。
 *        owner_trigger_methos は無視します。この変数は歴史的な経緯から
 *        書き込み時には監視ユーザの監視フラグを表さないためです。
 *        mwatch_pr->next は無視します。
 * (note)
 *   uri, usrid の組がdivy_mailwatch に登録されていることを呼び出し元で
 *   保証して下さい。
 *
 * @param r request_rec *
 * @param mwatch_pr divy_rdbo_mailwatch * 追加するメール監視状況
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_mailwatch_property(request_rec *r,
					const divy_rdbo_mailwatch *mwatch_pr,
                              		divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたmwatch_pr->uri, usrid が示すメール監視状態を削除する。
 *
 * delete/update 対象テーブル: divy_mailwatch
 * mwatch_pr 必須項目 : uri
 *
 * 	usrid が指定されなかった場合には、uri だけで削除します.
 * 	uri が指定されなかった場合には、usrid だけで削除します.
 * 	両方とも指定されなかったらNGです.
 *
 * (note) delflag = 1 の時は論理削除(update)します。
 *        owner_trigger_methos は無視します。この変数は歴史的な経緯から
 *        書き込み時には監視ユーザの監視フラグを表さないためです。
 *        mwatch_pr->next は無視します。
 *
 * @param r request_rec *
 * @param mwatch_pr divy_rdbo_mailwatch * 削除するメール監視状況
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_mailwatch_property(request_rec *r,
									const divy_rdbo_mailwatch *mwatch_pr,
                              		divy_db_transaction_ctx *ts_ctx);

/**
 * グループについているグループメール監視状態を全て取得し、
 * 相対URIをキーとしてこの状態をハッシュの値として返却する.
 *
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます
 * 	* tf_rdbo_group.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param gmw_h apr_hash_t ** (キー: 相対URI / 値: divy_rdbo_mailwatch *)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_groupmailwatch_properties(request_rec *r,
							apr_hash_t **gmw_h, divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたgrpid のグループが保持するデフォルトメール監視フラグ
 * (groupmailwatch) を所属ユーザusrid に付加する。
 *
 * また、usrid がNULL の時には、grpid のグループに所属している全ユーザに
 * grp_full_uri のメール監視フラグをコピーする。
 *
 * (note)
 * 	* usrid がNULL の時、grp のグループに所属しているユーザの
 * 	  メール監視状態が、既にdivy_mailwatch に記録していないことを
 * 	  呼び出し元にて保証して下さい。
 * 	* トランザクションが継続中であれば引き継ぎます
 * 	* tf_rdbo_group.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param usrid const char * ユーザID.
 * @param copymode int コピーモード
 * 		DIVY_CPMWATCH_NORMAL      : grpidのメール監視フラグをusridのユーザにコピー
 * 		DIVY_CPMWATCH_IGNORE_USER : grpidのメール監視フラグを所属全ユーザにコピー
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_copy_groupmailwatch_to_user(request_rec *r,
							const char *grpid,
							const char *usrid,
							int copymode,
							divy_db_transaction_ctx *ts_ctx);

/**
 * 指定された src_grp_uri のグループに所属するユーザの mailwatch 設定を
 * dst_grp_uri の mailwatch 設定としてコピーする。
 *
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます
 * 	* tf_rdbo_group.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param src_grp_uri const char * src グループのURI
 * @param dst_grp_uri const char * dst グループのURI
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_copy_usermailwatch_to_group(request_rec *r,
							const char *src_grp_uri,
							const char *dst_grp_uri,
							divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたuri のユーザリレーションを取得する
 * 
 * select 対象テーブル: divy_grpmem
 *
 * @param r request_rec *
 * @param uri const char * ユーザリレーションのuri
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_rusr(request_rec *r,
					const char *uri,
					divy_rdbo_rusr **rusr);

/**
 * 指定されたrusr のユーザリレーションを追加し、
 * rusr にrusr->grp_uri のグループが持っていたメール監視状態を設定する。
 * (note)
 *   ユーザが制限ユーザであればメール監視状態は設定されません。
 * 
 * insert対象テーブル: divy_grpmem
 * rusr 必須項目: usrid, grp_uri
 *
 * @param r request_rec *
 * @param rusr const const divy_rdbo_ruser * １つのユーザリレーション
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_rusr(request_rec *r,
					const divy_rdbo_rusr *rusr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたrusr のユーザリレーション削除する。
 * 
 * 削除対象テーブル: divy_grpmem
 * rusr 必須項目: usrid, grpid
 * (note)
 * 	ユーザ自身(ユーザエンティティ)の削除は行いません。
 * 	グループとのリレーションが消えるだけです。
 *
 * @param r request_rec *
 * @param rusr const const divy_rdbo_ruser * １つのユーザリレーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_rusr(request_rec *r,
					const divy_rdbo_rusr *rusr);

/**
 * 指定されたsrc_rusr が示すユーザリレーションをdist_rusr の示す
 * ユーザリレーションに移動する。
 *
 * update 対象テーブル: divy_grpmem
 * src_rusr  必須項目: usrid, grpid
 * dist_rusr 必須項目: usrid, grp_uri
 * (note)
 * 	dist_rusr のgrp_uri メンバが示すグループが存在しなければ
 * 	オペレーションは失敗します。
 *
 * @param r request_rec *
 * @param src_rusr const divy_rdbo_rusr * ソース
 * @param dist_rusr const divy_rdbo_rusr * ディスティネーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_rusr(request_rec *r,
					const divy_rdbo_rusr *src_rusr,
					const divy_rdbo_rusr *dist_rusr);

/**
 * 指定されたuri のSQLリレーションを取得する
 *
 * (note)
 * 	親グループに割り当てられたSQLの使用権は、下位グループにも継承される
 * 	という仕様になっているため、下位階層のグループには直接割り当てられて
 * 	いなくても、対応するSQLリレーションを取得して返却します。
 * 	なお、継承されてきたかどうかは、*rsqlのメンバinheritsql で確認します。
 * 
 * select 対象テーブル: divy_sqlmem, divy_sql, divy_grp
 *
 * @param r request_rec *
 * @param uri const char * SQLリレーションのuri
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_rsql(request_rec *r,
					const char *uri,
					divy_rdbo_rsql **rsql);

/**
 * 指定されたrsql のSQLリレーションを追加する。
 * 
 * insert対象テーブル: divy_sqlmem
 * rusr 必須項目: labelname, grp_uri
 *
 * @param r request_rec *
 * @param rsql const const divy_rdbo_rsql * １つのSQLリレーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_rsql(request_rec *r,
					const divy_rdbo_rsql *rsql);

/**
 * 指定されたrsql のSQLリレーションを削除する。
 *
 * 削除対象テーブル: divy_sqlmem
 * rsql 必須項目: sqlid, grpid
 * (note)
 * 	SQL自身(SQLエンティティ)の削除は行いません。
 * 	グループとのリレーションが消えるだけです。
 *
 * @param r request_rec *
 * @param rsql const divy_rdbo_rsql * 一つのSQLリレーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_rsql(request_rec *r,
					const divy_rdbo_rsql *rsql);	

/**
 * 指定されたsrc_rsql が示すSQLリレーションをdist_rsql の示す
 * SQLリレーションとして移動する。
 *
 * update 対象テーブル: divy_sqlmem
 * src_rusr  必須項目: sqlid, grpid
 * dist_rusr 必須項目: labelname, grp_uri
 *
 * @param r request_rec *
 * @param src_rsql const divy_rdbo_rsql * ソース
 * @param dist_rsql const divy_rdbo_rsql * ディスティネーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_rsql(request_rec *r,
					const divy_rdbo_rsql *src_rsql,
					const divy_rdbo_rsql *dist_rsql);

/**
 * 指定されたuri のグループリレーションを取得する
 * 
 * select 対象テーブル: divy_grpmem / divy_sqlmem, divy_sql, divy_grp
 * (note)
 * 	uri がどの種類(ユーザ / SQL) とのリレーションを表すかに
 * 	よって、取得する値が異なります。
 *
 * @param r request_rec *
 * @param uri const char * グループリレーションのuri
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_rgrp(request_rec *r,
					const char *uri,
					divy_rdbo_rgrp **rgrp);

/**
 * 指定されたrgrp のグループリレーションを追加する。
 * なお、ユーザのグループリレーションの場合、追加対象グループが持っている
 * デフォルトメール監視状態を、ユーザに設定します。
 * 
 * insert対象テーブル: divy_grpmem, divy_mailwatch / divy_sqlmem
 * rgrp 必須項目:
 * 	grpid, usrid     (kind_of_relation = DIVY_RGRP_TO_USER)
 *	grpid, labelname (kind_of_relation = DIVY_RGRP_TO_SQL)
 *
 * @param r request_rec *
 * @param rsql const const divy_rdbo_rsql * １つのSQLリレーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_rgrp(request_rec *r,
					const divy_rdbo_rgrp *rgrp);

/**
 * 指定されたrgrp のグループリレーションを削除する。
 *
 * 削除対象テーブル: divy_grpmem / divy_sqlmem
 * rgrp 必須項目:
 * 	grpid, usrid  (kind_of_relation = DIVY_RGRP_TO_USER)
 *	grpid, sqlid  (kind_of_relation = DIVY_RGRP_TO_SQL)
 *
 * (note)
 * 	グループ自身(グループエンティティ)の削除は行いません。
 * 	ユーザまたはSQLとのリレーションが消えるだけです。
 *
 * @param r request_rec *
 * @param rgrp const divy_rdbo_rgrp * 一つのグループリレーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_rgrp(request_rec *r,
					const divy_rdbo_rgrp *rgrp);

/**
 * 指定されたsrc_rsql が示すSQLリレーションをdist_rsql の示す
 * SQLリレーションとして移動する。
 *
 * update 対象テーブル: divy_sqlmem
 * rgrp 必須項目:
 * 	grpid, usrid  (kind_of_relation = DIVY_RGRP_TO_USER)
 *	grpid, sqlid  (kind_of_relation = DIVY_RGRP_TO_SQL)
 *
 * @param r request_rec *
 * @param src_rgrp const divy_rdbo_rgrp * ソース
 * @param dist_rgrp const divy_rdbo_rgrp * ディスティネーション
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_rgrp(request_rec *r,
					const divy_rdbo_rgrp *src_rgrp,
					const divy_rdbo_rgrp *dist_rgrp);

/**
 * 指定されたユーザリレーションのuriを解析して、ユーザIDとグループuriを
 * 切り出す。
 * (note) ユーザリレーションuri のフォーマット
 * 	$root/.management/GROUP/$grp_uri/.RU_$userid
 *
 * @param r request_rec *
 * @param uri const char * ユーザリレーションのuriを表す文字列
 * @param rusr divy_rdbo_rusr ** 解析した結果を格納する構造体
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_parse_rusr_uri(request_rec *r,
					const char *uri, divy_rdbo_rusr **rusr);

/**
 * 指定されたSQLリレーションのuriを解析して、SQL表示名とグループuriを
 * 切り出す。
 * (note) SQLリレーションuri のフォーマット
 *	$root/.management/GROUP/$grp_uri/.RS_$sqlname
 *
 * @param r request_rec *
 * @param uri const char * SQLリレーションのuriを表す文字列
 * @param rsql divy_rdbo_rsql ** 解析した結果を格納する構造体
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_parse_rsql_uri(request_rec *r,
					const char *uri, divy_rdbo_rsql **rsql);

/**
 * 指定されたグループリレーションのuriを解析して、ユーザIDまたはSQL表示名と
 * グループuriを切り出す。
 * (note) グループリレーションuri のフォーマット
 *	$root/.management/USER/$userid/.RG_$grpid
 *	$root/.management/SQL/$sqlname/.RG_$grpid
 *
 * @param r request_rec *
 * @param uri const char * グループリレーションのuriを表す文字列
 * @param rgrp divy_rdbo_rgrp ** 解析した結果を格納する構造体
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_parse_rgrp_uri(request_rec *r,
					const char *uri, divy_rdbo_rgrp **rgrp);

/**
 * 指定されたグループIDからなる集合grpid_hashを使ってdivy_grpテーブルを検索し、
 * uriとグループIDの組合せを取得。
 *
 * @param r request_rec *
 * @param grpid_set divy_cset_t * グループIDからなる文字列集合
 * @param grp_pr divy_rdbo_grp ** 取得した結果からなるリスト
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_grpuri_by_grpid(request_rec *r,
							divy_cset_t *grpid_set,
							divy_rdbo_grp **grp_pr);

/**
 * 現時点で全ユーザに割り当てられたQuotaサイズとファイル数を取得する。
 *
 * select対象テーブル: divy_usrdiskquota
 *
 * @param r request_rec *
 * @param maxst apr_int64_t * 全Quotaサイズの合計(byte単位)
 * @param maxres apr_int64_t * 全Quotaファイル数の合計(byte単位)
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_allocated_quota(request_rec *r,
						apr_int64_t *maxst,
						apr_int64_t *maxres);


/**
 * 指定されたuri 以下のリソースの内、Directロックされているリソース一覧を取得する。
 *
 * select 対象: dav_resource, dav_lock
 * 取得データ: uri, displayname, resourcetype
 *
 * @param r request_rec *
 * @param uri const char * 検索対象となるuri
 * @param rdb_r divy_rdbo_resource ** 取得したロックされたリソース一覧
 * @return int 処理ステータス(1: 失敗 / 0: 成功)
 */
DIVY_DECLARE(int) divy_rdbo_get_hierarchy_lockedresource(request_rec *r,
						const char *uri,
						divy_rdbo_resource **rdb_r);

/**
 * 指定されたrdb_r->rsid と一致するリソースのプロパティを取得する。
 * 取得した値は、引数rdb_r のプロパティに詰めて返却します。
 * 値が取得できたかどうかは、IS_EMPTY(rdb_r->uri) が真かどうかで判定して下さい。
 *
 * select 対象テーブル:	dav_resource
 *
 * (note) rdb_r 必須項目
 * 	rdb_r->rsid
 *
 * @param r request_rec * request_rec構造体へのポインタ
 * @param rdb_r divy_rdbo_resource * 検索対象&検索結果
 * @param propflag apr_uint32_t 特別に取得するプロパティの種類 (ORで指定する)
 * 	DIVY_GET_PROP_fullname      : 作成者,更新者名称プロパティを取得する
 * 	DIVY_GET_PROP_resourcestate : 状態・属性プロパティを取得する
 * @return 処理ステータス 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_rdbo_get_property_by_rsid(request_rec *r,
						divy_rdbo_resource *rdb_r,
						apr_uint32_t propflag);


/**
 * 指定されたrdb_r が表すコレクション直下のリソースをoffset から limit まで取得して
 * 返却する。rdb_r 自身のリソースは含みません。(autoindex 用)
 *
 * select 対象テーブル: dav_resource
 * rdb_r 必須項目: uri, depth, u_spec
 * (note)
 * 	* offset は 1から始まります
 * 	* limit が 0 の場合、検索は実施しません。
 * 	* ゴミ箱フォルダは返却しません。またゴミ箱フォルダ以下の検索も行いません
 *
 * @param r request_rec * request_rec構造体へのポインタ
 * @param rdb_r divy_rdbo_resource * 検索条件 & 検索結果
 * @param offset apr_int32_t オフセット
 * @param limit apr_int32_t 取得件数
 * @param propflag apr_uint32_t  特別に取得するプロパティの種類 (ORで指定する)
 * 	DIVY_GET_PROP_fullname      : 作成者,更新者名称プロパティを取得する
 * 	DIVY_GET_PROP_resourcestate : 状態・属性プロパティを取得する
 * 	DIVY_DEL_PROP_noviewattr    :  View属性が"付いていない"リソースを削り落とす(基点は除外)
 * @return 処理ステータス	0: 成功 / 1: 失敗 / DIVY_STCODE_REMAIN_DATA: 続きページがある
 */
DIVY_DECLARE(int) divy_rdbo_get_autoindex_data(request_rec *r,
						divy_rdbo_resource *rdb_r,
						apr_int32_t offset,
						apr_int32_t limit,
						apr_uint32_t propflag);
#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * システムQuota情報を取得して返却する。
 *
 * select 対象のテーブル: divy_diskquota
 * (note)
 * 	トランザクションが継続中であれば引き継ぎます。
 *
 * @param  r request_rec *
 * @param  sysquota_pr divy_rdbo_diskquota ** 取得したシステムQuotaの情報.なければNULL
 * @param  ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_systemquota_property(request_rec *r,
					divy_rdbo_diskquota **sysquota_pr,
					divy_db_transaction_ctx *ts_ctx);
#endif /* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * sysquota_pr.uri で指定されたLocation のシステムQuotaを取得する。
 * なお、物理Diskの情報は取得しません。その用途には関数
 * divy_rdbo_get_systemquota_property を利用してください。
 *
 * (note)
 * 	do_entrylock フラグが立っていたらエントリをロックして返却します。
 * 	その際、ts_ctx がNULLであれば、エラーとなります。
 * (note)
 * 	デッドロックが起きないことは呼び出し側で保証して下さい。
 * 	この関数は一切感知しません。
 *
 * @param r request_rec *
 * @param sysquota_pr divy_rdbo_diskquota * 対象LocationのURIと取得したQuota 情報
 * @param do_entrylock int 1: ロックをかけてエントリを取得 / 0: ロックをかけない
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 * 	(0: 成功 / 1: 失敗 / DIVY_STCODE_SYSQUOTA_NOT_EXIST: システムQuotaエントリが無かった)
 */
DIVY_DECLARE(int) divy_rdbo_get_systemquota(request_rec *r,
					divy_rdbo_diskquota *sysquota_pr,
					int do_entrylock,
			   		divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * 指定された sysquota_pr が示すシステムQuotaを登録/更新する。
 * (note)
 * 	外部仕様に従えば、システムが動いている限り、システムQuota情報は
 * 	divy_diskquotaに必ず設定されていなければなりません。
 * 	つまりこの関数が呼び出されるシチュエーションでは、そうであるという
 * 	ことです。
 * 	ところが、システムQuotaの外部仕様は後から追加されたため、
 * 	システムQuotaが存在しないまま動作しているシステムも存在しています。
 * 	仕方がないので、なければINSERTするという動作を行うことにしています。
 *
 * insert 対象テーブル: divy_diskquota
 * mwatch_pr 必須項目 : uri, type, maxst, maxres
 *	新規登録に限りusedstとmaxres は自動的に算出して設定します。
 *	更新時には、usedstとmaxresは無視されます。
 *
 * (note)
 *    訳あって、ここのロジックをmaintenance.sh でもコピーして使用しています。
 *    ここの方式を変更する場合には、maintenance.sh 側も修正して下さい。
 *
 * @param r request_rec *
 * @param sysquota_pr divy_rdbo_diskquota * 登録/更新するシステムQuota情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_systemquota_property(request_rec *r,
					divy_rdbo_diskquota *sysquota_pr,
                              		divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_GROUPQUOTA
/**
 * grpquota_pr.grpid で指定されたグループのグループQuotaを取得する。
 *
 * @param r request_rec *
 * @param grpquota_pr divy_rdbo_grpquota * 対象のグループIDと取得したQuota情報
 * @param do_entrylock int 1: ロックをかけてエントリ取得 / 0: ロックしない
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_groupquota(request_rec *r,
					divy_rdbo_grpquota *grpquota_pr,
					int do_entrylock,
					divy_db_transaction_ctx *ts_ctx);
#endif /* DIVY_SUPPORT_GROUPQUOTA */

/**
 * 指定されたuri の親コレクションに付いている状態・属性を取得する。
 * (note)
 * 	divy_rdbo_resourcestate のinherit メンバには値を設定しません。
 * 	inherit メンバを参照してはなりません。
 *
 * @param r request_rec *
 * @param uri const char *
 * @param type int 状態・属性プロパティの種類
 * @param rstate_pr divy_rdbo_resourcestate ** 状態, 属性プロパティ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_parent_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					divy_rdbo_resourcestate **rstate_pr);

/**
 * 指定されたuri, type の状態・属性を取得する。
 * (note)
 * 	consider_inherit が1であれば、uri自身が状態・属性プロパティを持って
 * 	いなくても親から継承していれば、自身が持っているかのようにプロパティを
 * 	返却します。
 * 	このケースでは divy_rdbo_resourcestate のinherit フラグを立てます。
 *
 * @param r request_rec *
 * @param uri const char *
 * @param type int 状態・属性プロパティの種類
 * @param rstate_pr divy_rdbo_resourcestate ** 状態・属性プロパティ
 * @param consider_inherit int 親から継承する状態・属性を考慮するかどうか
 * 			(1: 考慮する / 0: 考慮しない)
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_resourcestate_property(request_rec *r,
					const char *uri,
					int type,
					int consider_inherit,
					divy_rdbo_resourcestate **rstate_pr);

/**
 * 指定されたuri以下に存在する"有効な"type の状態・属性があるかどうかを調べる。
 *
 * @param r request_rec *
 * @param uri const char *
 * @param type int 状態・属性プロパティの種類
 * @param consider_own int 自分自身の状態・属性プロパティを考慮するかどうか(0: 考慮しない)
 * @return int 処理ステータス
 * 	1 : 失敗
 * 	DIVY_STCODE_EFFECTIVE_RSTATE_EXIST   : 有効な状態プロパティがあった
 * 	DIVY_STCODE_INEFFECTIVE_RSTATE_EXIST : 無効な状態プロパティを含んでいた
 */
DIVY_DECLARE(int) divy_rdbo_is_resourcestate_effective(request_rec *r,
					const char *uri, int type, int consider_own);

/**
 * 指定されたrstate_pr の状態・属性プロパティを作成/更新/削除する。
 * (note)
 * 	* トランザクションが継続中であればそれを引き継ぎます。
 * 	* rstate_pr->uri がグループコレクションより下のリソースでなければエラー
 *
 * @param r request_rec *
 * @param rstate_pr divy_rdbo_resourcestate * 追加される状態・属性プロパティ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_resourcestate_property(request_rec *r,
					divy_rdbo_resourcestate *rstate_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * src_uri 以下の状態・属性プロパティをdst_uriのURIになるよう変更(移動)する。
 * (note)
 * 	トランザクションを継承します。
 *
 * @param r request_rec *
 * @param src_uri const char * 移動元URI
 * @param dst_uri const char * 移動先URI
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
DIVY_DECLARE(int) divy_rdbo_move_resourcestate_property(request_rec *r,
							const char *src_uri,
							const char *dst_uri,
							divy_db_transaction_ctx *ts_ctx);

/**
 * uri 以下にある状態/属性プロパティを削除する。
 *
 * [ 削除対象テーブル ] divy_resourcestate
 *
 * @param r request_rec *
 * @param uri const char * 削除する状態/属性プロパティを持つURI
 * @param type int リソース種別
 * @return int 処理ステータス (0: 正常 / 1: 失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_resourcestate_property(request_rec *r,
					const char *uri, int type, divy_db_transaction_ctx *ts_ctx);

/**
 * 指定したURLのBOX情報を削除する
 * @param r       request_rec *
 * @param shorten const char*  グループのURL	
 * @return int 0: 成功 / >0 失敗
 *
 */
DIVY_DECLARE(int) divy_rdbo_remove_box_property(request_rec *r, const char* uri, divy_db_transaction_ctx *ts_ctx);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティを取得する.
 * [ 有効性 ]
 *   パスワードポリシーがサポートされていなければ *passpolicy_pr はNULLです
 *
 * @param r request_rec *
 * @param passpolicy_pr divy_rdbo_passwordpolicy * 登録/更新するパスワードポリシー情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_passwordpolicy_property(request_rec *r,
					int policyid,
					divy_rdbo_passwordpolicy **passpolicy_pr,
					divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティを登録または更新する.
 *
 * @param r request_rec *
 * @param passpolicy_pr divy_rdbo_passwordpolicy * 登録/更新するパスワードポリシー情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_passwordpolicy_property(request_rec *r,
					divy_rdbo_passwordpolicy *passpolicy_pr,
					divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ポリシーID policyid のパスワードポリシープロパティを削除する.
 *
 * @param r request_rec *
 * @param policyid int ポリシーID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_passwordpolicy_property(request_rec *r,
				int policyid, divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */


#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザのパスワードを登録または更新する.
 * (note)
 *   パスワードポリシー機能がサポートされていなければ、関連テーブルは
 *   更新しません.
 *   また、ユーザポリシーステータスがNULL ならば、エラーにはしませんが、
 *   ステータスのアップデートは行ないません.
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * 登録/更新するユーザのパスワード
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_userpassword(request_rec *r,
					divy_rdbo_usr *usr_pr,
					divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザID userid のユーザが持つパスワード履歴を取得する. 
 * [ 有効性 ]
 *   パスワードポリシーがサポートされていなければ *ph はNULLです
 *
 * @param r request_rec *
 * @param userid const char * ユーザID
 * @param ph divy_rdbo_passhistory ** パスワード履歴
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_passwordhistory(request_rec *r,
					const char *userid,
					divy_rdbo_passhistory **ph,
					divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザの持つパスワードポリシー状態を登録または更新する.
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます。
 * 	* tf_rdbo_user.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param passpolicystatus divy_rdbo_passpolicystatus * 
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
DIVY_DECLARE(int) divy_rdbo_update_user_passwordpolicy(request_rec *r,
					divy_rdbo_passpolicystatus *passpolicystatus,
					divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ユーザID src_userid のパスワードポリシーをdst_userid のユーザに
 * 移動する.
 *
 * @param r request_rec *
 * @param src_userid cosnt char * src ユーザID
 * @param dst_userid cosnt char * dst ユーザID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_move_user_passwordpolicy(request_rec *r,
						const char *src_userid,
						const char *dst_userid,
						divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid のユーザのパスワード履歴に追加する.
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます。
 * 	* tf_rdbo_user.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param userid const const char *
 * @param new_password const char *
 * @param ts_ctx divy_db_transaction_ctx *
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_update_user_passhistory(request_rec *r,
							const char *userid,
							const char *new_password,
							divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid のユーザのパスワードポリシー履歴を消去する.
 *
 * @param r request_rec *
 * @param userid const char * ユーザID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_passhistory(request_rec *r,
							const char *userid, divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * src_userid のパスワードポリシー履歴をdst_userid に移動する.
 *
 * @param r request_rec *
 * @param src_userid cosnt char * src ユーザID
 * @param dst_userid cosnt char * dst ユーザID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_move_user_passhistory(request_rec *r,
						const char *src_userid,
						const char *dst_userid,
						divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * 指定されたgrp_pr が示すグループフォルダ以下からbordertime よりも
 * 古い更新日付を持つファイルとフォルダを削除する.
 * なお、フォルダ内にファイルがある場合には維持します.
 * (note)
 *   ts_ctx のトランザクションを継続します.
 *
 * 関数 divy_rdbo_remove_resource() を参照
 *
 * @param r request_rec *
 * @param grp_pr divy_rdbo_grp * グループプロパティ
 * @param bordertime apr_time_t 境界時刻
 * @param isDeleteFolder int フォルダを削除するかどうか(1: 削除する / 0: 削除しない)
 * @param del_filelist divy_linkedlist_t ** 削除したファイルの物理相対パスリスト
 * @param del_res divy_rdbo_deletedres ** 削除されたフォルダ、リソース一覧を記述する
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_old_groupresources(request_rec *r,
								divy_rdbo_grp *grp_pr,
								apr_time_t bordertime,
								int isDeleteFolder,
								divy_linkedlist_t **del_filelist,
								divy_autodel_dresource **del_res,
								divy_db_transaction_ctx *ts_ctx);

/**
 * userid が示すユーザのプライベートコレクションを作成する。
 *
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます。
 * 	* tf_rdbo_user.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param userid const char * ユーザID
 * @param rdb_r divy_rdbo_resource *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
DIVY_DECLARE(int) divy_rdbo_create_private_collection(request_rec *r,
								const char *userid,
								divy_rdbo_resource *rdb_r,
								divy_db_transaction_ctx *ts_ctx);

/**
 * grpname が示すグループコレクションを生成する.
 *
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます
 * 	* tf_rdbo_group.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param grpname const char * グループ名
 * @param rdb_r divy_rdbo_resource * グループコレクションを表すリソース
 * @param groupid char ** 作成されたグループコレクション用のグループID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 成功可否 (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_create_group_collection(request_rec *r,
								const char *grpname,
								divy_rdbo_resource *rdb_r,
								char **groupid,
								divy_db_transaction_ctx *ts_ctx);

/**
 * uri のごみ箱を作成する。
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます
 * 	* tf_rdbo_user.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param uri const char * ごみ箱のURI
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 正常 / 1: 異常)
 */
DIVY_DECLARE(int) divy_rdbo_create_trash_property(request_rec *r,
								const char *uri, divy_db_transaction_ctx *ts_ctx);

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passpolicystatus のパスワードポリシーを登録する
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます
 * 	* tf_rdbo_user.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param passpolicystatus divy_rdbo_passpolicystatus * パスワードポリシー状態
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
DIVY_DECLARE(int) divy_rdbo_insert_user_passwordpolicy(request_rec *r,
						divy_rdbo_passpolicystatus *passpolicystatus,
						divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid が示すユーザのパスワードポリシー(状態)を削除する.
 * (note)
 * 	* トランザクションが継続中であれば引き継ぎます
 * 	* tf_rdbo_user.h 作成に伴いpublic 化しました
 *
 * @param r request_rec *
 * @param useric const char * ユーザID. NULLだと何もしません.
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
DIVY_DECLARE(int) divy_rdbo_remove_user_passwordpolicy(request_rec *r,
						const char *userid, divy_db_transaction_ctx *ts_ctx);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * confirmreading_pr が表す開封通知を追加する.
 * (note)
 *   トランザクションがなければNGとします.
 *
 * @param r request_rec *
 * @param cr_pr const const divy_rdbo_confirmreading *
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 */
DIVY_DECLARE(int) divy_rdbo_insert_confirmreading(request_rec *r,
					const divy_rdbo_confirmreading *cr_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * uri, userid が示す開封通知を削除する.
 *
 * uri, userid がNULLかどうかにより、動作が異なります.
 *
 * * userid == NULL : uri と一致する全ての開封通知を削除
 * * uri    == NULL : userid と一致する全ての開封通知を削除
 *
 * [ uri_scope について ]
 * * uri == NULL:
 * 	uri_scope は使用しません. 0を入れてください.
 *
 * * uri != NULL && uri_scope == 0:
 * 		対象リソースのみの開封通知を削除
 *
 * * uri != NULL && uri_scope == DAV_INFINITY:
 * 		対象リソースまたはそれ以下にある全ての開封通知を削除
 *
 * @param r request_rec *
 * @param uri const char * URI
 * @param userid const char * ユーザID
 * @param uri_scope int 削除スコープ (-1 / 0 / DAV_INFINITY)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_confirmreading(request_rec *r,
					const char *uri, const char *userid,
					int uri_scope,
					divy_db_transaction_ctx *ts_ctx);
/**
 * day 日以上設定されていた開封通知を全て解除する.
 *
 * @param r request_rec *
 * @param day int 日数
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_release_expired_confirmreading(request_rec *r,
					int day, divy_db_transaction_ctx *ts_ctx);

/**
 * uri, userid が示す開封通知を取得する.
 *
 * uri, userid がNULLかどうかにより、動作が異なります.
 *
 * * userid == NULL : uri と一致する全ての開封通知を取得
 * * uri    == NULL : userid と一致する全ての開封通知を取得
 * * NULL なし      : uri と userid に一致する開封通知を取得
 *
 * @param r request_rec *
 * @param uri const char * URI
 * @param userid const char * ユーザID
 * @param cr_pf divy_rdbo_confirmreading ** 取得した開封通知
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_confirmreading(request_rec *r,
					const char *uri, const char *userid,
					divy_rdbo_confirmreading **cr_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * uri が示すリソースに対し開封通知を仕掛けているユーザのユーザプロパティ一覧を
 * 取得する.
 * (note)
 *   非アクティブユーザのプロパティは取得しません.
 *
 * @param r request_rec *
 * @param uri const char * リソースのURI
 * @param usr_pr divy_rdbo_usr ** 取得したユーザプロパティ一覧 (複数あるときはnext に連結)
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_confirmreading_users(request_rec *r,
					const char *uri, divy_rdbo_usr **usr_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * src_userid が持っていた開封通知をest_userid のユーザに付け替える.
 *
 * @param r request_rec *
 * @param src_userid const char * src ユーザID
 * @param dst_userid const char * dst ユーザID
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_confirmreading_userid(request_rec *r,
					const char *src_userid, const char *dst_userid,
					divy_db_transaction_ctx *ts_ctx);

/**
 * src_uri が持っていた開封通知をdst_uri のリソースに付け替える.
 *
 * @param r request_rec *
 * @param src_uri const char * src リソースのURI
 * @param dst_uri const char * dst リソースのURI
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_confirmreading_uri(request_rec *r,
					const char *src_uri, const char *dst_uri,
					divy_db_transaction_ctx *ts_ctx);

/**
 * private 関数 _insert_grpmem の公開版
 *
 * @param r request_rec *
 * @param grpid const char * グループID
 * @param usrid const char * ユーザID
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 *
 */
DIVY_DECLARE(int) divy_rdbo_insert_grpmem(request_rec *r,
					const char *grpid, const char *usrid,
					divy_db_transaction_ctx *ts_ctx);

#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_TF_RDBO_H */

