/**
 * $Id$
 *
 * 特殊フォルダ「DBMS」関連の関数をまとめたヘッダファイル
 */
#ifndef INCLUDE_TF_RDBO_DBMS_H
#define INCLUDE_TF_RDBO_DBMS_H

#include "apr.h"
#include "apr_pools.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare public functions 
  ----------------------------------------------------------------------------*/
/**
 * DBMS の表示情報を検索・取得する。
 * (note)
 *	dbmsinformationsearch で使用します。
 *	このSEARCH専用になっていますので、他の用途には使用してはなりません。
 * 
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param screen divy_search_dbmsis_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功 / 1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_dbmsinformationsearch(divy_rdbo_search_params *params,
						divy_search_dbmsis_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * リポジトリDB に登録されているアクテイブなDBMS接続情報を取得して返却する。
 * リポジトリDB用の擬似エントリも取得しますが、接続情報は正しくありません.
 * ディレクティブの値を使ってきちんと置き換えてから利用して下さい.
 * (note)
 * 	この関数はDBプロバイダの管理モジュールが、プロバイダを初期化する際に
 * 	必要な情報を取得する目的で使用されます。
 * 	その他の関数からは使用してはなりません。
 * (note)
 * 	dbconn はトランザクションが開始されていない状態で、
 * 	かつcloseされていてはなりません。
 *
 * select 対象テーブル: divy_dbms
 *
 * @param p apr_pool_t *
 * @param dbconn DbConn * リポジトリDBのDbConn
 * @param dbms_pr divy_rdbo_dbms ** 取得したDBMS接続情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_dbmsinfo(apr_pool_t *p,
					DbConn *dbconn,
					divy_rdbo_dbms **dbms_pr);

/**
 * 指定されたdbmsidname が示すDBMS接続情報を取得して返却する。
 *
 * select 対象のテーブル: divy_dbms
 *
 * @param r request_rec *
 * @param dbmsidname const char * DBMS接続先別識別名称を表す文字列
 * @param dbms_pr divy_rdbo_dbms ** 取得したDBMS接続情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_dbms_property(request_rec *r,
						const char *dbmsidname,
						divy_rdbo_dbms **dbms_pr);

/**
 * 指定されたdbms_pr が表すDBMS情報を新規登録する。
 *
 * insert対象のテーブル: divy_dbms
 * (note) dbms_pr の必須項目
 * 	type, name, hostname, port, dbname, username, password, active
 *	registdtは自動的に補完します。
 *	上記以外のデータは、指定されていた値をそのまま反映します。
 * (note)
 *	同じname のDBMSは登録出来ません。
 *
 * (note) トランザクションが継続中であれば引き継ぎます
 *
 * @param r request_rec *
 * @param dbms_pr const divy_rdbo_dbms * 登録するDBMS情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_dbms_property(request_rec *r,
						const divy_rdbo_dbms *dbms_pr,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたdbms_pr が表すDBMS情報の内容で更新する。
 *
 * update対象のテーブル: divy_dbms
 * (note) dbms_pr の必須項目
 * 	dbmsid, type, hostname, port, dbname, username, password, active
 *
 * 更新される情報:
 *	type, hostname, port, dbname, username, password, active,
 *	dbpool, dbminspareconn, dbmaxspareconn, updatedt
 *	(updatedtは自動的に補完)
 *
 * 更新されない情報:
 * 	dbmsid, name, registdt
 *	
 * (note) 呼び出し条件
 * 	dbms_prのDBMSを利用するSQLがあるとき、activeを非アクティブに
 * 	することは出来ません。呼び出し元にて保証して下さい。
 *
 * (note) ts_ctxがない場合は失敗します。
 *
 * @param r request_rec *
 * @param dbms_pr const divy_rdbo_dbms * 更新するDBMS情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_dbms_property(request_rec *r,
						const divy_rdbo_dbms *dbms_pr,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたdbms_pr が表すDBMS情報を削除する
 *
 * delete対象のテーブル: divy_dbms
 * (note) dbms_pr の必須項目: dbmsid
 * (note) 呼び出し条件
 * 	dbms_pr のDBMSを利用するSQLが存在しないことを呼び出し元にて保証して下さい。
 *
 * @param r request_rec *
 * @param dbms_pr const divy_rdbo_dbms * 削除するDBMS情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_dbms_property(request_rec *r,
						const divy_rdbo_dbms *dbms_pr);

/**
 * 指定されたsrc_dbms_pr が表すDBMS情報の内容をdst_dbms_pr が示す
 * DBMS情報としてコピーする。
 *
 * 更新対象のテーブル: divy_dbms
 * (note) src_dbms_pr の必須項目 : dbmsid
 * (note) dst_dbms_pr の必須項目 : name
 *
 * 変更される情報: (コピーされない)
 * 	dbmsid, name, registdt, updatedt
 *
 * 変更されない情報:(そのままコピーされる)
 * 	type, hostname, port, dbname, username, password, comment,
 * 	dbpool, dbminspareconn, dbmaxspareconn, active
 *
 * @param r request_rec *
 * @param src_dbms_pr const divy_rdbo_dbms * コピー元DBMS情報
 * @param dst_dbms_pr const divy_rdbo_dbms * コピー先DBMS情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_copy_dbms_property(request_rec *r,
						const divy_rdbo_dbms *src_dbms_pr,
						const divy_rdbo_dbms *dst_dbms_pr);

/**
 * 指定されたsrc_dbms_pr が表すDBMS情報の内容をdst_dbms_pr が示す
 * DBMS情報として移動(名称変更)する。
 *
 * 更新対象のテーブル: divy_dbms
 * (note) src_dbms_pr の必須項目 : dbmsid
 * (note) dst_dbms_pr の必須項目 : name
 *
 * 変更される情報:
 * 	name, updatedt
 *
 * 変更されない情報:
 * 	dsmsid, type, hostname, port, dbname, username, password, comment,
 * 	dbpool, dbminspareconn, dbmaxspareconn, active, registdt
 *
 * @param r request_rec *
 * @param src_dbms_pr const divy_rdbo_dbms * 移動元DBMS情報
 * @param dst_dbms_pr const divy_rdbo_dbms * 移動先DBMS情報
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_dbms_property(request_rec *r,
						const divy_rdbo_dbms *src_dbms_pr,
						const divy_rdbo_dbms *dst_dbms_pr);

#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_TF_RDBO_DBMS_H */
