/**
 * $Id$
 *
 * 特殊フォルダ「グループ」関連の関数をまとめたヘッダファイル
 * (note)
 *   グループはユーザやSQL、リソースとの癒着が多く、全てをtf_rdbo.h から
 *   移行出来た訳ではありません. 更新系の関数は殆どNGでした.ご注意下さい.
 */
#ifndef INCLUDE_TF_RDBO_GROUP_H
#define INCLUDE_TF_RDBO_GROUP_H

#include "apr.h"
#include "apr_pools.h"
#include "tf_rdbo_util.h"
#include "tf_rdbo.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare public functions 
  ----------------------------------------------------------------------------*/
/**
 * 指定されたSQL文、検索オプションによりグループ情報・グループ所属ユーザ情報・
 * グループ所属SQL情報をリポジトリDBから検索して返却する。
 * 所属SQL情報検索の場合は継承SQLのセットをあわせて返却する。
 * (groupinformationsearch の content, detail, treelist)
 *
 * (note)
 * 	* 継承SQLのセットを返却するのはsearch.cで継承と所属の重複のない
 * 	  レスポンスを作成するのに必要なためです。
 * 	* Oracle のサポートは止めました
 *
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param screen divy_search_grpis_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功、1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_groupinformationsearch(divy_rdbo_search_params *params,
						divy_search_grpis_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * groupinformationsearch のavailableuser の実装.
 *
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param screen divy_search_grpis_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功、1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_groupinformation_availableuser(divy_rdbo_search_params *params,
						divy_search_grpis_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * groupinformationsearch のavailablesql の実装.
 *
 * @param params divy_rdbo_search_params * 入力パラメータ
 * @param screen divy_search_grpis_iscreen * 入力値
 * @param output divy_rdbo_search_output * 出力結果へのポインタ(アロケート済み)
 * @return int 処理ステータス (0:成功、1:失敗)
 */
DIVY_DECLARE(int) divy_rdbo_groupinformation_availablesql(divy_rdbo_search_params *params,
						divy_search_grpis_iscreen *screen,
						divy_rdbo_search_output *output);

/**
 * 指定されたrelativeuri(相対URI),depth が示すグループの情報を取得して返却する。
 *
 * select 対象テーブル: divy_grp, dav_resource
 * (note)
 *	grpuri = /$groupname_p1/$groupname_p2/$groupname
 *	※grpuri は $root/.management/GROUP/$groupname_p1/$groupname_p2/$groupname
 *	  とは異なり異なります。
 *
 * (note) トランザクションが継続中であれば引き継ぎます
 *
 * @param r request_rec *
 * @param relativeuri const char * グループのrelativeuriを表す文字列
 * @param depth int 深さ
 * @param grp_pr divy_rdbo_grp ** 取得したグループ情報
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_get_hierarchy_group_property(request_rec *r,
						const char *relativeuri,
						int depth,
						divy_rdbo_grp **grp_pr,
						divy_db_transaction_ctx *ts_ctx);

/**
 * grpname が示すグループのプロパティを検索する。
 *
 * @param r request_rec *
 * @param grpname const char * グループ名
 * @param grp_pr divy_rdbo_grp ** 取得したグループプロパティ。存在しなければNULL
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_get_group_property_by_name(request_rec *r,
					const char *grpname,
					divy_rdbo_grp **grp_pr);

/**
 * グループフォルダに格納されたリソースのURI resource_uri の所属グループの
 * グループプロパティを取得する.
 *
 * @param r request_rec *
 * @param resource_uri const char * グループフォルダ以下のリソースURI
 * @param grp_pr divy_rdbo_grp ** 取得したグループプロパティ。存在しなければNULL
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_get_group_property_by_resuri(request_rec *r,
					const char *resource_uri,
					divy_rdbo_grp **grp_pr);

/**
 * groupid が示すグループのプロパティを取得する.
 * (note)
 *   トランザクションが継続していれば引き継ぎます
 *
 * @param r request_rec *
 * @param groupid const char * グループID
 * @param grp_pr divy_rdbo_grp ** 取得したグループプロパティ。存在しなければNULL
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_get_group_property_by_groupid(request_rec *r,
					const char *groupid,
					divy_rdbo_grp **grp_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたuserid のユーザが所属するグループ一覧のリソースをoffset から limit個まで
 * 取得して返却する。
 * グループURIとグループ名を同期化する場合には、2階層目以降のグループは存在
 * しても表示しません。
 * (note)
 * 	* 作成者、更新者の名称は取得出来ません。
 * 	* グループ管理者機能がサポートされている場合、更新者名にはオーナ名を入れます.
 * 	* offset は 1から始まります
 * 	* 非アクティブグループは対象に含まれません
 *
 * @param r request_rec *
 * @param userid const char * ユーザID
 * @param offset apr_int32_t オフセット
 * @param limit apr_int32_t 取得件数 (-1 の時、unlimited になります)
 * @param rdb_r divy_rdbo_resource ** グループ一覧
 * @return 処理ステータス 0: 成功 / 1: 失敗 / DIVY_STCODE_REMAIN_DATA: 続きページがある
 */
DIVY_DECLARE(int) divy_rdbo_get_group_resource_by_userid(request_rec *r,
						const char * userid,
						apr_int32_t offset,
						apr_int32_t limit,
						divy_rdbo_resource **rdb_r);

/**
 * 指定されたuserid のユーザがアクセス可能なグループのグループコレクションURI
 * 一覧を取得してset にして返却する。
 *
 * [ アクセス可能である条件 ]
 *	* 対象ユーザが所属しているグループであること
 *	* 上記グループであっても、syncgrpuri がONであれば
 *	  単階層グループであること
 *	* アクティブなグループであること (2007/08/10 Fri)
 *
 * (note)
 * 	トランザクションがあれば継続します
 *
 * @param r request_rec *
 * @param userid const char* ユーザID
 * @param grpcol_uri_set divy_cset_t ** 取得したグループIDをキー値として持つset
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return 処理ステータス 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_rdbo_get_grpcoluris_by_userid(request_rec *r,
						const char *userid,
						divy_cset_t **grpcol_uri_set,
						divy_db_transaction_ctx *ts_ctx);

/**
 * grpuri (グループ相対URI)  が示すグループの"トップグループ"を取得する.
 * (note)
 *   * grpuri 自身がトップグループであれば、その情報を返します.
 *   * grpuri = /$groupname1/$groupname2 の場合、/$groupname1 のプロパティを返します.
 *
 * @param r request_rec *
 * @param grpuri const char * グループ相対URI
 * @param grp_pr divy_rdbo_grp ** 取得したグループプロパティ。存在しなければNULL
 * @param ts_ctx divy_db_transaction_ctx *
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_get_topgroup_property_by_grpuri(request_rec *r,
					const char *grpuri,
					divy_rdbo_grp **grp_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたgrp_pr が表すグループを新規登録する。
 * グループが持っているグループコレクションも作成する。
 *
 * insert 対象テーブル:  divy_grp, dav_resource
 * (note) grp_pr の必須項目: name, relativeuri
 *	registdtは自動的に補完します。
 *
 *	上記以外のデータは、指定されていた値をそのまま反映します。
 *	また、幾つかのデータは作成してinsert しています。
 * (note)
 *	同じ階層内に同名のグループがあってはなりません。
 *	単階層グループしか作成することができない場合には、呼び出し元にて
 *	そのようになるよう保証して下さい。
 *
 * (note) トランザクションが継続中であれば引き継ぎます
 *
 * @param r request_rec *
 * @param grp_pr const divy_rdbo_grp * 登録するグループ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_insert_group_property(request_rec *r,
						const divy_rdbo_grp *grp_pr,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたgrp_pr が表す内容でグループを更新する。
 *
 * update 対象テーブル: divy_grp, dav_resource, divy_mailwatch
 * (note) grp_pr 必須項目: grpid, relativeuri, rsid, grpcol_uri
 * 	(mailwatch_pr != NULLの時) mailwatch_pr->uri, mailwatch_pr->usrid
 * (note) 更新されるデータ
 * 	(grp)
 * 		comment, updatedt
 *		updatedt は自動的に補完します。
 *	(resource)
 *		lastmodifier_userid, getlastmodified
 *	(mailwatch)
 *		trigger_methods, usrid
 *
 *	これ以外のデータは更新できません。(grpid, name, uri depth, rsid, registdt)
 * (note)
 * 	ts_ctxがない場合は失敗します。
 *
 * @param r request_rec *
 * @param grp_pr const divy_rdbo_grp * 更新するグループ
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_update_group_property(request_rec *r, 
					const divy_rdbo_grp *grp_pr,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたgrp_pr が示すグループを削除する。
 * 指定グループの下にグループが存在した場合、それらも合わせて全て削除します。
 * グループが保持していたリソース以外の資源(SQL使用権、ユーザの所属状況、
 * メール監視状況など)もクリアします。
 *
 * (note)
 * 	* 中身が残っていたら削除させない等の条件があるのなら
 * 	  それは呼び出し側で保証して下さい。
 * 	* トランザクションが継続中であればそれを引き継ぎます。
 *
 * grp_pr の必須項目: grpid, relativeuri
 *
 * delete 対象テーブル:
 * 	divy_grp       : 対象グループと下位階層グループのエントリ
 * 	divy_grpmem    : 対象グループと下位階層グループのエントリ
 * 	divy_sqlmem    : 対象グループと下位階層グループのエントリ
 * 	divy_mailwatch : 対象グループと下位階層のグループのエントリ
 * 			 所属ユーザが持っていたmailwatch)
 * 	dav_resource   : 対象グループと下位階層グループのグループコレクション
 * 	dav_lock       : グループコレクション以下に残ってしまった不正ロック
 * 	divy_trash_info: ごみ箱フォルダのコレクション
 *
 * @param r request_rec *
 * @param grp_pr const divy_rdbo_grp * 削除するグループ
 * @param del_filelist divy_linkedlist_t ** グループコレクション以下に存在したリソース
 * @param ts_ctx divy_db_transaction_ctx * トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_remove_group_property(request_rec *r,
					const divy_rdbo_grp *grp_pr,
					divy_linkedlist_t **del_filelist,
					divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたsrc_grp_pr が示すグループをdst_grp_pr が示すグループとしてコピーする。
 * depthがinfinityの場合src_grp_pr の下にあったグループも一緒にコピーします。
 * また、このグループに所属していたユーザ、SQLもそのまま引き継がれます。
 * src_grp_pr が持っていたリソース情報はコピーされません。
 *
 * 更新対象テーブル:
 *    divy_grp, dav_resource, divy_sqlmem, divy_grpmem
 *
 * src_grp_pr の必須項目: grpid, name, relativeuri, rsid, grpcol_uri
 * dst_grp_pr の必須項目: name, relativeuri
 *
 * コピーされる情報:
 *	comment,
 *	src_grp_pr に所属していたユーザの所属状態,
 *	src_grp_pr が利用可能なSQL一覧の利用可能状態
 *
 * コピーされない情報:
 * 	grpid, name, relativeuri, depth, rsid, registdt, updatedt,
 * 	src_grp_pr が持っていたグループ・コレクション
 *
 * @param r request_rec *
 * @param depth int Depth ヘッダの値 (0 / DAV_INFINITY)
 * @param src_grp_pr const divy_rdbo_grp * コピー元グループ
 * @param dst_grp_pr const divy_rdbo_grp * コピー先グループ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_copy_group_property(request_rec *r, int depth,
					const divy_rdbo_grp *src_grp_pr,
					const divy_rdbo_grp *dst_grp_pr);

/**
 * 指定されたsrc_grp_pr が示すグループをdst_grp_pr が示すグループとして移動する。
 * src_grp_pr の下にあったグループも一緒に移動します。
 *
 * 更新対象テーブル:
 * 	divy_grp, dav_resource, divy_resourcestate, divy_mailwatch
 *
 * src_grp_pr の必須項目: grpid, name, relativeuri, grpcol_uri
 * dst_grp_pr の必須項目: name, relativeuri
 *
 * 変更される情報:
 *	name, relativeuri, depth, updatedt, grpcol_uri(条件に一致するとき)
 *	updatedt は、自動的に補完されます。
 *
 * 変更されない情報:
 * 	grpid, rsid, registdt
 * 	src_grp_pr が持っていたグループ・コレクション,
 * 	src_grp_pr に所属しているユーザ,
 * 	src_grp_pr が利用可能なSQL
 * 	src_grp_pr のgroupmailwatch
 * (note)
 * 	* syncgrpuri がONの場合であっても、ロックされているリソースが
 * 	  あるかどうか確認していません。ONのケースでは、呼び出し元で
 * 	  ロックされていないことを確認して下さい。
 * 	* syncgrpuri が変更された場合に対応出来る実装には出来ません。(原理的に)
 * 	  syncgrpuri が変更されていないことを呼び出し元で保証すること。
 *
 * @param r request_rec *
 * @param src_grp_pr const divy_rdbo_grp * 移動元グループ
 * @param dst_grp_pr const divy_rdbo_grp * 移動先グループ
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_rdbo_move_group_property(request_rec *r,
					const divy_rdbo_grp *src_grp_pr,
					const divy_rdbo_grp *dst_grp_pr);

/**
 * 指定された相対URI(top_grpuri) が示すトップグループとそのサブグループの
 * オーナをowerid に変更する.
 * (note)
 *   * サブグループのオーナIDも全て更新されます (外部仕様)
 *   * ownerid がNULLの場合、オーナは存在しないことになります (外部仕様)
 *
 * @param r request_rec *
 * @param top_grpuri const char * トップグループの相対URI
 * @param ownerid const char * 変更するownerid
 * @param ts_ctx divy_db_transaction_ctx *
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_update_group_ownerinfo(request_rec *r, const char *top_grpuri,
						const char *ownerid, divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたownerid が管理する全てグループでのグループリーダの役割を「解任」する.
 *
 * @param r request_rec *
 * @param ownerid const char * 対象となるグループリーダのownerid
 * @param ts_ctx divy_db_transaction_ctx *
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_rdbo_dismiss_groupleader(request_rec *r, const char *ownerid,
						divy_db_transaction_ctx *ts_ctx);

/**
 * 指定されたグループに所属しているユーザ一覧を取得する
 * システムユーザは取得できません。
 *
 * @param r request_rec *
 * @param usr_pr	divy_rdbo_user ** 
 * @param grp_pr	divy_grp *
 * @param ts_ctx divy_db_transaction_ctx *
 * @return int 処理ステータス
 */ 
DIVY_DECLARE(int) divy_rdbo_get_group_member(request_rec *r,
						divy_rdbo_usr **usr_pr, const divy_rdbo_grp *grp_pr,
						divy_db_transaction_ctx *ts_ctx);

#ifdef __cplusplus
}
#endif

#endif /* INCLUDE_TF_RDBO_GROUP_H */

